package org.qas.qtest.api.services.authenticate.model.transform;

import org.qas.api.AuthClientException;
import org.qas.api.Request;
import org.qas.api.http.HttpMethod;
import org.qas.api.internal.util.google.net.HttpHeaders;
import org.qas.api.transform.AbstractMarshaller;
import org.qas.qtest.api.services.authenticate.model.OAuthAuthenticateRequest;

import java.util.Locale;
import java.util.Map;

/**
 * OAuthAuthenticateRequestMarshaller
 *
 * @author Dzung Nguyen
 * @version $Id OAuthAuthenticateRequestMarshaller 2016-11-29 06:55:30z dzungvnguyen $
 * @since 1.0
 */
public class OAuthAuthenticateRequestMarshaller extends AbstractMarshaller<Request, OAuthAuthenticateRequest> {

  @Override
  protected void validateRequest(OAuthAuthenticateRequest in) throws Exception {
    if (in == null) {
      throw new AuthClientException("Invalid argument passed to call (...)");
    }
  }

  @Override
  protected void build(Request request, OAuthAuthenticateRequest in) {
    super.build(request, in);
    // setting content type.
    request.setHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded");
    request.setHeader(HttpHeaders.ACCEPT, "application/json");

    request.setHeader(HttpHeaders.AUTHORIZATION, normalizeToken(in.getToken(), "Basic"));

    // set method and resource.
    request.setHttpMethod(HttpMethod.POST);

    Map<String, String> requestParams = in.requestParams();

    if (requestParams != null && !requestParams.isEmpty()) {
      for (Map.Entry<String, String> param : requestParams.entrySet()) {
        request.addParameter(param.getKey(), param.getValue());
      }
    }
  }

  @Override
  protected String getServiceName() {
    return "AuthenticateService";
  }

  @Override
  protected String getActionName() {
    return "Authenticate";
  }

  @Override
  protected String getContent(OAuthAuthenticateRequest in) throws Exception {
    return null;
  }

  @Override
  protected StringBuilder getResourcePath(OAuthAuthenticateRequest in, StringBuilder rpBuilder) {
    return rpBuilder.append("/oauth/token");
  }

  /**
   * Normalize the token.
   *
   * @param token  the given token.
   * @param prefix the given prefix.
   * @return the normalized token.
   */
  static String normalizeToken(String token, String prefix) {
    String tokenLc = token.toLowerCase();

    if (tokenLc.startsWith(prefix.toLowerCase(Locale.US) + " ")) {
      return token;
    }

    return String.format("%s %s", prefix, token);
  }
}
