package org.qas.qtest.api.services.client;

import org.qas.api.AuthClientException;
import org.qas.api.AuthServiceException;
import org.qas.api.ClientConfiguration;
import org.qas.api.Request;
import org.qas.api.internal.util.google.base.Throwables;
import org.qas.api.transform.VoidJsonUnmarshaller;
import org.qas.qtest.api.auth.DefaultQTestCredentialsProviderChain;
import org.qas.qtest.api.auth.QTestCredentials;
import org.qas.qtest.api.auth.QTestCredentialsProvider;
import org.qas.qtest.api.auth.StaticQTestCredentialsProvider;
import org.qas.qtest.api.internal.QTestApiWebServiceClient;
import org.qas.qtest.api.services.client.model.Client;
import org.qas.qtest.api.services.client.model.GetClientRequest;
import org.qas.qtest.api.services.client.model.SetSessionUrlRequest;
import org.qas.qtest.api.services.client.model.transform.ClientJsonUnmarshaller;
import org.qas.qtest.api.services.client.model.transform.GetClientRequestMarshaller;
import org.qas.qtest.api.services.client.model.transform.SetSessionUrlRequestMarshaller;

/**
 * ClientServiceClient
 *
 * @author Dzung Nguyen
 * @version $Id ClientServiceClient 2014-11-11 13:45:30z dungvnguyen $
 * @since 1.0
 */
public class ClientServiceClient extends QTestApiWebServiceClient<ClientServiceClient>
    implements ClientService {
  /**
   * Constructs a new client to invoke service method on ClientService using
   * the default qTest credentials provider and default client configuration options.
   */
  public ClientServiceClient() {
    this(new DefaultQTestCredentialsProviderChain(), new ClientConfiguration());
  }

  /**
   * Constructs a new client to invoke service method on ClientService using
   * the default qTest credentials provider and client configuration options.
   *
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to ClientService
   */
  public ClientServiceClient(ClientConfiguration clientConfiguration) {
    this(new DefaultQTestCredentialsProviderChain(), clientConfiguration);
  }

  /**
   * Constructs a new client to invoke service method on ClientService using
   * the specified qTest credentials.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   */
  public ClientServiceClient(QTestCredentials credentials) {
    this(credentials, new ClientConfiguration());
  }

  /**
   * Constructs a new client to invoke service method on ClientService using
   * the specified qTest credentials and client configuration options.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to ClientService
   */
  public ClientServiceClient(QTestCredentials credentials, ClientConfiguration clientConfiguration) {
    super(clientConfiguration);
    this.credentialsProvider = new StaticQTestCredentialsProvider(credentials);
    init();
  }

  /**
   * Constructs a new client to invoke service method on ClientService using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   */
  public ClientServiceClient(QTestCredentialsProvider credentialsProvider) {
    this(credentialsProvider, new ClientConfiguration());
  }

  /**
   * Constructs a new client to invoke service method on ClientService using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to ClientService
   */
  public ClientServiceClient(QTestCredentialsProvider credentialsProvider, ClientConfiguration clientConfiguration) {
    super(clientConfiguration);
    this.credentialsProvider = credentialsProvider;

    init();
  }

  @Override
  public Client getClient(GetClientRequest getClientRequest) throws AuthServiceException {
    try {
      Request request = new GetClientRequestMarshaller().marshall(getClientRequest);
      return invoke(request, ClientJsonUnmarshaller.getInstance());
    } catch (Exception ex) {
      Throwables.propagateIfInstanceOf(ex, AuthClientException.class);
      throw new AuthClientException("Unknown error occurs during getting client", ex);
    }
  }

  @Override
  public void setSessionUrl(SetSessionUrlRequest setSessionUrlRequest) throws AuthServiceException {
    try {
      Request request = new SetSessionUrlRequestMarshaller().marshall(setSessionUrlRequest);
      invoke(request, new VoidJsonUnmarshaller());
    } catch (Exception ex) {
      Throwables.propagateIfInstanceOf(ex, AuthClientException.class);
      throw new AuthClientException("Unknown error occurs during setting session URL", ex);
    }
  }
}
