package org.qas.qtest.api.services.design;

import org.qas.api.AuthServiceException;
import org.qas.api.ClientConfiguration;
import org.qas.api.handler.AsyncHandler;
import org.qas.qtest.api.auth.DefaultQTestCredentialsProviderChain;
import org.qas.qtest.api.auth.QTestCredentials;
import org.qas.qtest.api.auth.QTestCredentialsProvider;
import org.qas.qtest.api.auth.StaticQTestCredentialsProvider;
import org.qas.qtest.api.internal.model.CreateObjectCommentRequest;
import org.qas.qtest.api.internal.model.Field;
import org.qas.qtest.api.services.design.model.*;

import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

/**
 * TestDesignServiceAsyncClient
 *
 * @author Dzung Nguyen
 * @version $Id TestDesignServiceAsyncClient 2014-04-06 05:12:30z dungvnguyen $
 * @since 1.0
 */
public class TestDesignServiceAsyncClient extends TestDesignServiceClient
  implements TestDesignServiceAsync {
  /**
   * Constructs a new client to invoke service method on TestDesignService using
   * the default qTest credentials provider and default client configuration options.
   */
  public TestDesignServiceAsyncClient() {
    this(new DefaultQTestCredentialsProviderChain(), new ClientConfiguration(), Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on TestDesignService using
   * the default qTest credentials provider and default client configuration options.
   *
   * @param executorService the executor service for executing asynchronous request.
   */
  public TestDesignServiceAsyncClient(ExecutorService executorService) {
    this(new DefaultQTestCredentialsProviderChain(), new ClientConfiguration(), executorService);
  }

  /**
   * Constructs a new client to invoke service method on TestDesignService using
   * the default qTest credentials provider and client configuration options.
   *
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to TestDesignService
   */
  public TestDesignServiceAsyncClient(ClientConfiguration clientConfiguration) {
    this(new DefaultQTestCredentialsProviderChain(), clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on TestDesignService using
   * the default qTest credentials provider and client configuration options.
   *
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to TestDesignService.
   * @param executorService the executor service for executing asynchronous request.
   */
  public TestDesignServiceAsyncClient(ClientConfiguration clientConfiguration, ExecutorService executorService) {
    this(new DefaultQTestCredentialsProviderChain(), clientConfiguration, executorService);
  }

  /**
   * Constructs a new client to invoke service method on TestDesignService using
   * the specified qTest credentials.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   */
  public TestDesignServiceAsyncClient(QTestCredentials credentials) {
    this(credentials, new ClientConfiguration(), Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on TestDesignService using
   * the specified qTest credentials.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param executorService the executor service for executing asynchronous request.
   */
  public TestDesignServiceAsyncClient(QTestCredentials credentials, ExecutorService executorService) {
    this(credentials, new ClientConfiguration(), executorService);
  }

  /**
   * Constructs a new client to invoke service method on TestDesignService using
   * the specified qTest credentials and client configuration options.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to TestDesignService
   */
  public TestDesignServiceAsyncClient(QTestCredentials credentials, ClientConfiguration clientConfiguration) {
    this(credentials, clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on TestDesignService using
   * the specified qTest credentials and client configuration options.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to TestDesignService
   * @param executorService the executor service for executing asynchronous request.
   */
  public TestDesignServiceAsyncClient(QTestCredentials credentials, ClientConfiguration clientConfiguration,
                                         ExecutorService executorService) {
    this(new StaticQTestCredentialsProvider(credentials), clientConfiguration, executorService);
  }

  /**
   * Constructs a new client to invoke service method on TestDesignService using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to TestDesignService
   */
  public TestDesignServiceAsyncClient(QTestCredentialsProvider credentialsProvider,
                                         ClientConfiguration clientConfiguration) {
    this(credentialsProvider, clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on TestDesignService using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to TestDesignService
   * @param executorService the executor service for executing asynchronous request.
   */
  public TestDesignServiceAsyncClient(QTestCredentialsProvider credentialsProvider,
                                         ClientConfiguration clientConfiguration, ExecutorService executorService) {
    super(credentialsProvider, clientConfiguration);
    this.executorService = executorService;
  }

  //~ implement methods =======================================================

  @Override
  public Future<List<TestCase>> listTestCaseAsync(final ListTestCaseRequest listTestCaseRequest) throws AuthServiceException {
    return executorService.submit(new Callable<List<TestCase>>() {
      @Override
      public List<TestCase> call() throws Exception {
        return listTestCase(listTestCaseRequest);
      }
    });
  }

  @Override
  public Future<List<TestCase>> listTestCaseAsync(final ListTestCaseRequest listTestCaseRequest,
                                                  final AsyncHandler<ListTestCaseRequest, List<TestCase>> asyncHandler) throws AuthServiceException {
    return executorService.submit(new Callable<List<TestCase>>() {
      @Override
      public List<TestCase> call() throws Exception {
        List<TestCase> result;

        try {
          result = listTestCase(listTestCaseRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(listTestCaseRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<TestCase> getTestCaseAsync(final GetTestCaseRequest testCaseRequest) throws AuthServiceException {
    return executorService.submit(new Callable<TestCase>() {
      @Override
      public TestCase call() throws Exception {
        return getTestCase(testCaseRequest);
      }
    });
  }

  @Override
  public Future<TestCase> getTestCaseAsync(final GetTestCaseRequest testCaseRequest, final AsyncHandler<GetTestCaseRequest, TestCase> asyncHandler) throws AuthServiceException {
    return executorService.submit(new Callable<TestCase>() {
      @Override
      public TestCase call() throws Exception {
        TestCase result;

        try {
          result = getTestCase(testCaseRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(testCaseRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<TestStep> getTestStepAsync(final GetTestStepRequest testStepRequest) throws AuthServiceException {
    return executorService.submit(new Callable<TestStep>() {
      @Override
      public TestStep call() throws Exception {
        return getTestStep(testStepRequest);
      }
    });
  }

  @Override
  public Future<TestStep> getTestStepAsync(final GetTestStepRequest testStepRequest, final AsyncHandler<GetTestStepRequest, TestStep> asyncHandler) throws AuthServiceException {
    return executorService.submit(new Callable<TestStep>() {
      @Override
      public TestStep call() throws Exception {
        TestStep result;

        try {
          result = getTestStep(testStepRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }
        asyncHandler.onSuccess(testStepRequest, result);

        return result;
      }
    });
  }

  @Override
  public Future<List<TestStep>> listTestStepAsync(final ListTestStepRequest testStepRequest) throws AuthServiceException {
    return executorService.submit(new Callable<List<TestStep>>() {
      @Override
      public List<TestStep> call() throws Exception {
        return listTestStep(testStepRequest);
      }
    });
  }

  @Override
  public Future<List<TestStep>> listTestStepAsync(final ListTestStepRequest testStepRequest, final AsyncHandler<ListTestStepRequest, List<TestStep>> asyncHandler) throws AuthServiceException {
    return executorService.submit(new Callable<List<TestStep>>() {
      @Override
      public List<TestStep> call() throws Exception {
        List<TestStep> result;

        try {
          result = listTestStep(testStepRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }
        asyncHandler.onSuccess(testStepRequest, result);

        return result;
      }
    });
  }

  @Override
  public Future<TestCase> createTestCaseAsync(final CreateTestCaseRequest testCaseRequest) throws AuthServiceException {
    return executorService.submit(new Callable<TestCase>() {
      @Override
      public TestCase call() throws Exception {
        return createTestCase(testCaseRequest);
      }
    });
  }

  @Override
  public Future<TestCase> createTestCaseAsync(final CreateTestCaseRequest testCaseRequest, final AsyncHandler<CreateTestCaseRequest, TestCase> asyncHandler) throws AuthServiceException {
    return executorService.submit(new Callable<TestCase>() {
      @Override
      public TestCase call() throws Exception {
        TestCase result;

        try {
          result = createTestCase(testCaseRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }
        asyncHandler.onSuccess(testCaseRequest, result);

        return result;
      }
    });
  }

  @Override
  public Future<TestStep> createTestStepAsync(final CreateTestStepRequest testStepRequest) throws AuthServiceException {
    return executorService.submit(new Callable<TestStep>() {
      @Override
      public TestStep call() throws Exception {
        return createTestStep(testStepRequest);
      }
    });
  }

  @Override
  public Future<TestStep> createTestStepAsync(final CreateTestStepRequest testStepRequest, final AsyncHandler<CreateTestStepRequest, TestStep> asyncHandler) throws AuthServiceException {
    return executorService.submit(new Callable<TestStep>() {
      @Override
      public TestStep call() throws Exception {
        TestStep result;

        try {
          result = createTestStep(testStepRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }
        asyncHandler.onSuccess(testStepRequest, result);

        return result;
      }
    });
  }

  @Override
  public Future<List<Field>> getTestCaseFieldsAsync(final GetTestCaseFieldsRequest getTestCaseFieldsRequest)
      throws AuthServiceException {
    return executorService.submit(new Callable<List<Field>>() {
      @Override
      public List<Field> call() throws Exception {
        return getTestCaseFields(getTestCaseFieldsRequest);
      }
    });
  }

  @Override
  public Future<List<Field>> getTestCaseFieldsAsync(final GetTestCaseFieldsRequest getTestCaseFieldsRequest,
                                                    final AsyncHandler<GetTestCaseFieldsRequest, List<Field>> asyncHandler)
      throws AuthServiceException {
    return executorService.submit(new Callable<List<Field>>() {
      @Override
      public List<Field> call() throws Exception {
        List<Field> result;

        try {
          result = getTestCaseFields(getTestCaseFieldsRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(getTestCaseFieldsRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<TestCase> addCommentAsync(final CreateObjectCommentRequest createObjectCommentRequest) throws AuthServiceException {
    return executorService.submit(new Callable<TestCase>() {
      @Override
      public TestCase call() throws Exception {
        return addComment(createObjectCommentRequest);
      }
    });
  }

  @Override
  public Future<TestCase> addCommentAsync(final CreateObjectCommentRequest createObjectCommentRequest,
                                          final AsyncHandler<CreateObjectCommentRequest, TestCase> asyncHandler) throws AuthServiceException {
    return executorService.submit(new Callable<TestCase>() {
      @Override
      public TestCase call() throws Exception {
        TestCase result;
        try {
          result = addComment(createObjectCommentRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(createObjectCommentRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<AutomationTestCase> createAutomationTestCaseAsync(
      final CreateAutomationTestCaseRequest automationTestCaseRequest) throws AuthServiceException {
    return executorService.submit(new Callable<AutomationTestCase>() {
      @Override
      public AutomationTestCase call() throws Exception {
        return createAutomationTestCase(automationTestCaseRequest);
      }
    });
  }

  @Override
  public Future<AutomationTestCase> createAutomationTestCaseAsync(
      final CreateAutomationTestCaseRequest automationTestCaseRequest,
      final AsyncHandler<CreateAutomationTestCaseRequest, AutomationTestCase> asyncHandler)
      throws AuthServiceException {
    return executorService.submit(new Callable<AutomationTestCase>() {
      @Override
      public AutomationTestCase call() throws Exception {
        AutomationTestCase result;

        try {
          result = createAutomationTestCase(automationTestCaseRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }
        asyncHandler.onSuccess(automationTestCaseRequest, result);

        return result;
      }
    });
  }

  @Override
  public void shutdown() {
    super.shutdown();
    executorService.shutdown();
  }
}
