package org.qas.qtest.api.services.execution;

import org.qas.api.AuthServiceException;
import org.qas.qtest.api.internal.QTestService;
import org.qas.qtest.api.services.execution.model.*;
import java.util.List;

/**
 * TestExecutionService
 *
 * @author Dzung Nguyen
 * @version $Id TestExecutionService 2014-03-28 05:58:30z dungvnguyen $
 * @since 1.0
 */
public interface TestExecutionService extends QTestService {
  /**
   * Submits automation test log to test execution service.
   *
   * @param automationTestLogRequest the given automation test log request.
   * @return the {@link TestLog} instance.
   * @throws AuthServiceException if an error occurs during submitting automation
   *        test log to test execution service.
   */
  TestLog submitAutomationTestLog(AutomationTestLogRequest automationTestLogRequest) throws AuthServiceException;

  /**
   * Submit test log to test execution service.
   *
   * @param submitTestLogRequest the given test log request.
   * @return the {@link TestLog} instance.
   * @throws AuthServiceException if an error occurs during submitting test log to
   *        test execution service.
   */
  TestLog submitTestLog(SubmitTestLogRequest submitTestLogRequest) throws AuthServiceException;

  /**
   * Gets test run information.
   *
   * @param getLastLogRequest the given {@link GetLastLogRequest} instance.
   * @return the {@link TestRun} instance.
   * @throws AuthServiceException if an error occurs during getting test-run information.
   */
  TestLog getLastLog(GetLastLogRequest getLastLogRequest) throws AuthServiceException;

  /**
   * List all execution status request from test execution service.
   *
   * @param executionStatusRequest the given {@link ListExecutionStatusRequest} object.
   * @return the list of {@link ExecutionStatus execution status} objects.
   * @throws AuthServiceException if an error occurs during listing all execution statuses.
   */
  List<ExecutionStatus> listExecutionStatus(ListExecutionStatusRequest executionStatusRequest) throws AuthServiceException;

  //~ Test Run ================================================================
  /**
   * Lists all test runs from the given {@link ListTestRunRequest list test run request}.
   *
   * @param testRunRequest the given {@link ListTestRunRequest list test run request} object.
   * @return the list of {@link TestRun test run} objects.
   * @throws AuthServiceException if an error occurs during listing all test runs.
   */
  List<TestRun> listTestRun(ListTestRunRequest testRunRequest) throws AuthServiceException;

  /**
   * Gets test run information.
   *
   * @param getTestRunRequest the given {@link GetTestRunRequest} instance.
   * @return the {@link TestRun} instance.
   * @throws AuthServiceException if an error occurs during getting test-run information.
   */
  TestRun getTestRun(GetTestRunRequest getTestRunRequest) throws AuthServiceException;

  /**
   * Create test-run.
   *
   * @param testRunRequest the given create test-run request instance.
   * @return the test-run instance.
   * @throws AuthServiceException if an error occurs during creating test-run.
   */
  TestRun createTestRun(CreateTestRunRequest testRunRequest) throws AuthServiceException;

  /**
   * Update test run.
   *
   * @param testRunRequest the given update test-run request.
   * @return the test-run instance.
   * @throws AuthServiceException if an error occurs during updating test-run.
   */
  TestRun updateTestRun(UpdateTestRunRequest testRunRequest) throws AuthServiceException;

  /**
   * Move test run.
   *
   * @param testRunRequest the given move test-run request.
   * @return the test-run instance.
   * @throws AuthServiceException if an error occurs during moving test-run.
   */
  TestRun moveTestRun(MoveTestRunRequest testRunRequest) throws AuthServiceException;

  /**
   * Delete test-run.
   *
   * @param testRunRequest the given delete test-run request.
   * @throws AuthServiceException if an error occur during deleting test-run.
   */
  void deleteTestRun(DeleteTestRunRequest testRunRequest) throws AuthServiceException;
  //~ Test Suite ==============================================================
  /**
   * Creates test-suite.
   *
   * @param testSuiteRequest the given create test-suite request instance.
   * @return the test-suite instance.
   * @throws AuthServiceException if an error occurs during creating test-suite.
   */
  TestSuite createTestSuite(CreateTestSuiteRequest testSuiteRequest) throws AuthServiceException;

  /**
   * Updates test-suite.
   *
   * @param testSuiteRequest the given update test-suite request instance.
   * @return the test-suite instance.
   * @throws AuthServiceException if an error occurs during updating test-suite.
   */
  TestSuite updateTestSuite(UpdateTestSuiteRequest testSuiteRequest) throws AuthServiceException;

  /**
   * Move test-suite.
   *
   * @param testSuiteRequest the given move test-suite request.
   * @return the test-suite instance.
   * @throws AuthServiceException if an error occurs during moving test-suite.
   */
  TestSuite moveTestSuite(MoveTestSuiteRequest testSuiteRequest) throws AuthServiceException;

  /**
   * Gets the test-suite information.
   *
   * @param testSuiteRequest the given get test-suite request instance.
   * @return the test suite instance.
   * @throws AuthServiceException if an error occurs during getting test suite.
   */
  TestSuite getTestSuite(GetTestSuiteRequest testSuiteRequest) throws AuthServiceException;

  /**
   * Lists all test suites from the given {@link ListTestSuiteRequest list test suite request}.
   *
   * @param testSuiteRequest the given {@link ListTestSuiteRequest list test suite request} object.
   * @return the list of {@link TestSuite test suite} objects.
   * @throws AuthServiceException if an error occurs during listing all test suites.
   */
  List<TestSuite> listTestSuite(ListTestSuiteRequest testSuiteRequest) throws AuthServiceException;

  /**
   * Delete test-suite.
   *
   * @param testSuiteRequest the given delete test-suite request instance.
   * @throws AuthServiceException if an error occurs during deleting test-suite.
   */
  void deleteTestSuite(DeleteTestSuiteRequest testSuiteRequest) throws AuthServiceException;
  //~ Test cycle ==============================================================
  /**
   * Gets the test-cycle information from the given get test-cycle request.
   *
   * @param testCycleRequest the given get test-cycle request.
   * @return the test cycle information.
   * @throws AuthServiceException if an error occurs during getting test-cycle information.
   */
  TestCycle getTestCycle(GetTestCycleRequest testCycleRequest) throws AuthServiceException;

  /**
   * Lists test-cycle information from the given list test-cycle request.
   *
   * @param testCycleRequest the given list test-cycle request.
   * @return the list of test-cycles information.
   * @throws AuthServiceException if an error occurs during listing test-cycle information.
   */
  List<TestCycle> listTestCycle(ListTestCycleRequest testCycleRequest) throws AuthServiceException;

  /**
   * Creates test-cycle from the given create test-cycle information.
   *
   * @param testCycleRequest the given test cycle request instance.
   * @return the test-cycle information.
   * @throws AuthServiceException if an error occurs during create test-cycle.
   */
  TestCycle createTestCycle(CreateTestCycleRequest testCycleRequest) throws AuthServiceException;

  /**
   * Update test-cycle from the given update test-cycle request instance.
   *
   * @param testCycleRequest the given update test cycle request instance.
   * @return the test cycle after update.
   * @throws AuthServiceException if an errro occur during update test cycle.
   */
  TestCycle updateTestCycle(UpdateTestCycleRequest testCycleRequest) throws AuthServiceException;

  /**
   * Move test-cycle from the given move test-cycle request instance.
   *
   * @param testCycleRequest the given move test-cycle request instance.
   * @return test cycle instance.
   * @throws AuthServiceException if an error occurs during moving test-cycle.
   */
  TestCycle moveTestCycle(MoveTestCycleRequest testCycleRequest) throws AuthServiceException;

  /**
   * Delete test-cycle from the given request.
   *
   * @param testCycleRequest the given test-cycle request.
   * @throws AuthServiceException if an error occurs during deleting test-cycle request.
   */
  void deleteTestCycle(DeleteTestCycleRequest testCycleRequest) throws AuthServiceException;
}
