package org.qas.qtest.api.services.execution.model;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.qas.api.DateTimeDeserializer;
import org.qas.api.DateTimeSerializer;
import org.qas.api.internal.util.google.collect.Lists;
import org.qas.qtest.api.internal.model.QTestBaseModel;
import org.qas.qtest.api.services.attachment.model.Attachment;
import org.qas.qtest.api.services.design.model.TestCase;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

/**
 * TestLog
 *
 * @author Dzung Nguyen
 * @version $Id TestLog 2014-03-28 07:10:30z dungvnguyen $
 * @since 1.0
 */
public class TestLog extends QTestBaseModel<TestLog> {

  @JsonProperty("id")
  private Long id;

  @JsonProperty("status")
  private ExecutionStatus status;

  @JsonProperty("test_case_version_id")
  private Long testCaseVersionId;

  @JsonProperty("exe_start_date")
  private Date executionStartDate;

  @JsonProperty("exe_end_date")
  private Date executionEndDate;

  @JsonProperty("user_id")
  private Long userId;

  @JsonProperty("note")
  private String note;

  @JsonProperty("class_name")
  private String className;

  @JsonProperty("test_case")
  private TestCase testCase;

  @JsonProperty("test_step_logs")
  private List<TestStepLog> testStepLogs;

  @JsonProperty("attachments")
  private List<Attachment> attachments;

  @JsonProperty("build_number")
  private String buildNumber;

  @JsonProperty("build_url")
  private String buildUrl;

  public TestLog() {
  }

  /**
   * @return the test log identifier.
   */
  public Long getId() {
    return id;
  }

  /**
   * Sets the test log identifier.
   *
   * @param id the given test log identifier to set.
   */
  public TestLog setId(Long id) {
    this.id = id;
    return this;
  }

  /**
   * Sets the test log identifier.
   *
   * @param id the given test log identifier to set.
   */
  public TestLog withId(Long id) {
    setId(id);
    return this;
  }

  /**
   * @return the test log status.
   */
  public ExecutionStatus getStatus() {
    return status;
  }

  /**
   * Sets the test log status.
   *
   * @param status the given test log status to set.
   */
  public TestLog setStatus(ExecutionStatus status) {
    this.status = status;
    return this;
  }

  /**
   * Sets the test log status.
   *
   * @param status the given test log status to set.
   */
  public TestLog withStatus(ExecutionStatus status) {
    setStatus(status);
    return this;
  }

  /**
   * @return the test case version id.
   */
  public Long getTestCaseVersionId() {
    return testCaseVersionId;
  }

  /**
   * Sets test case version id.
   *
   * @param testCaseVersionId the given test case version id to set.
   */
  public TestLog setTestCaseVersionId(Long testCaseVersionId) {
    this.testCaseVersionId = testCaseVersionId;
    return this;
  }

  /**
   * Sets test case version id.
   *
   * @param testCaseVersionId the given test case version id to set.
   */
  public TestLog withTestCaseVersionId(Long testCaseVersionId) {
    setTestCaseVersionId(testCaseVersionId);
    return this;
  }

  /**
   * @return the time when starting running test run.
   */
  @JsonSerialize(using = DateTimeSerializer.class, typing = JsonSerialize.Typing.STATIC, include = JsonSerialize.Inclusion.NON_NULL)
  public Date getExecutionStartDate() {
    return executionStartDate;
  }

  /**
   * Sets the time when starting running test run.
   *
   * @param executionStartDate the given time when starting running test run to set.
   */
  @JsonDeserialize(using = DateTimeDeserializer.class)
  public TestLog setExecutionStartDate(Date executionStartDate) {
    this.executionStartDate = executionStartDate;
    return this;
  }

  /**
   * Sets the time when starting running test run.
   *
   * @param executionStartDate the given time when starting running test run to set.
   */
  public TestLog withExecutionStartDate(Date executionStartDate) {
    setExecutionStartDate(executionStartDate);
    return this;
  }

  /**
   * @return the time when finishing running test run.
   */
  @JsonSerialize(using = DateTimeSerializer.class, typing = JsonSerialize.Typing.STATIC, include = JsonSerialize.Inclusion.NON_NULL)
  public Date getExecutionEndDate() {
    return executionEndDate;
  }

  /**
   * Sets the execution end date.
   *
   * @param executionEndDate the given execution end date.
   */
  @JsonDeserialize(using = DateTimeDeserializer.class)
  public TestLog setExecutionEndDate(Date executionEndDate) {
    this.executionEndDate = executionEndDate;
    return this;
  }

  /**
   * Sets the execution end date.
   *
   * @param executionEndDate the given execution end date.
   */
  public TestLog withExecutionEndDate(Date executionEndDate) {
    setExecutionEndDate(executionEndDate);
    return this;
  }

  /**
   * @return id of user who ran test run.
   */
  public Long getUserId() {
    return userId;
  }

  /**
   * Sets the id of user who ran test run.
   *
   * @param userId the given id of user who ran test run.
   */
  public TestLog setUserId(Long userId) {
    this.userId = userId;
    return this;
  }

  /**
   * Sets the id of user who ran test run.
   *
   * @param userId the given id of user who ran test run.
   */
  public TestLog withUserId(Long userId) {
    setUserId(userId);
    return this;
  }

  /**
   * @return the test log note.
   */
  public String getNote() {
    return note;
  }

  /**
   * Sets the test log note.
   *
   * @param note the given test log note to set.
   */
  public TestLog setNote(String note) {
    this.note = note;
    return this;
  }

  /**
   * Sets the test log note.
   *
   * @param note the given test log note to set.
   */
  public TestLog withNote(String note) {
    setNote(note);
    return this;
  }

  /**
   * @return the class name associated to test suite.
   */
  public String getClassName() {
    return className;
  }

  /**
   * Sets the class name associated to test suite.
   *
   * @param className the given class name to set.
   */
  public TestLog setClassName(String className) {
    this.className = className;
    return this;
  }

  /**
   * Sets the class name associated to test suite.
   *
   * @param className the given class name to set.
   */
  public TestLog withClassName(String className) {
    setClassName(className);
    return this;
  }

  /**
   * @return the test case instance.
   */
  public TestCase getTestCase() {
    return testCase;
  }

  /**
   * Sets the test case instance.
   *
   * @param testCase the given test case to set.
   */
  public TestLog setTestCase(TestCase testCase) {
    this.testCase = testCase;
    return this;
  }

  /**
   * Sets the test case instance.
   *
   * @param testCase the given test case to set.
   */
  public TestLog withTestCase(TestCase testCase) {
    setTestCase(testCase);
    return this;
  }

  /**
   * @return the list of {@link TestStepLog} objects.
   */
  public List<TestStepLog> getTestStepLogs() {
    if (null == testStepLogs) {
      return Collections.emptyList();
    }
    return testStepLogs;
  }

  /**
   * Sets the test step logs.
   *
   * @param testStepLogs the given list of test step logs to set.
   */
  public TestLog setTestStepLogs(List<TestStepLog> testStepLogs) {
    this.testStepLogs = testStepLogs;
    return this;
  }

  /**
   * Sets the test step logs.
   *
   * @param testStepLogs the given list of test step logs to set.
   */
  public TestLog withTestStepLogs(List<TestStepLog> testStepLogs) {
    setTestStepLogs(testStepLogs);
    return this;
  }

  /**
   * Add the test step log.
   *
   * @param testStepLog the given test step log to add.
   * @return current {@link TestLog} object.
   */
  public TestLog addTestStepLog(TestStepLog testStepLog) {
    if (testStepLogs == null) {
      testStepLogs = Lists.newLinkedList();
    }
    testStepLogs.add(testStepLog);
    return this;
  }

  /**
   * @return the list of testlog's attachments.
   */
  public List<Attachment> getAttachments() {
    if (null == attachments) {
      return Collections.emptyList();
    }

    return attachments;
  }

  /**
   * Sets the list of testlog's attachments.
   *
   * @param attachments the given list of testlog's attachment to set.
   */
  public TestLog setAttachments(List<Attachment> attachments) {
    if (attachments == null || attachments.size() == 0) {
      this.attachments = null;
    } else {
      this.attachments = attachments;
    }
    return this;
  }

  /**
   * Sets the list of testlog's attachments.
   *
   * @param attachments the given list of testlog's attachment to set.
   */
  public TestLog withAttachments(List<Attachment> attachments) {
    setAttachments(attachments);
    return this;
  }

  /**
   * Adds attachment to test log.
   *
   * @param attachment the given attachment to add to test log.
   * @return the {@link TestLog} object.
   */
  public TestLog addAttachment(Attachment attachment) {
    if (attachments == null) {
      attachments = new ArrayList<>();
    }
    attachments.add(attachment);
    return this;
  }

  @Override
  protected TestLog clone() {
    TestLog that = new TestLog();

    that.setPropertiesFrom(this);

    return that;
  }

  @Override
  public String elementName() {
    return "test-log";
  }

  @Override
  public String jsonElementName() {
    return "test_log";
  }
  /**
   * Get build number
   *
   * @return build number value
   */
  public String getBuildNumber() {
    return buildNumber;
  }

  /**
   * Set build number to test log
   *
   * @param buildNumber build number value
   * @return this instance
   */
  public TestLog setBuildNumber(String buildNumber) {
    this.buildNumber = buildNumber;
    return this;
  }

  /**
   * Set build number to test log
   *
   * @param buildNumber build number value
   * @return this instance
   */
  public TestLog withBuildNumber(String buildNumber) {
    return this.setBuildNumber(buildNumber);
  }

  /**
   * Get build url
   *
   * @return build url
   */
  public String getBuildUrl() {
    return buildUrl;
  }

  /**
   * Set build url
   *
   * @param buildUrl build url value
   * @return this instance
   */
  public TestLog setBuildUrl(String buildUrl) {
    this.buildUrl = buildUrl;
    return this;
  }

  /**
   * Set build url
   *
   * @param buildUrl build url
   * @return this instance
   */
  public TestLog withBuildUrl(String buildUrl) {
    return setBuildUrl(buildUrl);
  }
}
