/*
 * Copyright (c) 2010-2020 Tencent Cloud. All rights reserved.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in all
 *  copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 *  SOFTWARE.
 */

package com.tencent.libavif;

import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.ColorFilter;
import android.graphics.Paint;
import android.graphics.PixelFormat;
import android.graphics.Rect;
import android.graphics.drawable.Animatable;
import android.graphics.drawable.Drawable;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.os.SystemClock;
import android.util.Log;

import androidx.annotation.NonNull;

import com.tencent.libqcloudavif.*;
import com.tencent.qcloud.image.avif.Avif;

import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

public class AvifSequenceDrawable extends Drawable implements Animatable {
    private static final String TAG = "AvifSequence";
    public static final boolean DEBUG = true;
    /**
     * These constants are chosen to imitate common browser behavior for WebP/GIF.
     * If other decoders are added, this behavior should be moved into the WebP/GIF decoders.
     * <p>
     * Note that 0 delay is undefined behavior in the GIF standard.
     */
    private static final long MIN_DELAY_MS = 20;
    private static final long DEFAULT_DELAY_MS = 100;

    private final ScheduledThreadPoolExecutor mExecutor = AvifRenderingExecutor.getInstance();


    private static final BitmapProvider sAllocatingBitmapProvider = new BitmapProvider() {
        @Override
        public Bitmap acquireBitmap(int minWidth, int minHeight) {
            return Bitmap.createBitmap(minWidth, minHeight, Bitmap.Config.ARGB_8888);
        }

        @Override
        public void releaseBitmap(Bitmap bitmap) {
        }
    };
    private final Handler mInvalidateHandler = new Handler(Looper.getMainLooper()) {
        @Override
        public void handleMessage(@NonNull Message msg) {
            invalidateSelf();
        }
    };
    private final int mWidth;
    private final int mHeight;
    private long mNextFrameRenderTime = Long.MIN_VALUE;

    boolean mIsRunning = false;
    private ScheduledFuture<?> mDecodeTaskFuture;

    /**
     * Register a callback to be invoked when a FrameSequenceDrawable finishes looping.
     * @param onFinishedListener FinishedListener
     * @see #setLoopBehavior(int)
     */
    public void setOnFinishedListener(OnFinishedListener onFinishedListener) {
        mOnFinishedListener = onFinishedListener;
    }

    /**
     * Loop a finite number of times, which can be set using setLoopCount. Default to loop once.
     */
    public static final int LOOP_FINITE = 1;

    /**
     * Loop continuously. The OnFinishedListener will never be called.
     */
    public static final int LOOP_INF = 2;

    /**
     * Use loop count stored in source data, or LOOP_ONCE if not present.
     */
    public static final int LOOP_DEFAULT = 3;

    /**
     * Define looping behavior of frame sequence.
     * <p>
     * Must be one of  LOOP_INF, LOOP_DEFAULT, or LOOP_FINITE.
     * @param loopBehavior loopBehavior
     */
    public void setLoopBehavior(int loopBehavior) {
        mLoopBehavior = loopBehavior;
    }

    /**
     * Set the number of loops in LOOP_FINITE mode. The number must be a postive integer.
     * @param loopCount loop Count
     */
    public void setLoopCount(int loopCount) {
        mLoopCount = loopCount;
    }

    private final AvifDecoder mAvifDecoder;
    private final long bytesLength;

    private final Paint mPaint = new Paint(Paint.FILTER_BITMAP_FLAG);
    private final Rect mSrcRect;

    //Protects the fields below
    private final Object mLock = new Object();

    private final BitmapProvider mBitmapProvider;
    private boolean mDestroyed = false;
    private Bitmap mBitmap;

    private int mCurrentLoop;
    private int mLoopBehavior = LOOP_DEFAULT;
    private int mLoopCount = 1;

    private int mImageIndex;

    private OnFinishedListener mOnFinishedListener;

    /**
     * Runs on decoding thread, only modifies mBackBitmap's pixels
     */
    private final Runnable mDecodeTask = new Runnable() {
        @Override
        public void run() {
            synchronized (mLock) {
                if (!mIsRunning || mDestroyed) return;

                final long start = SystemClock.uptimeMillis();

                boolean exceptionDuringDecode = false;
                long invalidateTimeMs = 0;
                try {
                    Avif.AvifFrame avifFrame = Avif.animationDecodeFrame(mAvifDecoder, bytesLength, mImageIndex, mBitmap);
                    invalidateTimeMs = avifFrame.getDelayTime();
//                    Log.d(TAG, "run: invalidateMs " + invalidateTimeMs + "   " + mAvifDecoder.getImageIndex() + "   " + mAvifDecoder.getImageCount());
                } catch (Exception e) {
                    // Exception during decode: continue, but delay next frame indefinitely.
                    Log.e(TAG, "exception during decode: " + e);
                    exceptionDuringDecode = true;
                }

                if (invalidateTimeMs < MIN_DELAY_MS) {
                    invalidateTimeMs = DEFAULT_DELAY_MS;
                }
                mNextFrameRenderTime = exceptionDuringDecode ? Long.MIN_VALUE : invalidateTimeMs + start;
//                final int imageIndex = mAvifDecoder.getImageIndex();
                if (mImageIndex >= mAvifDecoder.getImageCount() - 1) {
                    mImageIndex = 0;
                    mCurrentLoop++;
                    if ((mLoopBehavior == LOOP_FINITE && mCurrentLoop == mLoopCount)) {
                        scheduleSelf(mFinishedCallbackRunnable, 0);
                    }
                } else {
                    mImageIndex++;
                }
                mInvalidateHandler.sendEmptyMessage(0);

            }
        }
    };

    private final Runnable mFinishedCallbackRunnable = new Runnable() {
        @Override
        public void run() {
            stop();
            if (mOnFinishedListener != null) {
                mOnFinishedListener.onFinished(AvifSequenceDrawable.this);
            }
        }
    };

    private static Bitmap acquireAndValidateBitmap(BitmapProvider bitmapProvider,
                                                   int minWidth, int minHeight) {
        Bitmap bitmap = bitmapProvider.acquireBitmap(minWidth, minHeight);

        if (bitmap.getWidth() < minWidth
                || bitmap.getHeight() < minHeight
                || bitmap.getConfig() != Bitmap.Config.ARGB_8888) {
            throw new IllegalArgumentException("Invalid bitmap provided");
        }

        return bitmap;
    }

    public AvifSequenceDrawable(AvifDecoder avifDecoder, long bytesLength) {
        this(avifDecoder, bytesLength, sAllocatingBitmapProvider);
    }

    public AvifSequenceDrawable(AvifDecoder avifDecoder, long bytesLength, BitmapProvider bitmapProvider) {
        if (avifDecoder == null || bitmapProvider == null) throw new IllegalArgumentException();

        this.bytesLength = bytesLength;
        this.mAvifDecoder = avifDecoder;
        mWidth = this.mAvifDecoder.getWidth();
        mHeight = this.mAvifDecoder.getHeight();

        mBitmapProvider = bitmapProvider;
        mBitmap = acquireAndValidateBitmap(bitmapProvider, mWidth, mHeight);
        mSrcRect = new Rect(0, 0, mWidth, mHeight);

        synchronized (mLock) {
            Avif.animationDecodeFrame(mAvifDecoder, bytesLength, mImageIndex, mBitmap);
        }
    }

    private void checkDestroyedLocked() {
        if (mDestroyed) {
            throw new IllegalStateException("Cannot perform operation on recycled drawable");
        }
    }


    private void scheduleNextRender() {
        if (isRunning() && mNextFrameRenderTime != Long.MIN_VALUE) {
            final long renderDelay = Math.max(0, mNextFrameRenderTime - SystemClock.uptimeMillis());
            mNextFrameRenderTime = Long.MIN_VALUE;
            mExecutor.remove(mDecodeTask);
            mDecodeTaskFuture = mExecutor.schedule(mDecodeTask, renderDelay, TimeUnit.MILLISECONDS);
        }
    }

    public boolean isDestroyed() {
        synchronized (mLock) {
            return mDestroyed;
        }
    }

    /**
     * Marks the drawable as permanently recycled (and thus unusable), and releases any owned
     * Bitmaps drawable to its BitmapProvider, if attached.
     * <p>
     * If no BitmapProvider is attached to the drawable, recycle() is called on the Bitmaps.
     */
    public void destroy() {
        if (mBitmapProvider == null) {
            throw new IllegalStateException("BitmapProvider must be non-null");
        }
        synchronized (mLock) {
            checkDestroyedLocked();

            mDestroyed = true;

            mAvifDecoder.destroy();
        }

        mBitmapProvider.releaseBitmap(mBitmap);
    }


    @Override
    public void draw(@NonNull Canvas canvas) {
        synchronized (mLock) {
            canvas.drawBitmap(mBitmap, mSrcRect, getBounds(), mPaint);
        }
    }


    @Override
    public void invalidateSelf() {
        super.invalidateSelf();

        scheduleNextRender();
    }

    @Override
    public void start() {
        synchronized (mLock) {
            if (mIsRunning) {
                return;
            }
            mIsRunning = true;
        }

        checkDestroyedLocked();
        mNextFrameRenderTime = SystemClock.uptimeMillis();
        mCurrentLoop = 0;
        scheduleNextRender();
    }

    @Override
    public void stop() {
        synchronized (mLock) {
            if (!mIsRunning) {
                return;
            }
            mIsRunning = false;
        }
        if (mDecodeTaskFuture != null) {
            mDecodeTaskFuture.cancel(false);
        }

        mCurrentLoop = 0;

        // 加上reset() 在多动图列表 例如内存测试时会崩溃（但是destroy()前调用reset()就没事 例如 fresco的AvisImage.dispose()）
        // libc: Fatal signal 11 (SIGSEGV), code 1 (SEGV_MAPERR), fault addr 0xbf31231b in tid 5728 (pool-4-thread-1), pid 5673 (geloader.sample)
//        synchronized (mLock) {
//            mAvifDecoder.reset();
//        }
    }

    @Override
    public boolean isRunning() {
        synchronized (mLock) {
            return mIsRunning && !mDestroyed;
        }
    }


    @Override
    public boolean setVisible(boolean visible, boolean restart) {
        boolean changed = super.setVisible(visible, restart);
//        if (DEBUG) {
//            Log.d(TAG, "setVisible: " + visible + "   " + restart + "   " + changed);
//        }

        if (visible) {
            if (restart) {
                stop();
                start();
            }
            if (changed) {
                start();
            }
        } else if (changed) {
            stop();
        }

        return changed;
    }

    // drawing properties

    @Override
    public void setFilterBitmap(boolean filter) {
        mPaint.setFilterBitmap(filter);
    }

    @Override
    public void setAlpha(int alpha) {
        mPaint.setAlpha(alpha);
    }

    @Override
    public void setColorFilter(ColorFilter colorFilter) {
        mPaint.setColorFilter(colorFilter);
    }

    @Override
    public int getIntrinsicWidth() {
        return mWidth;
    }

    @Override
    public int getIntrinsicHeight() {
        return mHeight;
    }

    @Override
    public int getOpacity() {
        return PixelFormat.TRANSLUCENT;
    }

    public int getSize() {
        return mWidth * mHeight * 4;
    }

    /**
     * Default executor for rendering tasks - {@link ScheduledThreadPoolExecutor}
     * with 1 worker thread and {@link DiscardPolicy}.
     */
    static final class AvifRenderingExecutor extends ScheduledThreadPoolExecutor {

        private static final AvifRenderingExecutor INSTANCE = new AvifRenderingExecutor();

        static AvifRenderingExecutor getInstance() {
            return INSTANCE;
        }

        private AvifRenderingExecutor() {
            super(1, new DiscardPolicy());
        }
    }

    public interface OnFinishedListener {
        /**
         * Called when a FrameSequenceDrawable has finished looping.
         * <p>
         * Note that this is will not be called if the drawable is explicitly
         * stopped, or marked invisible.
         * @param drawable drawable
         */
        public void onFinished(AvifSequenceDrawable drawable);
    }

    public interface BitmapProvider {
        /**
         * Called by FrameSequenceDrawable to aquire an 8888 Bitmap with minimum dimensions.
         * @param minWidth minWidth
         * @param minHeight minHeight
         * @return Bitmap
         */
        public Bitmap acquireBitmap(int minWidth, int minHeight);

        /**
         * Called by FrameSequenceDrawable to release a Bitmap it no longer needs. The Bitmap
         * will no longer be used at all by the drawable, so it is safe to reuse elsewhere.
         * <p>
         * This method may be called by FrameSequenceDrawable on any thread.
         * @param bitmap bitmap
         */
        public void releaseBitmap(Bitmap bitmap);
    }
}
