/*
 * Copyright (c) 2010-2020 Tencent Cloud. All rights reserved.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in all
 *  copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 *  SOFTWARE.
 */

package com.tencent.qcloud.image.avif;

import android.graphics.Bitmap;
import android.graphics.Point;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.bumptech.glide.load.engine.bitmap_recycle.ArrayPool;
import com.bumptech.glide.load.engine.bitmap_recycle.BitmapPool;
import com.tencent.libqcloudavif.AvifDecoder;
import com.tencent.libqcloudavif.AvifImage;
import com.tencent.qcloud.image.avif.track.AvifTrack;
import com.tencent.qcloud.image.decoder.exception.CiAnimationDecodeException;
import com.tencent.qcloud.image.decoder.exception.CiDecodeException;
import com.tencent.qcloud.image.decoder.exception.CiPrepareException;

import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;

/**
 * AVIF解码工具类
 */
public class Avif {
    private static final String TAG = "AVIFDecoderUtil";

    static {
        System.loadLibrary("qcloud-avif");
    }

    // AVIF 文件头字节数
    public static final int AVIF_HEADER_SIZE = 12;

    /**
     * 原图解码
     * @param bytes 图片的字节数组
     * @return 结果bitmap
     */
    public static Bitmap decode(@NonNull byte[] bytes) {
        return decode(bytes, -1, null, null);
    }


    /**
     * 宽度等比解码
     * @param bytes 图片的字节数组
     * @param dstWidth 解码目标宽度（仅支持等比缩小）
     * @return 结果bitmap
     */
    public static Bitmap decode(@NonNull byte[] bytes, int dstWidth){
        return decode(bytes, dstWidth, null, null);
    }

    /**
     * 宽度等比解码
     * @param bytes 图片的字节数组
     * @param dstWidth 解码目标宽度（仅支持等比缩小）
     * @param bitmapPool 用于获取bitmap的bitmapPool
     * @return 结果bitmap
     */
    public static Bitmap decode(@NonNull byte[] bytes, int dstWidth, @NonNull BitmapPool bitmapPool) {
        return decode(bytes, dstWidth, null, bitmapPool);
    }

    /**
     * 宽度等比解码
     * @param bytes 图片的字节数组
     * @param dstWidth 解码目标宽度（仅支持等比缩小）
     * @param bitmap 解码到该位图（也可以传空 将会生成一个新的位图）
     * @param bitmapPool 用于获取bitmap的bitmapPool
     * @return 结果bitmap
     */
    public static Bitmap decode(@NonNull byte[] bytes, int dstWidth, @Nullable Bitmap bitmap, @Nullable BitmapPool bitmapPool) {
        final AvifDecoder decoder = AvifDecoder.fromByteArray(bytes);
        Bitmap bitmapResult = decode(decoder, dstWidth, bytes.length, bitmap, bitmapPool);
        decoder.destroy();
        return bitmapResult;
    }

    /**
     * 区域缩放解码
     * @param bytes 图片的字节数组
     * @param x 区域左上角x坐标
     * @param y 区域左上角y坐标
     * @param width 区域宽度
     * @param height 区域高度
     * @param inSampleSize 缩放比, 大于1的时候才生效，小于等于1的情况下不作缩放
     * @return 结果bitmap
     */
    public static Bitmap decode(byte[] bytes, int x, int y, int width, int height, int inSampleSize){
        AvifDecoder avifDecoder = AvifDecoder.fromByteArray(bytes);
        int result = avifDecoder.nextImage();
        if(result != 0){
            String diag = avifDecoder.getDiag();
            if(diag == null) diag = "";
            long start = System.nanoTime();
            AvifTrack.getInstance().reportPrepareError(start, bytes.length, String.valueOf(result), diag);
            throw new CiPrepareException(
                    CiDecodeException.CI_DECODE_TYPE_AVIF,
                    CiPrepareException.CI_PREPARE_STEP_START,
                    result, bytes.length
            );
        } else {
            Bitmap bitmap = decodeRegion(avifDecoder, bytes.length, x, y, width, height, inSampleSize);
            avifDecoder.destroy();
            return bitmap;
        }
    }

    /**
     * 根据原图宽高和目标宽度 计算目标高度
     * @param imageWidth 原图宽度
     * @param imageHeight 原图高度
     * @param dstWidth 目标宽度
     * @return 目标宽高
     */
    public static Point proportionalScaling(int imageWidth, int imageHeight, int dstWidth){
        int dstHeight;
        if (dstWidth <= 0 || dstWidth == imageWidth) {
            dstWidth = imageWidth;
            dstHeight = imageHeight;
        } else {
            dstHeight = (int) ((double) imageHeight / (double) imageWidth * dstWidth);
            if (dstWidth > imageWidth || dstHeight > imageHeight) {
                dstWidth = imageWidth;
                dstHeight = imageHeight;
            }
        }
        return new Point(dstWidth, dstHeight);
    }

    /**
     * 宽度等比解码
     * @param decoder avif解码器实例
     * @param dstWidth 解码目标宽度（仅支持等比缩小）
     * @param dataSize 需要解码的数据大小
     * @param bitmap 解码到该位图（也可以传空 将会生成一个新的位图）
     * @param bitmapPool 用于获取bitmap的bitmapPool
     * @return 结果bitmap
     */
    public static Bitmap decode(@NonNull AvifDecoder decoder, int dstWidth, long dataSize, @Nullable Bitmap bitmap, @Nullable BitmapPool bitmapPool){
        int imageWidth = decoder.getWidth();
        int imageHeight = decoder.getHeight();

        long start = System.nanoTime();
        // SIZE_ORIGINAL
        if(dstWidth <= 0){
            dstWidth = -1;
        }
        try {
            Point dstPoint = proportionalScaling(imageWidth, imageHeight, dstWidth);

            if(bitmapPool != null){
                bitmap = bitmapPool.getDirty(dstPoint.x, dstPoint.y, Bitmap.Config.ARGB_8888);
            }
            if(bitmap == null){
                bitmap = Bitmap.createBitmap(dstPoint.x, dstPoint.y, Bitmap.Config.ARGB_8888);
            }
            int result = decoder.nextImage();
            if(result != 0){
                String diag = decoder.getDiag();
                if(diag == null) diag = "";
                AvifTrack.getInstance().reportError(start, dataSize, imageWidth, imageHeight, String.valueOf(result), diag);
                throw new CiDecodeException(
                        CiDecodeException.CI_DECODE_TYPE_AVIF, result, diag,
                        dataSize, dstPoint.x, dstPoint.y, imageWidth, imageHeight
                );
            } else {
                // 如果接受的bitmap宽高和要解码的宽高不一致，则以bitmap为准
                if(bitmap.getWidth() != dstPoint.x || bitmap.getHeight() != dstPoint.y){
                    dstPoint.set(bitmap.getWidth(), bitmap.getHeight());
                }
                final AvifImage image = decoder.getScaledImage(dstPoint.x, dstPoint.y);
                image.getBitmap(bitmap, true);
                AvifTrack.getInstance().reportSuccess(start, dataSize, imageWidth, imageHeight);
                image.destroy();
                return bitmap;
            }
        } catch (Exception e){
            if(e instanceof CiDecodeException){
                throw e;
            } else {
                String diag = decoder.getDiag();
                if(diag == null) diag = "";
                AvifTrack.getInstance().reportError(start, dataSize, imageWidth, imageHeight, e.getClass().getName(), e.getMessage() + " diag=" + diag);
                throw new CiDecodeException(
                        CiDecodeException.CI_DECODE_TYPE_AVIF, e, diag,
                        dataSize, dstWidth, -1, imageWidth, imageHeight
                );
            }
        }
    }

    /**
     * 通过解码器解码其中一帧（需要提前创建AvifDecoder）
     * @param decoder 解码器（提前创建AvifDecoder）
     * @param bytesLength 需要解码的数据长度
     * @param index 帧索引
     * @param bitmap 解码到该位图（也可以传空 将会生成一个新的位图在TpgFrame中返回）
     * @return 动图解码一帧返回内容 包括位图和延时
     */
    public static AvifFrame animationDecodeFrame(@NonNull AvifDecoder decoder, long bytesLength, int index, @Nullable Bitmap bitmap){
        long start = System.nanoTime();
        try {
            int result = decoder.nthImage(index);
            if(result != 0){
                String diag = decoder.getDiag();
                if(diag == null) diag = "";
                AvifTrack.getInstance().reportAnimationError(start, bytesLength, index, decoder.getImageCount(),
                        decoder.getWidth(), decoder.getHeight(),
                        String.valueOf(result), diag);
                throw new CiAnimationDecodeException(
                        CiAnimationDecodeException.CI_DECODE_TYPE_AVIF,
                        result, diag, bytesLength, index, decoder.getImageCount(),
                        decoder.getWidth(), decoder.getHeight()
                );
            } else {
                AvifImage avifImage = decoder.getImage();
                int width = avifImage.getWidth();
                int height = avifImage.getHeight();
                if(bitmap == null){
                    bitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
                }
//                bitmap = avifImage.getBitmap(true);
                avifImage.getBitmap(bitmap, true);
                AvifTrack.getInstance().reportAnimationSuccess(start, bytesLength, index, decoder.getImageCount(), width, height);
                long duration = avifImage.getDuration();
                avifImage.destroy();
                return new AvifFrame(bitmap, duration);
            }
        } catch (Exception e){
            if(e instanceof CiAnimationDecodeException){
                throw e;
            } else {
                String diag = decoder.getDiag();
                if(diag == null) diag = "";
                AvifTrack.getInstance().reportAnimationError(start, bytesLength, index, decoder.getImageCount(), decoder.getWidth(), decoder.getHeight(),
                        e.getClass().getName(), e.getMessage() + " diag=" + diag);
                throw new CiAnimationDecodeException(
                        CiAnimationDecodeException.CI_DECODE_TYPE_AVIF,
                        e, diag, bytesLength, index, decoder.getImageCount(),
                        decoder.getWidth(), decoder.getHeight()
                );
            }
        }
    }

    /**
     * 通过解码器解码其中一帧（需要提前创建AvifDecoder）
     * @param decoder 解码器（提前创建AvifDecoder）
     * @param bytesLength 需要解码的数据长度
     * @param index 帧索引
     * @return 动图解码一帧返回内容 包括位图和延时
     */
    public static AvifImage animationDecodeFrame(@NonNull AvifDecoder decoder, long bytesLength, int index){
        long start = System.nanoTime();
        try {
            int result = decoder.nthImage(index);
            if(result != 0){
                String diag = decoder.getDiag();
                if(diag == null) diag = "";
                AvifTrack.getInstance().reportAnimationError(start, bytesLength, index, decoder.getImageCount(),
                        decoder.getWidth(), decoder.getHeight(),
                        String.valueOf(result), diag);
                throw new CiAnimationDecodeException(
                        CiAnimationDecodeException.CI_DECODE_TYPE_AVIF,
                        result, diag, bytesLength, index, decoder.getImageCount(),
                        decoder.getWidth(), decoder.getHeight()
                );
            } else {
                AvifImage avifImage = decoder.getImage();
                int width = avifImage.getWidth();
                int height = avifImage.getHeight();
                AvifTrack.getInstance().reportAnimationSuccess(start, bytesLength, index, decoder.getImageCount(), width, height);
                return avifImage;
            }
        } catch (Exception e){
            if(e instanceof CiAnimationDecodeException){
                throw e;
            } else {
                String diag = decoder.getDiag();
                if(diag == null) diag = "";
                AvifTrack.getInstance().reportAnimationError(start, bytesLength, index, decoder.getImageCount(), decoder.getWidth(), decoder.getHeight(),
                        e.getClass().getName(), e.getMessage() + " diag=" + diag);
                throw new CiAnimationDecodeException(
                        CiAnimationDecodeException.CI_DECODE_TYPE_AVIF,
                        e, diag, bytesLength, index, decoder.getImageCount(),
                        decoder.getWidth(), decoder.getHeight()
                );
            }
        }
    }

    /**
     * 解码动图某一帧
     * @param bytes 需要解码的数据（字节数组）
     * @param index 帧索引
     * @return 动图解码一帧返回内容 包括位图和延时
     */
    public static AvifFrame decodeAnimationFrame(@NonNull byte[] bytes, int index){
        AvifDecoder decoder = AvifDecoder.fromByteArray(bytes);
        AvifFrame avifFrame = animationDecodeFrame(decoder, bytes.length, index, null);
        decoder.destroy();
        return avifFrame;
    }

    /**
     * 解码动图某一帧
     * @param bytes 需要解码的数据（字节数组）
     * @param index 帧索引
     * @param bitmap 解码到该位图（也可以传空 将会生成一个新的位图在TpgFrame中返回）
     * @return 动图解码一帧返回内容 包括位图和延时
     */
    public static AvifFrame decodeAnimationFrame(@NonNull byte[] bytes, int index, @Nullable Bitmap bitmap){
        AvifDecoder decoder = AvifDecoder.fromByteArray(bytes);
        AvifFrame avifFrame = animationDecodeFrame(decoder, bytes.length, index, bitmap);
        decoder.destroy();
        return avifFrame;
    }

    /**
     * 局部解码（需要提前创建AvifDecoder并nextImage）
     * @param decoder 解码器（提前创建AvifDecoder并nextImage）
     * @param bytesLength 需要解码的数据长度
     * @param x 区域左上角x坐标
     * @param y 区域左上角y坐标
     * @param width 区域宽度
     * @param height 区域高度
     * @param inSampleSize 缩放比, 大于1的时候才生效，小于等于1的情况下不作缩放
     * @return 解码位图
     */
    public static Bitmap decodeRegion(@NonNull AvifDecoder decoder, long bytesLength, int x, int y, int width, int height, int inSampleSize){
        x = x/inSampleSize;
        y = y/inSampleSize;
        width = width/inSampleSize;
        height = height/inSampleSize;

        // YUV420P的UV分量要求必须能被2整除
        if(x % 2 != 0){
            x -= 1;
        }
        if(y % 2 != 0){
            y -= 1;
        }

        long start = System.nanoTime();
        try {
            int imageWidth = decoder.getWidth();
            int imageHeight = decoder.getHeight();
            AvifImage avifImage = decoder.getRectedImage(x, y, width, height, inSampleSize);
            Bitmap bitmap = avifImage.getBitmap(true);
            AvifTrack.getInstance().reportRectedSuccess(start, bytesLength, width, height, x , y, inSampleSize, imageWidth, imageHeight);
            avifImage.destroy();
            return bitmap;
        } catch (Exception e){
            if(e instanceof CiDecodeException){
                throw e;
            } else {
                String diag = decoder.getDiag();
                if(diag == null) diag = "";
                AvifTrack.getInstance().reportRectedError(start, bytesLength, width, height, x , y, inSampleSize, decoder.getWidth(), decoder.getHeight(),
                        e.getClass().getName(), e.getMessage() + " diag=" + diag);
                throw new CiDecodeException(
                        CiDecodeException.CI_DECODE_TYPE_AVIF, e, diag,
                        bytesLength, decoder.getWidth(), decoder.getHeight(),
                        x , y, width, height, inSampleSize
                );
            }
        }
    }



    /**
     * 动图解码一帧返回内容 包括位图和延时
     */
    public static class AvifFrame{
        private final Bitmap bitmap;
        private final long delayTime;

        public AvifFrame(Bitmap bitmap, long delayTime) {
            this.bitmap = bitmap;
            this.delayTime = delayTime;
        }

        public Bitmap getBitmap() {
            return bitmap;
        }

        public long getDelayTime() {
            return delayTime;
        }
    }

    public static boolean isAvif(byte[] buffer){
        return AvifDecoder.isAvif(buffer);
    }
    public static boolean isAvif(@NonNull InputStream source, @NonNull ArrayPool byteArrayPool){
        byte[] arr = byteArrayPool.get(AVIF_HEADER_SIZE, byte[].class);
        try {
            if(source.read(arr) == -1){
                return false;
            }
        } catch (IOException e) {
            e.printStackTrace();
            return false;
        }
        byteArrayPool.put(arr);

        if (arr == null) {
            return false;
        }
        return isAvif(arr);
    }
    public static boolean isAvif(@NonNull ByteBuffer source){
        byte[] arr = new byte[AVIF_HEADER_SIZE];
        source.get(arr, 0, AVIF_HEADER_SIZE);
        return isAvif(arr);
    }

    public static boolean isAvis(byte[] buffer){
        return AvifDecoder.isAvis(buffer);
    }
    public static boolean isAvis(@NonNull InputStream source, @NonNull ArrayPool byteArrayPool){
        byte[] arr = byteArrayPool.get(AVIF_HEADER_SIZE, byte[].class);
        try {
            if(source.read(arr) == -1){
                return false;
            }
        } catch (IOException e) {
            e.printStackTrace();
            return false;
        }
        byteArrayPool.put(arr);

        if (arr == null) {
            return false;
        }
        return isAvis(arr);
    }
    public static boolean isAvis(@NonNull ByteBuffer source){
        byte[] arr = new byte[AVIF_HEADER_SIZE];
        source.get(arr, 0, AVIF_HEADER_SIZE);
        return isAvis(arr);
    }

    public static AvifDecoder getAvifDecoder(byte[] buffer){
        return AvifDecoder.fromByteArray(buffer);
    }

    public static AvifDecoder getAvifDecoder(@NonNull ByteBuffer source){
        return AvifDecoder.fromByteBuffer(source);
    }

    public static Point getDecoderWH(byte[] buffer){
        AvifDecoder avifDecoder = AvifDecoder.fromByteArray(buffer);
        return new Point(avifDecoder.getWidth(), avifDecoder.getHeight());
    }
}
