/*
 * Copyright (c) 2010-2020 Tencent Cloud. All rights reserved.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in all
 *  copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 *  SOFTWARE.
 */

package com.tencent.qcloud.image.avif.subsampling;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Point;
import android.graphics.Rect;
import android.net.Uri;
import android.os.Build;

import androidx.annotation.NonNull;

import com.davemorrissey.labs.subscaleview.decoder.ImageRegionDecoder;
import com.tencent.libqcloudavif.AvifDecoder;
import com.tencent.qcloud.image.avif.Avif;
import com.tencent.qcloud.image.avif.track.AvifTrack;
import com.tencent.qcloud.image.decoder.exception.CiDecodeException;
import com.tencent.qcloud.image.decoder.exception.CiPrepareException;
import com.tencent.qcloud.image.decoder.subsampling.SubsamplingUtil;

import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

public class AvifSubsamplingImageRegionDecoder implements ImageRegionDecoder {
    private AvifDecoder avifDecoder = null;
    private long bytesLength;
    private final ReadWriteLock decoderLock = new ReentrantReadWriteLock(true);

    @NonNull
    @Override
    public Point init(Context context, @NonNull Uri uri) throws Exception {
        byte[] bytes = SubsamplingUtil.getBytes(context, uri);
        bytesLength = bytes.length;
        if(Avif.isAvif(bytes) || Avif.isAvis(bytes)){
            avifDecoder = Avif.getAvifDecoder(bytes);
            int result = avifDecoder.nextImage();
            if(result != 0){
                String diag = avifDecoder.getDiag();
                if(diag == null) diag = "";
                long start = System.nanoTime();
                AvifTrack.getInstance().reportPrepareError(start, bytesLength, String.valueOf(result), diag);
                throw new CiPrepareException(
                        CiDecodeException.CI_DECODE_TYPE_AVIF,
                        CiPrepareException.CI_PREPARE_STEP_START,
                        result, bytesLength
                );
            } else {
                return new Point(avifDecoder.getWidth(), avifDecoder.getHeight());
            }
        } else {
            throw new RuntimeException("Non-avif format, please use the default decoder(SkiaImageRegionDecoder) or other decoder");
        }
    }

    @NonNull
    @Override
    public Bitmap decodeRegion(@NonNull Rect sRect, int sampleSize) {
        getDecodeLock().lock();
        try {
            if (avifDecoder != null) {
                Bitmap bitmap = Avif.decodeRegion(avifDecoder, bytesLength, sRect.left, sRect.top, sRect.width(), sRect.height(), sampleSize);
                if (bitmap == null) {
                    throw new RuntimeException("avif image decoder returned null bitmap - image format may not be supported");
                }
                return bitmap;
            } else {
                throw new IllegalStateException("Cannot decode region after decoder has been recycled");
            }
        } finally {
            getDecodeLock().unlock();
        }
    }

    @Override
    public boolean isReady() {
        return avifDecoder != null;
    }

    @Override
    public void recycle() {
        decoderLock.writeLock().lock();
        try {
            if(avifDecoder != null) {
                avifDecoder.reset();
                avifDecoder.destroy();
                avifDecoder = null;
            }
        } finally {
            decoderLock.writeLock().unlock();
        }
    }

    private Lock getDecodeLock() {
        if (Build.VERSION.SDK_INT < 21) {
            return decoderLock.writeLock();
        } else {
            return decoderLock.readLock();
        }
    }
}
