/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.qcloud.chdfs.permission.RangerAccessType;
import com.qcloud.cos.utils.StringUtils;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BufferedFSInputStream;
import org.apache.hadoop.fs.CosNConfigKeys;
import org.apache.hadoop.fs.CosNCopyFileContext;
import org.apache.hadoop.fs.CosNCopyFileTask;
import org.apache.hadoop.fs.CosNDeleteFileContext;
import org.apache.hadoop.fs.CosNDeleteFileTask;
import org.apache.hadoop.fs.CosNExtendedFSDataOutputStream;
import org.apache.hadoop.fs.CosNFSDataOutputStream;
import org.apache.hadoop.fs.CosNFSInputStream;
import org.apache.hadoop.fs.CosNFileStatus;
import org.apache.hadoop.fs.CosNPartialListing;
import org.apache.hadoop.fs.CosNResultInfo;
import org.apache.hadoop.fs.CosNSymlinkMetadata;
import org.apache.hadoop.fs.CosNUtils;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FSInputStream;
import org.apache.hadoop.fs.FSLinkResolver;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileChecksum;
import org.apache.hadoop.fs.FileMetadata;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileSystemLinkResolver;
import org.apache.hadoop.fs.NativeFileSystemStore;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathIOException;
import org.apache.hadoop.fs.RangerCredentialsClient;
import org.apache.hadoop.fs.UnresolvedLinkException;
import org.apache.hadoop.fs.UnsupportedFileSystemException;
import org.apache.hadoop.fs.XAttrSetFlag;
import org.apache.hadoop.fs.cosn.BufferPool;
import org.apache.hadoop.fs.cosn.CRC32CCheckSum;
import org.apache.hadoop.fs.cosn.CRC64Checksum;
import org.apache.hadoop.fs.cosn.LocalRandomAccessMappedBufferPool;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.Progressable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CosNFileSystem
extends FileSystem {
    private static final Logger LOG = LoggerFactory.getLogger(CosNFileSystem.class);
    static final String SCHEME = "cosn";
    static final String PATH_DELIMITER = "/";
    static final Charset METADATA_ENCODING = StandardCharsets.UTF_8;
    static final int MAX_XATTR_SIZE = 1024;
    static final int BUCKET_LIST_LIMIT = 999;
    static final int POSIX_BUCKET_LIST_LIMIT = 5000;
    private URI uri;
    private String bucket;
    private boolean isPosixBucket;
    private NativeFileSystemStore nativeStore;
    private boolean isDefaultNativeStore;
    private boolean isCreateRecursiveCheckDstDir;
    private Path workingDir;
    private String owner = "Unknown";
    private String group = "Unknown";
    private ExecutorService boundedIOThreadPool;
    private ExecutorService boundedCopyThreadPool;
    private RangerCredentialsClient rangerCredentialsClient;
    static final String CSE_ALOGORITHM_USER_METADATA = "client-side-encryption-cek-alg";
    private int symbolicLinkSizeThreshold;

    public CosNFileSystem() {
    }

    public CosNFileSystem(NativeFileSystemStore nativeStore) {
        this.nativeStore = nativeStore;
        this.isDefaultNativeStore = false;
    }

    public CosNFileSystem withBucket(String bucket) {
        this.bucket = bucket;
        return this;
    }

    public CosNFileSystem withPosixBucket(boolean isPosixBucket) {
        this.isPosixBucket = isPosixBucket;
        return this;
    }

    public CosNFileSystem withStore(NativeFileSystemStore nativeStore) {
        this.nativeStore = nativeStore;
        this.isDefaultNativeStore = false;
        return this;
    }

    public CosNFileSystem withRangerCredentialsClient(RangerCredentialsClient rc) {
        this.rangerCredentialsClient = rc;
        return this;
    }

    public String getScheme() {
        return SCHEME;
    }

    public void initialize(URI uri, Configuration conf) throws IOException {
        long threadKeepAlive;
        super.initialize(uri, conf);
        this.setConf(conf);
        if (null == this.bucket) {
            this.bucket = uri.getHost();
        }
        if (null == this.nativeStore) {
            this.nativeStore = CosNUtils.createDefaultStore(conf);
            this.isDefaultNativeStore = true;
            this.nativeStore.initialize(uri, conf);
        }
        this.uri = URI.create(uri.getScheme() + "://" + uri.getAuthority());
        this.workingDir = new Path("/user", System.getProperty("user.name")).makeQualified(this.uri, this.getWorkingDirectory());
        this.owner = this.getOwnerId();
        this.group = this.getGroupId();
        LOG.debug("uri: {}, bucket: {}, working dir: {}, owner: {}, group: {}.\nconfiguration: {}.", new Object[]{uri, this.bucket, this.workingDir, this.owner, this.group, conf});
        BufferPool.getInstance().initialize(this.getConf());
        if (this.getConf().getBoolean("fs.cosn.posix_extension.enabled", false)) {
            LocalRandomAccessMappedBufferPool.getInstance().initialize(this.getConf());
        }
        int uploadThreadPoolSize = this.getConf().getInt("fs.cosn.upload_thread_pool", 10);
        int readAheadPoolSize = this.getConf().getInt("fs.cosn.read.ahead.queue.size", 8);
        Preconditions.checkArgument((uploadThreadPoolSize > 0 ? 1 : 0) != 0, (Object)String.format("The uploadThreadPoolSize[%d] should be positive.", uploadThreadPoolSize));
        Preconditions.checkArgument((readAheadPoolSize > 0 ? 1 : 0) != 0, (Object)String.format("The readAheadQueueSize[%d] should be positive.", readAheadPoolSize));
        int ioCoreTaskSize = uploadThreadPoolSize + readAheadPoolSize;
        int ioMaxTaskSize = Math.max(uploadThreadPoolSize + readAheadPoolSize, Runtime.getRuntime().availableProcessors() * 2 + 1);
        if (this.getConf().get("fs.cosn.io_thread_pool.maxSize") != null) {
            int ioThreadPoolMaxSize = this.getConf().getInt("fs.cosn.io_thread_pool.maxSize", CosNConfigKeys.DEFAULT_IO_THREAD_POOL_MAX_SIZE);
            Preconditions.checkArgument((ioThreadPoolMaxSize > 0 ? 1 : 0) != 0, (Object)String.format("The ioThreadPoolMaxSize[%d] should be positive.", ioThreadPoolMaxSize));
            ioCoreTaskSize = Math.min(ioCoreTaskSize, ioThreadPoolMaxSize);
            ioMaxTaskSize = ioThreadPoolMaxSize;
        }
        Preconditions.checkArgument(((threadKeepAlive = this.getConf().getLong("fs.cosn.threads.keep_alive_time", 60L)) > 0L ? 1 : 0) != 0, (Object)String.format("The threadKeepAlive [%d] should be positive.", threadKeepAlive));
        this.boundedIOThreadPool = new ThreadPoolExecutor(ioCoreTaskSize, ioMaxTaskSize, threadKeepAlive, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(ioCoreTaskSize), new ThreadFactoryBuilder().setNameFormat("cos-transfer-shared-%d").setDaemon(true).build(), new RejectedExecutionHandler(){

            @Override
            public void rejectedExecution(Runnable r, ThreadPoolExecutor executor) {
                if (!executor.isShutdown()) {
                    try {
                        executor.getQueue().put(r);
                    }
                    catch (InterruptedException e) {
                        LOG.error("put a io task into the download thread pool occurs an exception.", (Throwable)e);
                        throw new RejectedExecutionException("Putting the io task failed due to the interruption", e);
                    }
                } else {
                    LOG.error("The bounded io thread pool has been shutdown.");
                    throw new RejectedExecutionException("The bounded io thread pool has been shutdown");
                }
            }
        });
        int copyThreadPoolSize = this.getConf().getInt("fs.cosn.copy_thread_pool", 3);
        Preconditions.checkArgument((copyThreadPoolSize > 0 ? 1 : 0) != 0, (Object)String.format("The copyThreadPoolSize[%d] should be positive.", copyThreadPoolSize));
        this.boundedCopyThreadPool = new ThreadPoolExecutor(copyThreadPoolSize, copyThreadPoolSize, threadKeepAlive, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(copyThreadPoolSize), new ThreadFactoryBuilder().setNameFormat("cos-copy-%d").setDaemon(true).build(), new RejectedExecutionHandler(){

            @Override
            public void rejectedExecution(Runnable r, ThreadPoolExecutor executor) {
                if (!executor.isShutdown()) {
                    try {
                        executor.getQueue().put(r);
                    }
                    catch (InterruptedException e) {
                        LOG.error("put a copy task into the download thread pool occurs an exception.", (Throwable)e);
                    }
                }
            }
        });
        this.symbolicLinkSizeThreshold = this.getConf().getInt("fs.cosn.symbolic_link.sizeThreshold", 4096);
    }

    public URI getUri() {
        return this.uri;
    }

    public Path getHomeDirectory() {
        String homePrefix = this.getConf().get("dfs.user.home.dir.prefix");
        if (null != homePrefix) {
            return this.makeQualified(new Path(homePrefix + PATH_DELIMITER + this.getOwnerId()));
        }
        return super.getHomeDirectory();
    }

    public FSDataOutputStream append(Path f, int bufferSize, Progressable progress) throws IOException {
        if (this.isPosixBucket) {
            throw new UnsupportedOperationException("The posix bucket does not support append operation through S3 gateway");
        }
        if (this.getConf().getBoolean("fs.cosn.client-side-encryption.enabled", false)) {
            throw new UnsupportedOperationException("Not supported currently with client side encryption enabled");
        }
        try {
            if (this.getXAttr(f, CSE_ALOGORITHM_USER_METADATA) != null) {
                throw new UnsupportedOperationException("Not supported currently because the file is encrypted by client side");
            }
        }
        catch (IOException e) {
            throw new IOException("Failed to check whether the file is encrypted.", e);
        }
        FileStatus fileStatus = this.getFileStatus(f);
        if (fileStatus.isSymlink()) {
            f = this.getLinkTarget(f);
            fileStatus = this.getFileStatus(f);
        }
        if (fileStatus.isDirectory()) {
            throw new FileAlreadyExistsException(f + " is a directory.");
        }
        LOG.debug("Append the file path: {}.", (Object)f);
        Path absolutePath = this.makeAbsolute(f);
        String cosKey = CosNFileSystem.pathToKey(absolutePath);
        if (this.getConf().getBoolean("fs.cosn.posix_extension.enabled", false)) {
            Object seekableOutputStream;
            try {
                Class<?> seekableOutputStreamClass = Class.forName("org.apache.hadoop.fs.CosNSeekableFSDataOutputStream$SeekableOutputStream");
                Constructor<?> constructor = seekableOutputStreamClass.getConstructor(Configuration.class, NativeFileSystemStore.class, String.class, ExecutorService.class);
                seekableOutputStream = constructor.newInstance(this.getConf(), this.nativeStore, cosKey, this.boundedIOThreadPool);
            }
            catch (ClassNotFoundException e) {
                throw new IOException("org.apache.hadoop.fs.CosNSeekableFSDataOutputStream$SeekableOutputStream can not be found. please make sure that ofs-sdk-definition.jar is placed in the classpath.", e);
            }
            catch (InvocationTargetException e) {
                throw new IOException(e);
            }
            catch (NoSuchMethodException e) {
                throw new IOException("Failed to find the constructor of the org.apache.hadoop.fs.CosNSeekableFSDataOutputStream$SeekableOutputStream", e);
            }
            catch (InstantiationException e) {
                throw new IOException("Failed to create the object of the org.apache.hadoop.fs.CosNSeekableFSDataOutputStream$SeekableOutputStream", e);
            }
            catch (IllegalAccessException e) {
                throw new IOException("Failed to access the constructor of the org.apache.hadoop.fs.CosNSeekableFSDataOutputStream$SeekableOutputStream", e);
            }
            try {
                Class<?> seekableFSDataOutputStreamClass = Class.forName("org.apache.hadoop.fs.CosNSeekableFSDataOutputStream");
                Constructor<?> constructor = seekableFSDataOutputStreamClass.getConstructor(Class.forName("org.apache.hadoop.fs.CosNSeekableFSDataOutputStream$SeekableOutputStream"), FileSystem.Statistics.class);
                return (FSDataOutputStream)constructor.newInstance(seekableOutputStream, this.statistics);
            }
            catch (ClassNotFoundException e) {
                throw new IOException("org.apache.hadoop.fs.CosNSeekableFSDataOutputStream can not be found. please make sure that ofs-sdk-definition.jar is placed in the classpath.", e);
            }
            catch (NoSuchMethodException e) {
                throw new IOException("Failed to find the constructor of the org.apache.hadoop.fs.CosNSeekableFSDataOutputStream", e);
            }
            catch (InvocationTargetException e) {
                throw new IOException(e);
            }
            catch (InstantiationException e) {
                throw new IOException("Failed to create the object of the org.apache.hadoop.fs.CosNSeekableFSDataOutputStream", e);
            }
            catch (IllegalAccessException e) {
                throw new IOException("Failed to access the constructor of the org.apache.hadoop.fs.CosNSeekableFSDataOutputStream", e);
            }
        }
        return new FSDataOutputStream((OutputStream)new CosNExtendedFSDataOutputStream(this.getConf(), this.nativeStore, cosKey, this.boundedIOThreadPool, true), this.statistics, fileStatus.getLen());
    }

    public boolean truncate(Path f, long newLength) throws IOException {
        block33: {
            if (this.isPosixBucket) {
                throw new UnsupportedOperationException("The posix bucket does not support the truncate operation through S3 gateway.");
            }
            if (this.getConf().getBoolean("fs.cosn.client-side-encryption.enabled", false)) {
                throw new UnsupportedOperationException("Not supported currently with client side encryption enabled");
            }
            try {
                if (this.getXAttr(f, CSE_ALOGORITHM_USER_METADATA) != null) {
                    throw new UnsupportedOperationException("Not supported currently because the file is encrypted by client side");
                }
            }
            catch (IOException e) {
                throw new IOException("Failed to check whether the file is encrypted.", e);
            }
            FileStatus fileStatus = this.getFileStatus(f);
            if (fileStatus.isSymlink()) {
                f = this.getLinkTarget(f);
                fileStatus = this.getFileStatus(f);
            }
            if (fileStatus.isDirectory()) {
                throw new FileNotFoundException(f + " is a directory.");
            }
            if (newLength < 0L || newLength > fileStatus.getLen()) {
                throw new HadoopIllegalArgumentException(String.format("The new length [%d] of the truncate operation must be positive and less than the origin length.", newLength));
            }
            LOG.debug("Truncate the file path: {} to the new length: {}.", (Object)f, (Object)newLength);
            Path absolutePath = this.makeAbsolute(f);
            String cosKey = CosNFileSystem.pathToKey(absolutePath);
            try (FSDataOutputStream outputStream = new FSDataOutputStream((OutputStream)new CosNExtendedFSDataOutputStream(this.getConf(), this.nativeStore, cosKey, this.boundedIOThreadPool), this.statistics);){
                if (newLength <= 0L) break block33;
                try (InputStream inputStream = this.nativeStore.retrieveBlock(cosKey, 0L, newLength - 1L);){
                    byte[] chunk = new byte[4096];
                    int readBytes = inputStream.read(chunk);
                    while (readBytes != -1) {
                        outputStream.write(chunk, 0, readBytes);
                        readBytes = inputStream.read(chunk);
                    }
                }
            }
        }
        return true;
    }

    public FSDataOutputStream create(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        try {
            FileStatus targetFileStatus = this.getFileStatus(f);
            if (targetFileStatus.isSymlink()) {
                f = this.getLinkTarget(f);
                targetFileStatus = this.getFileStatus(f);
            }
            if (targetFileStatus.isFile() && !overwrite) {
                throw new FileAlreadyExistsException("File already exists: " + f);
            }
            if (targetFileStatus.isDirectory()) {
                throw new FileAlreadyExistsException("Directory already exists: " + f);
            }
        }
        catch (FileNotFoundException targetFileStatus) {
            // empty catch block
        }
        Path absolutePath = this.makeAbsolute(f);
        String key = CosNFileSystem.pathToKey(absolutePath);
        if (this.getConf().getBoolean("fs.cosn.flush.enabled", false)) {
            return new FSDataOutputStream((OutputStream)new CosNExtendedFSDataOutputStream(this.getConf(), this.nativeStore, key, this.boundedIOThreadPool), this.statistics);
        }
        return new FSDataOutputStream((OutputStream)new CosNFSDataOutputStream(this.getConf(), this.nativeStore, key, this.boundedIOThreadPool), this.statistics);
    }

    private boolean rejectRootDirectoryDelete(boolean isEmptyDir, boolean recursive) throws PathIOException {
        if (isEmptyDir) {
            return true;
        }
        if (recursive) {
            return false;
        }
        throw new PathIOException(this.bucket, "Can not delete root path");
    }

    public boolean delete(Path f, boolean recursive) throws IOException {
        FileStatus status;
        try {
            status = this.getFileStatus(f);
        }
        catch (FileNotFoundException e) {
            LOG.debug("Delete called for '{}', but the file does not exist and returning the false.", (Object)f);
            return false;
        }
        Path absolutePath = this.makeAbsolute(f);
        String key = CosNFileSystem.pathToKey(absolutePath);
        if (key.compareToIgnoreCase(PATH_DELIMITER) == 0) {
            FileStatus[] fileStatuses = this.listStatus(f);
            return this.rejectRootDirectoryDelete(fileStatuses.length == 0, recursive);
        }
        if (status.isDirectory()) {
            if (!key.endsWith(PATH_DELIMITER)) {
                key = key + PATH_DELIMITER;
            }
            if (!recursive && this.listStatus(f).length > 0) {
                throw new IOException("Can not delete " + f + " as is a not empty directory and recurse option is false");
            }
            if (!this.isPosixBucket) {
                this.internalRecursiveDelete(key, 999);
            } else {
                this.internalAutoRecursiveDelete(key);
            }
        } else {
            LOG.debug("Delete the cos key [{}].", (Object)key);
            this.nativeStore.delete(key);
        }
        if (!this.isPosixBucket) {
            this.createParentDirectoryIfNecessary(f);
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void internalRecursiveDelete(String key, int listMaxLength) throws IOException {
        CosNPartialListing listing;
        CosNDeleteFileContext deleteFileContext = new CosNDeleteFileContext();
        int deleteToFinishes = 0;
        String priorLastKey = null;
        block7: do {
            listing = this.nativeStore.list(key, listMaxLength, priorLastKey, true);
            for (FileMetadata file : listing.getFiles()) {
                this.checkPermission(new Path(file.getKey()), RangerAccessType.DELETE);
                this.boundedCopyThreadPool.execute(new CosNDeleteFileTask(this.nativeStore, file.getKey(), deleteFileContext));
                ++deleteToFinishes;
                if (!deleteFileContext.isDeleteSuccess()) break;
            }
            for (FileMetadata commonPrefix : listing.getCommonPrefixes()) {
                this.checkPermission(new Path(commonPrefix.getKey()), RangerAccessType.DELETE);
                this.boundedCopyThreadPool.execute(new CosNDeleteFileTask(this.nativeStore, commonPrefix.getKey(), deleteFileContext));
                ++deleteToFinishes;
                if (!deleteFileContext.isDeleteSuccess()) continue block7;
            }
        } while ((priorLastKey = listing.getPriorLastKey()) != null);
        deleteFileContext.lock();
        try {
            deleteFileContext.awaitAllFinish(deleteToFinishes);
        }
        catch (InterruptedException e) {
            LOG.warn("interrupted when wait delete to finish");
        }
        finally {
            deleteFileContext.unlock();
        }
        if (!deleteFileContext.isDeleteSuccess() && deleteFileContext.hasException()) {
            throw deleteFileContext.getIOException();
        }
        try {
            LOG.debug("Delete the cos key [{}].", (Object)key);
            this.nativeStore.delete(key);
        }
        catch (Exception e) {
            LOG.error("Delete the key failed.");
        }
    }

    private void internalAutoRecursiveDelete(String key) throws IOException {
        LOG.debug("Delete the cos key auto recursive [{}].", (Object)key);
        this.nativeStore.deleteRecursive(key);
    }

    public FileStatus getFileStatus(Path f) throws IOException {
        CosNSymlinkMetadata cosNSymlinkMetadata;
        Path absolutePath = this.makeAbsolute(f);
        String key = CosNFileSystem.pathToKey(absolutePath);
        if (key.length() == 0 || key.equals(PATH_DELIMITER)) {
            return this.newDirectory(absolutePath);
        }
        CosNResultInfo getObjectMetadataResultInfo = new CosNResultInfo();
        FileMetadata meta = this.nativeStore.retrieveMetadata(key, getObjectMetadataResultInfo);
        if (meta != null) {
            if (meta.isFile()) {
                LOG.debug("Retrieve the cos key [{}] to find that it is a file.", (Object)key);
                return this.newFile(meta, absolutePath);
            }
            LOG.debug("Retrieve the cos key [{}] to find that it is a directory.", (Object)key);
            return this.newDirectory(meta, absolutePath);
        }
        CosNResultInfo getSymlinkResultInfo = new CosNResultInfo();
        if (this.supportsSymlinks() && null != (cosNSymlinkMetadata = this.nativeStore.retrieveSymlinkMetadata(key, getSymlinkResultInfo))) {
            return this.newSymlink(cosNSymlinkMetadata, absolutePath);
        }
        if (this.isPosixBucket) {
            throw new FileNotFoundException("No such file or directory in posix bucket'" + absolutePath + "'");
        }
        if (this.getConf().getBoolean("fs.cosn.filestatus.list.op.enabled", true)) {
            if (!key.endsWith(PATH_DELIMITER)) {
                key = key + PATH_DELIMITER;
            }
            int maxKeys = this.getConf().getInt("fs.cosn.filestatus.list_max_keys", 2);
            LOG.debug("List the cos key [{}] to judge whether it is a directory or not. max keys [{}]", (Object)key, (Object)maxKeys);
            CosNResultInfo listObjectsResultInfo = new CosNResultInfo();
            CosNPartialListing listing = this.nativeStore.list(key, maxKeys, listObjectsResultInfo);
            if (listing.getFiles().length > 0 || listing.getCommonPrefixes().length > 0) {
                LOG.debug("List the cos key [{}] to find that it is a directory.", (Object)key);
                return this.newDirectory(absolutePath);
            }
            if (listObjectsResultInfo.isKeySameToPrefix()) {
                LOG.info("List the cos key [{}] same to prefix, getSymlink-id:[{}] head-id:[{}], list-id:[{}], list-type:[{}], thread-id:[{}], thread-name:[{}]", new Object[]{key, getSymlinkResultInfo.getRequestID(), getObjectMetadataResultInfo.getRequestID(), listObjectsResultInfo.getRequestID(), listObjectsResultInfo.isKeySameToPrefix(), Thread.currentThread().getId(), Thread.currentThread().getName()});
            }
            LOG.debug("Can not find the cos key [{}] on COS.", (Object)key);
        }
        throw new FileNotFoundException("No such file or directory '" + absolutePath + "'");
    }

    public FileStatus[] listStatus(Path f) throws IOException {
        CosNPartialListing listing;
        FileStatus fileStatus;
        Path absolutePath = this.makeAbsolute(f);
        String key = CosNFileSystem.pathToKey(absolutePath);
        int listMaxLength = 999;
        if (this.isPosixBucket) {
            listMaxLength = 5000;
        }
        if (key.length() > 0 && ((fileStatus = this.getFileStatus(f)).isFile() || fileStatus.isSymlink())) {
            return new FileStatus[]{fileStatus};
        }
        if (!key.endsWith(PATH_DELIMITER)) {
            key = key + PATH_DELIMITER;
        }
        if (this.isPosixBucket) {
            try {
                this.getFileStatus(f);
            }
            catch (FileNotFoundException e) {
                throw new FileNotFoundException("No such file or directory:" + f);
            }
        }
        URI pathUri = absolutePath.toUri();
        TreeSet<FileStatus> status = new TreeSet<FileStatus>();
        String priorLastKey = null;
        do {
            Path subPath;
            listing = this.nativeStore.list(key, listMaxLength, priorLastKey, false);
            for (FileMetadata fileMetadata : listing.getFiles()) {
                CosNSymlinkMetadata cosNSymlinkMetadata;
                subPath = CosNFileSystem.keyToPath(fileMetadata.getKey());
                if (fileMetadata.getKey().equals(key)) {
                    LOG.debug("This is just the directory we have been asked to list. cos key: {}.", (Object)fileMetadata.getKey());
                    continue;
                }
                if (this.supportsSymlinks() && fileMetadata.getLength() < (long)this.symbolicLinkSizeThreshold && null != (cosNSymlinkMetadata = this.nativeStore.retrieveSymlinkMetadata(fileMetadata.getKey()))) {
                    status.add(this.newSymlink(cosNSymlinkMetadata, subPath));
                    continue;
                }
                status.add(this.newFile(fileMetadata, subPath));
            }
            for (FileMetadata commonPrefix : listing.getCommonPrefixes()) {
                subPath = CosNFileSystem.keyToPath(commonPrefix.getKey());
                String relativePath = pathUri.relativize(subPath.toUri()).getPath();
                status.add(this.newDirectory(commonPrefix, new Path(absolutePath, relativePath)));
            }
        } while ((priorLastKey = listing.getPriorLastKey()) != null);
        return status.toArray(new FileStatus[status.size()]);
    }

    private FileStatus newFile(FileMetadata meta, Path path) {
        return new CosNFileStatus(meta.getLength(), false, 1, this.getDefaultBlockSize(), meta.getLastModified(), 0L, null, this.owner, this.group, path.makeQualified(this.getUri(), this.getWorkingDirectory()), meta.getETag(), meta.getCrc64ecm(), meta.getVersionId());
    }

    private FileStatus newDirectory(Path path) {
        return new CosNFileStatus(0L, true, 1, 0L, 0L, 0L, null, this.owner, this.group, path.makeQualified(this.getUri(), this.getWorkingDirectory()));
    }

    private FileStatus newSymlink(CosNSymlinkMetadata metadata, Path path) {
        FileStatus symlinkStatus = this.newFile(metadata, path);
        Path targetPath = CosNFileSystem.keyToPath(metadata.getTarget());
        symlinkStatus.setSymlink(this.makeAbsolute(targetPath).makeQualified(this.getUri(), this.getWorkingDirectory()));
        return symlinkStatus;
    }

    private FileStatus newDirectory(FileMetadata meta, Path path) {
        if (meta == null) {
            return this.newDirectory(path);
        }
        return new CosNFileStatus(0L, true, 1, 0L, meta.getLastModified(), 0L, null, this.owner, this.group, path.makeQualified(this.getUri(), this.getWorkingDirectory()), meta.getETag(), meta.getCrc64ecm(), meta.getVersionId());
    }

    private void validatePath(Path path) throws IOException {
        Path parent = path.getParent();
        while (true) {
            try {
                FileStatus fileStatus = this.getFileStatus(parent);
                if (!fileStatus.isDirectory()) {
                    throw new FileAlreadyExistsException(String.format("Can't make directory for path '%s', it is a file.", parent));
                }
            }
            catch (FileNotFoundException fileNotFoundException) {
                if ((parent = parent.getParent()) != null) continue;
            }
            break;
        }
    }

    public boolean mkdirs(Path f, FsPermission permission) throws IOException {
        try {
            FileStatus fileStatus = this.getFileStatus(f);
            if (fileStatus.isDirectory()) {
                return true;
            }
            throw new FileAlreadyExistsException("Path is a file: " + f);
        }
        catch (FileNotFoundException e) {
            this.validatePath(f);
            boolean result = this.isPosixBucket ? this.mkDirAutoRecursively(f, permission) : this.mkDirRecursively(f, permission);
            return result;
        }
    }

    public boolean mkDirRecursively(Path f, FsPermission permission) throws IOException {
        Path path;
        LOG.debug("Make the directory recursively. Path: {}, FsPermission: {}.", (Object)f, (Object)permission);
        Path absolutePath = this.makeAbsolute(f);
        ArrayList<Path> paths = new ArrayList<Path>();
        do {
            paths.add(absolutePath);
        } while ((absolutePath = absolutePath.getParent()) != null);
        Iterator iterator = paths.iterator();
        while (iterator.hasNext() && !(path = (Path)iterator.next()).isRoot()) {
            try {
                FileStatus fileStatus = this.getFileStatus(path);
                if (fileStatus.isFile()) {
                    throw new FileAlreadyExistsException(String.format("Can't make directory for path '%s' since it is a file.", f));
                }
                if (!fileStatus.isDirectory()) continue;
                if (fileStatus.getModificationTime() <= 0L) {
                    throw new FileNotFoundException("Dir '" + path + "' doesn't exist in COS");
                }
                break;
            }
            catch (FileNotFoundException e) {
                LOG.debug("Make the directory [{}] on COS.", (Object)path);
                String folderPath = CosNFileSystem.pathToKey(this.makeAbsolute(path));
                if (!folderPath.endsWith(PATH_DELIMITER)) {
                    folderPath = folderPath + PATH_DELIMITER;
                }
                this.nativeStore.storeEmptyFile(folderPath);
            }
        }
        return true;
    }

    public boolean mkDirAutoRecursively(Path f, FsPermission permission) throws IOException {
        LOG.debug("Make the directory recursively auto. Path: {}, FsPermission: {}.", (Object)f, (Object)permission);
        String folderPath = CosNFileSystem.pathToKey(this.makeAbsolute(f));
        if (!folderPath.endsWith(PATH_DELIMITER)) {
            folderPath = folderPath + PATH_DELIMITER;
        }
        this.nativeStore.storeEmptyFile(folderPath);
        return true;
    }

    public FSDataInputStream open(Path f, int bufferSize) throws IOException {
        FileStatus fileStatus = this.getFileStatus(f);
        if (fileStatus.isSymlink()) {
            f = this.getLinkTarget(f);
            fileStatus = this.getFileStatus(f);
        }
        if (fileStatus.isDirectory()) {
            throw new FileNotFoundException("'" + f + "' is a directory");
        }
        LOG.info("Opening '" + f + "' for reading");
        Path absolutePath = this.makeAbsolute(f);
        String key = CosNFileSystem.pathToKey(absolutePath);
        return new FSDataInputStream((InputStream)new BufferedFSInputStream((FSInputStream)new CosNFSInputStream(this.getConf(), this.nativeStore, this.statistics, key, fileStatus.getLen(), this.boundedIOThreadPool), bufferSize));
    }

    public boolean rename(Path src, Path dst) throws IOException {
        FileStatus srcFileStatus;
        block13: {
            Path dstParentPath;
            if (src.isRoot()) {
                LOG.debug("Cannot rename the root directory of a filesystem.");
                return false;
            }
            try {
                srcFileStatus = this.getFileStatus(src);
            }
            catch (FileNotFoundException e) {
                LOG.debug("The source path [{}] is not exist.", (Object)src);
                return false;
            }
            if (src.equals((Object)dst)) {
                LOG.debug("The source path and the dest path refer to the same file or directory: {}", (Object)dst);
                throw new IOException("the source path and dest path refer to the same file or directory");
            }
            for (dstParentPath = dst.getParent(); null != dstParentPath && !src.equals((Object)dstParentPath); dstParentPath = dstParentPath.getParent()) {
            }
            if (null != dstParentPath) {
                LOG.debug("It is not allowed to rename a parent directory:{} to its subdirectory:{}.", (Object)src, (Object)dst);
                throw new IOException(String.format("It is not allowed to rename a parent directory:%s to its subdirectory:%s", src, dst));
            }
            FileStatus dstFileStatus = null;
            try {
                FileStatus[] statuses;
                dstFileStatus = this.getFileStatus(dst);
                if (dstFileStatus.isFile()) {
                    LOG.debug("File: {} already exists.", (Object)dstFileStatus.getPath());
                    return false;
                }
                dst = new Path(dst, src.getName());
                try {
                    statuses = this.listStatus(dst);
                }
                catch (FileNotFoundException e) {
                    statuses = null;
                }
                if (null != statuses && statuses.length > 0) {
                    LOG.debug("Cannot rename {} to {}, file already exists.", (Object)src, (Object)dst);
                    return false;
                }
            }
            catch (FileNotFoundException e) {
                Path tempDstParentPath = dst.getParent();
                FileStatus dstParentStatus = this.getFileStatus(tempDstParentPath);
                if (dstParentStatus.isDirectory()) break block13;
                throw new IOException(String.format("Cannot rename %s to %s, %s is a file", src, dst, dst.getParent()));
            }
        }
        if (!this.isPosixBucket) {
            return this.internalCopyAndDelete(src, dst, srcFileStatus.isDirectory(), srcFileStatus.isSymlink());
        }
        return this.internalRename(src, dst);
    }

    private boolean internalCopyAndDelete(Path srcPath, Path dstPath, boolean isDir, boolean isSymlink) throws IOException {
        boolean result = false;
        result = isDir ? this.copyDirectory(srcPath, dstPath) : (isSymlink ? this.copySymlink(srcPath, dstPath) : this.copyFile(srcPath, dstPath));
        return result ? this.delete(srcPath, true) : false;
    }

    private boolean internalRename(Path srcPath, Path dstPath) throws IOException {
        String srcKey = CosNFileSystem.pathToKey(srcPath);
        String dstKey = CosNFileSystem.pathToKey(dstPath);
        this.nativeStore.rename(srcKey, dstKey);
        return true;
    }

    private boolean copyFile(Path srcPath, Path dstPath) throws IOException {
        String srcKey = CosNFileSystem.pathToKey(srcPath);
        String dstKey = CosNFileSystem.pathToKey(dstPath);
        this.nativeStore.copy(srcKey, dstKey);
        return true;
    }

    private boolean copySymlink(Path srcSymlink, Path dstSymlink) throws IOException {
        Path targetFilePath = this.getLinkTarget(srcSymlink);
        this.createSymlink(targetFilePath, dstSymlink, false);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean copyDirectory(Path srcPath, Path dstPath) throws IOException {
        CosNPartialListing objectList;
        String dstKey;
        String srcKey = CosNFileSystem.pathToKey(srcPath);
        if (!srcKey.endsWith(PATH_DELIMITER)) {
            srcKey = srcKey + PATH_DELIMITER;
        }
        if (!(dstKey = CosNFileSystem.pathToKey(dstPath)).endsWith(PATH_DELIMITER)) {
            dstKey = dstKey + PATH_DELIMITER;
        }
        if (dstKey.startsWith(srcKey)) {
            throw new IOException("can not copy a directory to a subdirectory of self");
        }
        if (this.nativeStore.retrieveMetadata(srcKey) == null) {
            this.nativeStore.storeEmptyFile(srcKey);
        } else {
            this.nativeStore.copy(srcKey, dstKey);
        }
        CosNCopyFileContext copyFileContext = new CosNCopyFileContext();
        int copiesToFinishes = 0;
        String priorLastKey = null;
        block5: do {
            objectList = this.nativeStore.list(srcKey, 999, priorLastKey, true);
            for (FileMetadata file : objectList.getFiles()) {
                this.checkPermission(new Path(file.getKey()), RangerAccessType.DELETE);
                this.boundedCopyThreadPool.execute(new CosNCopyFileTask(this.nativeStore, file.getKey(), dstKey.concat(file.getKey().substring(srcKey.length())), copyFileContext));
                ++copiesToFinishes;
                if (!copyFileContext.isCopySuccess()) continue block5;
            }
        } while (null != (priorLastKey = objectList.getPriorLastKey()));
        copyFileContext.lock();
        try {
            copyFileContext.awaitAllFinish(copiesToFinishes);
        }
        catch (InterruptedException e) {
            LOG.warn("interrupted when wait copies to finish");
        }
        finally {
            copyFileContext.unlock();
        }
        return copyFileContext.isCopySuccess();
    }

    private void createParentDirectoryIfNecessary(Path path) throws IOException {
        String parentKey;
        Path parent = path.getParent();
        if (!(null == parent || parent.isRoot() || StringUtils.isNullOrEmpty((String)(parentKey = CosNFileSystem.pathToKey(parent))) || this.exists(parent))) {
            LOG.debug("Create a parent directory [{}] for the path [{}].", (Object)parent, (Object)path);
            if (!parentKey.endsWith(PATH_DELIMITER)) {
                parentKey = parentKey + PATH_DELIMITER;
            }
            this.nativeStore.storeEmptyFile(parentKey);
        }
    }

    public long getDefaultBlockSize() {
        return this.getConf().getLong("fs.cosn.block.size", 0x8000000L);
    }

    public void setWorkingDirectory(Path newDir) {
        this.workingDir = newDir;
    }

    public Path getWorkingDirectory() {
        return this.workingDir;
    }

    public FileChecksum getFileChecksum(Path f, long length) throws IOException {
        Preconditions.checkArgument((length >= 0L ? 1 : 0) != 0);
        if (this.getFileStatus(f).isSymlink()) {
            f = this.getLinkTarget(f);
        }
        if (this.getConf().getBoolean("fs.cosn.crc64.checksum.enabled", false)) {
            Path absolutePath = this.makeAbsolute(f);
            String key = CosNFileSystem.pathToKey(absolutePath);
            FileMetadata fileMetadata = this.nativeStore.retrieveMetadata(key);
            if (null == fileMetadata) {
                throw new FileNotFoundException("File or directory doesn't exist: " + f);
            }
            String crc64ecm = fileMetadata.getCrc64ecm();
            return crc64ecm != null ? new CRC64Checksum(crc64ecm) : super.getFileChecksum(f, length);
        }
        if (this.getConf().getBoolean("fs.cosn.crc32c.checksum.enabled", false)) {
            Path absolutePath = this.makeAbsolute(f);
            String key = CosNFileSystem.pathToKey(absolutePath);
            FileMetadata fileMetadata = this.nativeStore.retrieveMetadata(key);
            if (null == fileMetadata) {
                throw new FileNotFoundException("File or directory doesn't exist: " + f);
            }
            String crc32cm = fileMetadata.getCrc32cm();
            return crc32cm != null ? new CRC32CCheckSum(crc32cm) : super.getFileChecksum(f, length);
        }
        return super.getFileChecksum(f, length);
    }

    public void setXAttr(Path f, String name, byte[] value, EnumSet<XAttrSetFlag> flag) throws IOException {
        Path absolutePath;
        String key;
        FileMetadata fileMetadata;
        if (name.getBytes(METADATA_ENCODING).length + value.length > 1024) {
            throw new HadoopIllegalArgumentException(String.format("The maximum combined size of the name and value of an extended attribute in bytes should be less than or equal to %d", 1024));
        }
        if (this.getFileStatus(f).isSymlink()) {
            f = this.getLinkTarget(f);
        }
        if (null == (fileMetadata = this.nativeStore.retrieveMetadata(key = CosNFileSystem.pathToKey(absolutePath = this.makeAbsolute(f))))) {
            throw new FileNotFoundException("File or directory doesn't exist: " + f);
        }
        boolean xAttrExists = null != fileMetadata.getUserAttributes() && fileMetadata.getUserAttributes().containsKey(name);
        XAttrSetFlag.validate((String)name, (boolean)xAttrExists, flag);
        if (fileMetadata.isFile()) {
            this.nativeStore.storeFileAttribute(key, name, value);
        } else {
            this.nativeStore.storeDirAttribute(key, name, value);
        }
    }

    public byte[] getXAttr(Path f, String name) throws IOException {
        Path absolutePath;
        String key;
        FileMetadata fileMetadata;
        if (this.getFileStatus(f).isSymlink()) {
            f = this.getLinkTarget(f);
        }
        if (null == (fileMetadata = this.nativeStore.retrieveMetadata(key = CosNFileSystem.pathToKey(absolutePath = this.makeAbsolute(f))))) {
            throw new FileNotFoundException("File or directory doesn't exist: " + f);
        }
        if (null != fileMetadata.getUserAttributes()) {
            return fileMetadata.getUserAttributes().get(name);
        }
        return null;
    }

    public Map<String, byte[]> getXAttrs(Path f, List<String> names) throws IOException {
        Path absolutePath;
        String key;
        FileMetadata fileMetadata;
        if (this.getFileStatus(f).isSymlink()) {
            f = this.getLinkTarget(f);
        }
        if (null == (fileMetadata = this.nativeStore.retrieveMetadata(key = CosNFileSystem.pathToKey(absolutePath = this.makeAbsolute(f))))) {
            throw new FileNotFoundException("File or directory doesn't exist: " + f);
        }
        HashMap<String, byte[]> attrs = null;
        if (null != fileMetadata.getUserAttributes()) {
            attrs = new HashMap<String, byte[]>();
            for (String name : names) {
                if (!fileMetadata.getUserAttributes().containsKey(name)) continue;
                attrs.put(name, fileMetadata.getUserAttributes().get(name));
            }
        }
        return attrs;
    }

    public Map<String, byte[]> getXAttrs(Path f) throws IOException {
        Path absolutePath;
        String key;
        FileMetadata fileMetadata;
        if (this.getFileStatus(f).isSymlink()) {
            f = this.getLinkTarget(f);
        }
        if (null == (fileMetadata = this.nativeStore.retrieveMetadata(key = CosNFileSystem.pathToKey(absolutePath = this.makeAbsolute(f))))) {
            throw new FileNotFoundException("File or directory doesn't exist: " + f);
        }
        return fileMetadata.getUserAttributes();
    }

    public void removeXAttr(Path f, String name) throws IOException {
        boolean xAttrExists;
        Path absolutPath;
        String key;
        FileMetadata fileMetadata;
        if (this.getFileStatus(f).isSymlink()) {
            f = this.getLinkTarget(f);
        }
        if (null == (fileMetadata = this.nativeStore.retrieveMetadata(key = CosNFileSystem.pathToKey(absolutPath = this.makeAbsolute(f))))) {
            throw new FileNotFoundException("File or directory doesn't exist: " + f);
        }
        boolean bl = xAttrExists = null != fileMetadata.getUserAttributes() && fileMetadata.getUserAttributes().containsKey(name);
        if (xAttrExists) {
            if (fileMetadata.isFile()) {
                this.nativeStore.removeFileAttribute(key, name);
            } else {
                this.nativeStore.removeDirAttribute(key, name);
            }
        }
    }

    public List<String> listXAttrs(Path f) throws IOException {
        Path absolutePath;
        String key;
        FileMetadata fileMetadata;
        if (this.getFileStatus(f).isSymlink()) {
            f = this.getLinkTarget(f);
        }
        if (null == (fileMetadata = this.nativeStore.retrieveMetadata(key = CosNFileSystem.pathToKey(absolutePath = this.makeAbsolute(f))))) {
            throw new FileNotFoundException("File or directory doesn't exist: " + f);
        }
        return new ArrayList<String>(fileMetadata.getUserAttributes().keySet());
    }

    public void createSymlink(Path target, Path link, boolean createParent) throws AccessControlException, FileAlreadyExistsException, FileNotFoundException, ParentNotDirectoryException, UnsupportedFileSystemException, IOException {
        if (!this.supportsSymlinks()) {
            super.createSymlink(target, link, createParent);
        }
        try {
            FileStatus parentStatus = this.getFileStatus(link.getParent());
            if (!parentStatus.isDirectory()) {
                throw new ParentNotDirectoryException(String.format("The parent path of the symlink [%s] is not a directory.", link));
            }
        }
        catch (FileNotFoundException parentDirNotFoundException) {
            if (createParent) {
                LOG.debug("The parent directory of the symlink [{}] does not exist, creating it.", (Object)link.getParent());
                if (!this.mkdirs(link.getParent())) {
                    throw new IOException(String.format("Failed to create the parent directory of the symlink [%s].", link));
                }
            }
            throw parentDirNotFoundException;
        }
        try {
            FileStatus fileStatus = this.getFileStatus(link);
            if (null != fileStatus && (fileStatus.isFile() || fileStatus.isSymlink())) {
                throw new FileAlreadyExistsException("File already exists: " + link);
            }
            if (null != fileStatus && fileStatus.isDirectory()) {
                throw new FileAlreadyExistsException("Directory already exists: " + link);
            }
        }
        catch (FileNotFoundException fileStatus) {
            // empty catch block
        }
        Path targetAbsolutePath = this.makeAbsolute(target);
        Path linkAbsolutePath = this.makeAbsolute(link);
        String targetKey = CosNFileSystem.pathToKey(targetAbsolutePath);
        String linkKey = CosNFileSystem.pathToKey(linkAbsolutePath);
        this.nativeStore.createSymlink(linkKey, targetKey);
    }

    public FileStatus getFileLinkStatus(Path f) throws AccessControlException, FileNotFoundException, UnsupportedFileSystemException, IOException {
        if (!this.supportsSymlinks()) {
            return super.getFileLinkStatus(f);
        }
        Path absolutePath = this.makeAbsolute(f);
        String symlinkKey = CosNFileSystem.pathToKey(absolutePath);
        CosNSymlinkMetadata cosNSymlinkMetadata = this.nativeStore.retrieveSymlinkMetadata(symlinkKey);
        if (null == cosNSymlinkMetadata) {
            throw new FileNotFoundException("Symbolic does not exist: " + f);
        }
        FileStatus fileStatus = this.newSymlink(cosNSymlinkMetadata, absolutePath);
        if (fileStatus.isSymlink()) {
            Path targetQual = FSLinkResolver.qualifySymlinkTarget((URI)this.uri, (Path)fileStatus.getPath(), (Path)fileStatus.getSymlink());
            fileStatus.setSymlink(targetQual);
        }
        return fileStatus;
    }

    public boolean supportsSymlinks() {
        return this.getConf().getBoolean("fs.cosn.support_symlink.enabled", false);
    }

    public Path getLinkTarget(Path f) throws IOException {
        if (!this.supportsSymlinks()) {
            return super.getLinkTarget(f);
        }
        Path absolutePath = this.makeAbsolute(f);
        return (Path)new FileSystemLinkResolver<Path>(){

            public Path doCall(Path path) throws IOException, UnresolvedLinkException {
                Path targetPath = CosNFileSystem.this.resolveLink(path);
                try {
                    FileStatus targetFileStatus = CosNFileSystem.this.getFileStatus(targetPath);
                    if (targetFileStatus.isSymlink()) {
                        throw new UnresolvedLinkException();
                    }
                }
                catch (FileNotFoundException fileNotFoundException) {
                    // empty catch block
                }
                return targetPath;
            }

            public Path next(FileSystem fileSystem, Path path) throws IOException {
                return fileSystem.getLinkTarget(path);
            }
        }.resolve((FileSystem)this, absolutePath);
    }

    protected Path resolveLink(Path f) throws IOException {
        if (!this.supportsSymlinks()) {
            return super.resolveLink(f);
        }
        Path absolutePath = this.makeAbsolute(f);
        String symlinkKey = CosNFileSystem.pathToKey(absolutePath);
        String targetKey = this.nativeStore.getSymlink(symlinkKey);
        if (null == targetKey) {
            throw new FileNotFoundException("Symbolic does not exist: " + f);
        }
        Path targetPath = CosNFileSystem.keyToPath(targetKey);
        return this.makeAbsolute(targetPath).makeQualified(this.getUri(), this.getWorkingDirectory());
    }

    public void close() throws IOException {
        try {
            super.close();
        }
        finally {
            this.boundedIOThreadPool.shutdown();
            this.boundedCopyThreadPool.shutdown();
            BufferPool.getInstance().close();
            if (null != this.nativeStore && this.isDefaultNativeStore) {
                this.nativeStore.close();
            }
        }
    }

    public NativeFileSystemStore getStore() {
        return this.nativeStore;
    }

    private String getOwnerId() {
        UserGroupInformation currentUgi;
        try {
            currentUgi = UserGroupInformation.getCurrentUser();
        }
        catch (IOException e) {
            LOG.warn("get current user failed! use user.name prop", (Throwable)e);
            return System.getProperty("user.name");
        }
        String shortUserName = "";
        if (currentUgi != null) {
            shortUserName = currentUgi.getShortUserName();
        }
        if (shortUserName == null) {
            LOG.warn("get short user name failed! use user.name prop");
            shortUserName = System.getProperty("user.name");
        }
        return shortUserName;
    }

    private String getGroupId() {
        UserGroupInformation currentUgi = null;
        try {
            currentUgi = UserGroupInformation.getCurrentUser();
        }
        catch (IOException e) {
            LOG.warn("get current user failed! use user.name prop", (Throwable)e);
            return System.getProperty("user.name");
        }
        if (currentUgi != null) {
            String[] groupNames = currentUgi.getGroupNames();
            if (groupNames.length != 0) {
                return groupNames[0];
            }
            return this.getOwnerId();
        }
        return System.getProperty("user.name");
    }

    private String getOwnerInfo(boolean getOwnerId) {
        String ownerInfoId = "";
        try {
            int c;
            String userName = System.getProperty("user.name");
            String command = "id -u " + userName;
            if (!getOwnerId) {
                command = "id -g " + userName;
            }
            Process child = Runtime.getRuntime().exec(command);
            child.waitFor();
            InputStream in = child.getInputStream();
            StringBuilder strBuffer = new StringBuilder();
            while ((c = in.read()) != -1) {
                strBuffer.append((char)c);
            }
            in.close();
            ownerInfoId = strBuffer.toString();
        }
        catch (IOException | InterruptedException e) {
            LOG.error("getOwnerInfo occur a exception", (Throwable)e);
        }
        return ownerInfoId;
    }

    private void checkPermission(Path f, RangerAccessType rangerAccessType) throws IOException {
        this.rangerCredentialsClient.doCheckPermission(f, rangerAccessType, this.getOwnerId(), this.getWorkingDirectory());
    }

    private Path makeAbsolute(Path path) {
        if (path.isAbsolute()) {
            return path;
        }
        return new Path(this.workingDir, path);
    }

    public static String pathToKey(Path path) {
        if (path.toUri().getScheme() != null && path.toUri().getPath().isEmpty()) {
            return "";
        }
        if (!path.isAbsolute()) {
            throw new IllegalArgumentException("Path must be absolute: " + path);
        }
        String ret = path.toUri().getPath();
        if (ret.endsWith(PATH_DELIMITER) && ret.indexOf(PATH_DELIMITER) != ret.length() - 1) {
            ret = ret.substring(0, ret.length() - 1);
        }
        return ret;
    }

    public static Path keyToPath(String key) {
        if (!key.startsWith(PATH_DELIMITER)) {
            return new Path(PATH_DELIMITER + key);
        }
        return new Path(key);
    }
}

