/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.cosn.multipart.upload;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.qcloud.cos.model.PartETag;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import org.apache.commons.codec.binary.Hex;
import org.apache.hadoop.fs.FileMetadata;
import org.apache.hadoop.fs.NativeFileSystemStore;
import org.apache.hadoop.fs.cosn.BufferInputStream;
import org.apache.hadoop.fs.cosn.LocalRandomAccessMappedBufferPool;
import org.apache.hadoop.fs.cosn.MD5Utils;
import org.apache.hadoop.fs.cosn.buffer.CosNRandomAccessMappedBuffer;
import org.apache.hadoop.fs.cosn.multipart.upload.UploadPartCopy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MultipartManager {
    private static final Logger LOG = LoggerFactory.getLogger(MultipartManager.class);
    private final long MAX_FILE_SIZE;
    private final long partSize;
    private final NativeFileSystemStore nativeStore;
    private final String cosKey;
    private volatile String uploadId;
    private final SortedMap<Integer, PartETag> partETags = Collections.synchronizedSortedMap(new TreeMap());
    private final List<LocalPart> localParts = Collections.synchronizedList(new ArrayList());
    private final ListeningExecutorService listeningExecutorService;
    private volatile boolean splitPartProcess;
    private volatile boolean committed;
    private volatile boolean aborted;
    private volatile boolean closed;

    public MultipartManager(NativeFileSystemStore nativeStore, String cosKey, long partSize, ExecutorService executorService) {
        this.partSize = partSize;
        this.MAX_FILE_SIZE = this.partSize * 10000L;
        this.nativeStore = nativeStore;
        this.cosKey = cosKey;
        this.uploadId = null;
        this.committed = true;
        this.aborted = false;
        this.closed = false;
        this.listeningExecutorService = MoreExecutors.listeningDecorator((ExecutorService)executorService);
    }

    public void resumeForWrite() throws IOException {
        this.checkOpened();
        FileMetadata fileMetadata = this.nativeStore.retrieveMetadata(this.cosKey);
        if (null == fileMetadata) {
            throw new FileNotFoundException(String.format("The cosKey [%s] is not exists.", this.cosKey));
        }
        this.splitParts(fileMetadata.getLength());
    }

    public void splitParts(long newLen) throws IOException {
        long deltaPadding;
        long copyRemaining;
        this.checkOpened();
        Preconditions.checkArgument((newLen >= 0L && newLen <= this.MAX_FILE_SIZE ? 1 : 0) != 0, (Object)String.format("The newLen should be in range [%d, %d].", 0, this.MAX_FILE_SIZE));
        FileMetadata fileMetadata = this.nativeStore.retrieveMetadata(this.cosKey);
        if (null == fileMetadata) {
            throw new IOException(String.format("The cos key [%s] is not found.", this.cosKey));
        }
        if (!fileMetadata.isFile()) {
            throw new IOException("The cos key [%s] is a directory object. Can not split parts for it.");
        }
        this.reset();
        this.splitPartProcess = true;
        if (copyRemaining > 0L) {
            long firstByte = 0L;
            long lastByte = 0L;
            if (copyRemaining >= this.partSize) {
                this.initializeMultipartUploadIfNeed();
                try {
                    lastByte = firstByte + this.partSize - 1L;
                    for (copyRemaining = Math.min(newLen, fileMetadata.getLength()); copyRemaining >= this.partSize; copyRemaining -= lastByte - firstByte + 1L) {
                        LOG.debug("Executing the uploadPartCopy [cosKey: {}, uploadId: {}, partNumber: {}].", new Object[]{this.cosKey, this.uploadId, this.localParts.size() + 1});
                        UploadPartCopy uploadPartCopy = new UploadPartCopy(this.cosKey, this.cosKey, this.localParts.size() + 1, firstByte, lastByte);
                        this.uploadPartCopy(uploadPartCopy);
                        this.localParts.add(null);
                        firstByte = lastByte + 1L;
                        lastByte = firstByte + this.partSize - 1L;
                    }
                }
                catch (Exception exception) {
                    LOG.error("Failed to breakDown the cos key [{}]. Abort it.", (Object)this.cosKey, (Object)exception);
                    this.abort();
                    throw new IOException(exception);
                }
            }
            if (copyRemaining > 0L) {
                this.initializeNewLocalCurrentPart();
                LocalPart lastPart = this.localParts.get(this.localParts.size() - 1);
                lastByte = firstByte + copyRemaining - 1L;
                this.fetchBlockFromRemote(lastPart.getBuffer(), firstByte, lastByte);
                lastPart.getBuffer().flipRead();
            }
        }
        if ((deltaPadding = newLen - Math.min(newLen, fileMetadata.getLength())) > 0L) {
            long startPos = Math.min(newLen, fileMetadata.getLength());
            long endPos = newLen - 1L;
            this.padBytes(startPos, endPos);
        }
        this.splitPartProcess = false;
        this.committed = false;
        this.aborted = false;
    }

    public LocalPart getPart(int partNumber) throws IOException {
        this.checkOpened();
        if (this.aborted) {
            throw new IOException("The writing operation for the current file has been committed or aborted.");
        }
        if (this.committed) {
            this.resumeForWrite();
        }
        Preconditions.checkArgument((partNumber > 0 ? 1 : 0) != 0, (Object)String.format("The partNumber [%d] should be a positive integer.", partNumber));
        if (partNumber <= this.localParts.size()) {
            LocalPart part = this.localParts.get(partNumber - 1);
            if (null == part) {
                this.downloadPart(partNumber);
            }
        } else {
            if (this.localParts.size() == 0 || this.localParts.get(this.localParts.size() - 1) == null) {
                this.initializeNewLocalCurrentPart();
            }
            LocalPart lastPart = this.localParts.get(this.localParts.size() - 1);
            lastPart.getBuffer().flipWrite();
            long startPos = (long)this.localParts.size() * this.partSize - (long)(lastPart.getBuffer().limit() - lastPart.getBuffer().getMaxReadablePosition());
            long endPos = (long)(partNumber - 1) * this.partSize - 1L;
            if (startPos <= endPos) {
                this.padBytes(startPos, endPos);
            }
            if (this.localParts.size() == partNumber - 1) {
                this.initializeNewLocalCurrentPart();
            }
        }
        return this.localParts.get(partNumber - 1);
    }

    public void abort() {
        this.checkOpened();
        if (this.aborted) {
            LOG.warn("All modifications have been aborted. Skip the aborting operation.");
            return;
        }
        LOG.info("Aborting the MPU [{}]...", (Object)this.uploadId);
        this.releaseRemoteParts();
        this.releaseLocalParts();
        this.aborted = true;
    }

    public void commitLocalToRemote() throws IOException {
        this.checkOpened();
        if (this.committed) {
            LOG.info("All local modifications has been committed. Skip to the committing operation.");
            return;
        }
        if (this.aborted) {
            LOG.warn("All local modifications has been aborted. Nothing need to be committed.");
            return;
        }
        LOG.info("Committing all local parts to remote... ");
        if (null == this.uploadId && this.localParts.size() == 0) {
            LOG.info("Committing a empty file to remote...");
            this.nativeStore.storeEmptyFile(this.cosKey);
            return;
        }
        if (this.uploadId == null && this.localParts.size() == 1 && this.localParts.get(0) != null) {
            LocalPart lastPart = this.localParts.get(0);
            byte[] md5Hash = null;
            try {
                md5Hash = MD5Utils.calculate(lastPart.getBuffer());
            }
            catch (IOException | NoSuchAlgorithmException exception) {
                LOG.warn("Failed to calculate the MD5 hash for the single part.", (Throwable)exception);
            }
            this.nativeStore.storeFile(this.cosKey, new BufferInputStream(lastPart.getBuffer()), md5Hash, lastPart.getBuffer().flipRead().remaining());
            lastPart.setDirty(false);
        } else {
            this.initializeMultipartUploadIfNeed();
            ArrayList<ListenableFuture> uploadPartFutures = new ArrayList<ListenableFuture>();
            for (int index = 0; index < this.localParts.size(); ++index) {
                final LocalPart part = this.localParts.get(index);
                if (null == part || !part.isDirty()) continue;
                final int partNumber = index + 1;
                ListenableFuture uploadPartFuture = this.listeningExecutorService.submit((Callable)new Callable<PartETag>(){

                    @Override
                    public PartETag call() throws Exception {
                        LOG.debug("Starting to upload the part number [{}] for the MPU [{}].", (Object)partNumber, (Object)MultipartManager.this.uploadId);
                        byte[] md5Hash = null;
                        try {
                            md5Hash = MD5Utils.calculate(part.getBuffer());
                        }
                        catch (IOException | NoSuchAlgorithmException exception) {
                            LOG.warn("Failed to calculate the MD5 hash for the part [{}].", (Object)partNumber, (Object)exception);
                        }
                        part.getBuffer().flipRead();
                        PartETag partETag = MultipartManager.this.nativeStore.uploadPart(new BufferInputStream(part.getBuffer()), MultipartManager.this.cosKey, MultipartManager.this.uploadId, partNumber, part.getBuffer().remaining(), md5Hash, partNumber == MultipartManager.this.localParts.size());
                        MultipartManager.this.partETags.put(partNumber, partETag);
                        part.setDirty(false);
                        LOG.debug("Upload the part number [{}] successfully.", (Object)partETag.getPartNumber());
                        return partETag;
                    }
                });
                uploadPartFutures.add(uploadPartFuture);
            }
            try {
                LOG.info("Waiting to finish part uploads...");
                Futures.allAsList(uploadPartFutures).get();
            }
            catch (InterruptedException e) {
                LOG.error("Interrupt the part upload...", (Throwable)e);
                return;
            }
            catch (ExecutionException e) {
                LOG.error("Cancelling futures...", (Throwable)e);
                for (ListenableFuture future : uploadPartFutures) {
                    future.cancel(true);
                }
                this.abort();
                String exceptionMsg = String.format("multipart upload with id: %s to %s.", this.uploadId, this.cosKey);
                throw new IOException(exceptionMsg);
            }
            this.nativeStore.completeMultipartUpload(this.cosKey, this.uploadId, new ArrayList<PartETag>(this.partETags.values()));
            LOG.info("Complete the MPU [{}] successfully.", (Object)this.uploadId);
        }
        this.committed = true;
    }

    public void close() {
        if (this.closed) {
            return;
        }
        this.releaseRemoteParts();
        this.releaseLocalParts();
        this.aborted = true;
        this.committed = true;
        this.closed = true;
    }

    public long getCurrentSize() {
        this.checkOpened();
        long currentFileSize = 0L;
        for (LocalPart entry : this.localParts) {
            if (null == entry) {
                currentFileSize += this.partSize;
                continue;
            }
            currentFileSize += (long)entry.getBuffer().flipRead().remaining();
        }
        return currentFileSize;
    }

    public long getPartSize() {
        return this.partSize;
    }

    public long getMaxFileSizeLimit() {
        return this.MAX_FILE_SIZE;
    }

    private void uploadPartCopy(UploadPartCopy uploadPartCopy) throws IOException {
        this.checkOpened();
        Preconditions.checkNotNull((Object)uploadPartCopy, (Object)"uploadPartCopy");
        LOG.debug("Start to copy the part: {}.", (Object)uploadPartCopy);
        PartETag partETag = this.nativeStore.uploadPartCopy(this.uploadId, uploadPartCopy.getSrcKey(), uploadPartCopy.getDestKey(), uploadPartCopy.getPartNumber(), uploadPartCopy.getFirstByte(), uploadPartCopy.getLastByte());
        this.partETags.put(uploadPartCopy.getPartNumber(), partETag);
    }

    private void downloadPart(int partNumber) throws IOException {
        this.checkOpened();
        Preconditions.checkArgument((partNumber > 0 && partNumber <= 10000 ? 1 : 0) != 0, (Object)"The partNumber should be a positive integer and less than or equal to 10000.");
        FileMetadata fileMetadata = this.nativeStore.retrieveMetadata(this.cosKey);
        long startPos = (long)(partNumber - 1) * this.partSize;
        long endPos = Math.min((long)partNumber * this.partSize - 1L, fileMetadata.getLength());
        if (startPos > endPos) {
            throw new IOException(String.format("The partNumber pulled [%d] exceeds file size [%d]. part size: %d.", partNumber, fileMetadata.getLength(), this.partSize));
        }
        CosNRandomAccessMappedBuffer randomAccessMappedBuffer = this.getLocalPartResource(MultipartManager.generateLocalPartName(this.cosKey, this.uploadId, partNumber), (int)this.partSize);
        this.fetchBlockFromRemote(randomAccessMappedBuffer, startPos, endPos);
        this.localParts.set(partNumber - 1, new LocalPart(randomAccessMappedBuffer));
    }

    private void fetchBlockFromRemote(CosNRandomAccessMappedBuffer buffer, long startPos, long endPos) throws IOException {
        Preconditions.checkArgument((startPos >= 0L ? 1 : 0) != 0, (Object)String.format("The startPos [%d] should be a non-negative integer.", startPos));
        Preconditions.checkArgument((endPos >= 0L ? 1 : 0) != 0, (Object)String.format("The endPos [%d] should be a non-negative integer.", endPos));
        Preconditions.checkArgument((startPos <= endPos ? 1 : 0) != 0, (Object)String.format("The startPos [%d] should be less than or equals to the endPos [%d].", startPos, endPos));
        Preconditions.checkArgument((endPos - startPos + 1L <= (long)buffer.remaining() ? 1 : 0) != 0, (Object)String.format("The range [%d, %d] exceeds the buffer remaining capacity [%d].", startPos, endPos, buffer.remaining()));
        long remaining = endPos - startPos + 1L;
        if (remaining > 0L) {
            try (InputStream inputStream = this.nativeStore.retrieveBlock(this.cosKey, startPos, endPos);){
                byte[] chunk = new byte[(int)Math.min(4096L, remaining)];
                int readBytes = inputStream.read(chunk);
                buffer.flipWrite();
                while (readBytes > 0 && remaining > 0L) {
                    buffer.put(chunk, 0, readBytes);
                    chunk = new byte[(int)Math.min(4096L, remaining -= (long)readBytes)];
                    readBytes = inputStream.read(chunk);
                }
            }
        }
        buffer.flipRead();
    }

    private void padBytes(long startPos, long endPos) throws IOException {
        int index;
        this.checkOpened();
        Preconditions.checkArgument((startPos >= 0L && endPos >= 0L ? 1 : 0) != 0, (Object)String.format("The startPos [%d] and the endPos [%d] should be a non-negative integer.", startPos, endPos));
        Preconditions.checkArgument((startPos <= endPos ? 1 : 0) != 0, (Object)String.format("The startPos [%d] should be less than the endPos [%d].", startPos, endPos));
        if (this.localParts.size() == 0 || this.localParts.get(this.localParts.size() - 1) == null) {
            this.initializeNewLocalCurrentPart();
        }
        LocalPart lastPart = this.localParts.get(this.localParts.size() - 1);
        lastPart.getBuffer().flipWrite();
        long prePaddingSize = (long)(this.localParts.size() - 1) * this.partSize + (long)lastPart.getBuffer().remaining() + (endPos - startPos + 1L);
        Preconditions.checkArgument((prePaddingSize <= this.MAX_FILE_SIZE ? 1 : 0) != 0, (Object)String.format("The bytes [%d] padded exceeds the maximum file limit [%d]", prePaddingSize, this.MAX_FILE_SIZE));
        int partStartIndex = (int)(startPos / this.partSize);
        int partStartOffset = (int)(startPos % this.partSize);
        int partEndIndex = (int)(endPos / this.partSize);
        int partEndOffset = (int)(endPos % this.partSize);
        for (index = this.localParts.size(); index <= partStartIndex; ++index) {
            this.initializeNewLocalCurrentPart();
            lastPart = this.localParts.get(this.localParts.size() - 1);
            lastPart.getBuffer().flipWrite();
            while (lastPart.getBuffer().hasRemaining()) {
                byte[] chunk = new byte[(int)Math.min(4096L, (long)lastPart.getBuffer().remaining())];
                Arrays.fill(chunk, (byte)0);
                lastPart.getBuffer().put(chunk, 0, chunk.length);
                lastPart.setDirty(true);
            }
            lastPart.getBuffer().flipRead();
        }
        for (index = partStartIndex; index <= partEndIndex; ++index) {
            byte[] chunk;
            if (this.localParts.size() <= index) {
                this.initializeNewLocalCurrentPart();
            }
            LocalPart part = this.localParts.get(index);
            part.getBuffer().flipWrite();
            if (index == partStartIndex) {
                part.getBuffer().position(partStartOffset);
            }
            if (index == partEndIndex) {
                while (part.getBuffer().position() <= partEndOffset) {
                    chunk = new byte[(int)Math.min(4096L, (long)(partEndOffset - part.getBuffer().position() + 1))];
                    Arrays.fill(chunk, (byte)0);
                    part.getBuffer().put(chunk, 0, chunk.length);
                    part.setDirty(true);
                }
            } else {
                while (part.getBuffer().hasRemaining()) {
                    chunk = new byte[(int)Math.min(4096L, (long)part.getBuffer().remaining())];
                    Arrays.fill(chunk, (byte)0);
                    part.getBuffer().put(chunk, 0, chunk.length);
                    part.setDirty(true);
                }
            }
            part.getBuffer().flipRead();
        }
    }

    private void releaseLocalParts() {
        this.checkOpened();
        Iterator<LocalPart> iterator = this.localParts.iterator();
        while (iterator.hasNext()) {
            LocalPart part = iterator.next();
            if (null != part) {
                LocalRandomAccessMappedBufferPool.getInstance().releaseFile(part.getBuffer());
            }
            iterator.remove();
        }
        this.localParts.clear();
    }

    private void releaseRemoteParts() {
        this.checkOpened();
        if (this.committed || this.aborted) {
            LOG.debug("All parts have been committed or aborted. Skip to release for remote parts.");
            return;
        }
        try {
            if (null != this.uploadId && !this.uploadId.isEmpty()) {
                LOG.info("Begin to release remote parts for the cos key [{}]. upload id: {}.", (Object)this.cosKey, (Object)this.uploadId);
                try {
                    this.nativeStore.abortMultipartUpload(this.cosKey, this.uploadId);
                }
                catch (IOException e) {
                    LOG.warn("Abort the MPU [{}] for the cos key [{}].", new Object[]{this.uploadId, this.cosKey, e});
                }
            }
        }
        finally {
            this.partETags.clear();
            this.uploadId = null;
            this.aborted = true;
        }
    }

    private void initializeMultipartUploadIfNeed() throws IOException {
        this.checkOpened();
        if (null == this.uploadId) {
            LOG.info("Initialize a multipart upload for the cos key [{}].", (Object)this.cosKey);
            this.uploadId = this.nativeStore.getUploadId(this.cosKey);
            this.aborted = false;
            this.committed = false;
        }
    }

    private void initializeNewLocalCurrentPart() throws IOException {
        this.checkOpened();
        CosNRandomAccessMappedBuffer buffer = this.getLocalPartResource(MultipartManager.generateLocalPartName(this.cosKey, this.uploadId, this.localParts.size() + 1), (int)this.partSize);
        buffer.clear();
        LocalPart localPart = new LocalPart(buffer);
        localPart.setDirty(true);
        this.localParts.add(localPart);
    }

    private void reset() {
        this.checkOpened();
        this.releaseRemoteParts();
        this.releaseLocalParts();
        this.uploadId = null;
        this.partETags.clear();
        this.aborted = false;
    }

    private void checkOpened() {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"The multipart manager has been closed.");
    }

    private CosNRandomAccessMappedBuffer getLocalPartResource(String fileName, int size) throws IOException {
        this.checkOpened();
        if (LocalRandomAccessMappedBufferPool.getInstance().shouldRelease() && !this.splitPartProcess) {
            LOG.info("Begin to release the local cache for the seekable write.");
            this.commitLocalToRemote();
            this.releaseLocalParts();
            this.resumeForWrite();
        }
        return LocalRandomAccessMappedBufferPool.getInstance().create(fileName, size);
    }

    private static String generateLocalPartName(String cosKey, String uploadId, int partNumber) {
        String cacheFileName;
        try {
            cacheFileName = Hex.encodeHexString((byte[])MD5Utils.calculate(cosKey));
        }
        catch (NoSuchAlgorithmException e) {
            LOG.warn("Failed to calculate the md5 of the cosKey [{}]. Replace it with another form.", (Object)cosKey, (Object)e);
            cacheFileName = cosKey.replace("/", "_");
        }
        if (null == uploadId) {
            return String.format("%s_null_%d", cacheFileName, partNumber);
        }
        return String.format("%s_%s_%s", cosKey, uploadId, partNumber);
    }

    public static final class LocalPart {
        private final CosNRandomAccessMappedBuffer buffer;
        private volatile boolean dirty;

        private LocalPart(CosNRandomAccessMappedBuffer buffer) {
            this.buffer = buffer;
            this.dirty = false;
        }

        public CosNRandomAccessMappedBuffer getBuffer() {
            return this.buffer;
        }

        public void setDirty(boolean dirty) {
            this.dirty = dirty;
        }

        public boolean isDirty() {
            return this.dirty;
        }
    }
}

