package com.tencent.cloud.dlc.jdbc;

import com.qcloud.cos.COSClient;
import com.qcloud.cos.ClientConfig;
import com.qcloud.cos.auth.BasicCOSCredentials;
import com.qcloud.cos.auth.BasicSessionCredentials;
import com.qcloud.cos.region.Region;
import com.tencent.cloud.dlc.jdbc.DlcResultSetMetaData;
import com.tencent.cloud.dlc.jdbc.DlcStatement;
import com.tencent.cloud.dlc.jdbc.client.Account;
import com.tencent.cloud.dlc.jdbc.client.Dlc;
import com.tencent.cloud.dlc.jdbc.cos.DataPage;
import com.tencent.cloud.dlc.jdbc.cos.DlcCosDataParser;
import com.tencent.cloud.dlc.jdbc.utils.StringUtils;
import com.tencent.cloud.dlc.jdbc.utils.TaskUtils;
import com.tencentcloudapi.dlc.v20210125.models.TaskResponseInfo;

import java.io.IOException;
import java.sql.SQLException;
import java.time.Duration;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class COSResult {
    private static final Pattern S3_URI_PATTERN = Pattern.compile("^cosn://([^/]+)/(.+)$");
    private static final Pattern TOKEN_PATTERN = Pattern.compile("^nextMarker=(.*)&nextKey=(.+)&offset=([0-9]+)$");

    private final COSClient cosClient;
    private final String bucketName;
    private final String key;
    private final Duration timeout;
    private final TaskUtils.TaskType taskType;
    private DlcCosDataParser cosDataParser;
    private final int fetchSize;

    public COSResult(TaskResponseInfo queryExecution, Duration timeout, DlcStatement statement) throws SQLException {
        Dlc dlc = statement.connHandle.getDlc();
        Account account = dlc.getAccount();
        Region region = new Region(dlc.getRegion());
        ClientConfig clientConfig = new ClientConfig(region);
        COSClient cosClient = new COSClient(StringUtils.isNullOrEmpty(account.getToken())
                ? new BasicCOSCredentials(account.getSecretId(), account.getSecretKey())
                : new BasicSessionCredentials(account.getSecretId(), account.getSecretKey(),
                account.getToken()), clientConfig);
        this.cosClient = cosClient;
        this.timeout = timeout;
        this.taskType = statement.connHandle.getTaskType();
        this.fetchSize = statement.getFetchSize();
        Matcher matcher = S3_URI_PATTERN.matcher(queryExecution.getOutputPath());
        this.cosDataParser = new DlcCosDataParser();
        if (matcher.matches()) {
            this.bucketName = matcher.group(1);
            this.key = matcher.group(2);
        } else {
            throw new IllegalArgumentException(String.format("The output location \"%s\" is malformed", queryExecution.getOutputPath()));
        }
    }

    public DlcResultSetMetaData getMetaData() throws SQLException {
        try {
           return cosDataParser.readMetadata(cosClient,bucketName,key);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    public DataPage getResult(String nextToken) throws SQLException {
        try { String nextMarker = null;
            String nextFile = null;
            long offset = 0;
            if (!StringUtils.isNullOrEmpty(nextToken)) {
                Matcher matcher = TOKEN_PATTERN.matcher(nextToken);
                if (!matcher.matches()) {
                    throw new IllegalArgumentException("Invalid nextToken");
                }
                nextMarker = matcher.group(1);
                nextFile = matcher.group(2);
                offset = Integer.parseInt(matcher.group(3));
            }
            String extension = ".csv";
            if (taskType.equals(TaskUtils.TaskType.SQLTask)){
                extension = "";
            }

           return cosDataParser.readCsvDataPage(cosClient,bucketName,key,extension,fetchSize,nextMarker, nextFile, offset);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }



}
