/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 *
 */

package com.tencent.cloud.dlc.jdbc;


import com.tencent.cloud.dlc.jdbc.COSResult;
import com.tencent.cloud.dlc.jdbc.DlcResultSet;
import com.tencent.cloud.dlc.jdbc.DlcResultSetMetaData;
import com.tencent.cloud.dlc.jdbc.DlcStatement;
import com.tencent.cloud.dlc.jdbc.cos.DataPage;
import com.tencent.cloud.dlc.jdbc.utils.StringUtils;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Iterator;

public class COSResultSet extends DlcResultSet implements ResultSet {

  /**
   * For logging the time consumption for fetching 10000 rows
   */
  private static final long ACCUM_FETCHED_ROWS = 100000;
  /**
   * The maximum retry time we allow to tolerate the network problem
   */
  private static final int READER_REOPEN_TIME_MAX = 5;
  protected Iterator<Object[]> currentRows;
  protected Object[] currentRow;
  private long fetchedRows = 0;
  private boolean isClosed = false;
  private long startTime;
  private String nextToken;
  private int rowNumber;
  private COSResult cosResult;

  COSResultSet(DlcStatement stmt, DlcResultSetMetaData meta, COSResult cosResult, String nextToken)
          throws SQLException {
    this(stmt, meta, System.currentTimeMillis(), nextToken);
    this.cosResult = cosResult;
  }


  COSResultSet(DlcStatement stmt, DlcResultSetMetaData meta, long startTime, String nextToken)
          throws SQLException {
    super(stmt.getConnection(), stmt, meta);
    int maxRows = stmt.resultSetMaxRows;
    this.nextToken = nextToken;
    this.startTime = startTime;
  }

  protected void checkClosed() throws SQLException {
    if (isClosed) {
      throw new SQLException("The result set has been closed");
    }
  }

  @Override
  public int getRow() throws SQLException {
    checkClosed();
    return (int) fetchedRows;
  }

  @Override
  public int getType() throws SQLException {
    return ResultSet.TYPE_FORWARD_ONLY;
  }

  @Override
  public boolean isClosed() throws SQLException {
    return isClosed;
  }

  @Override
  public void close() throws SQLException {
    if (isClosed) {
      return;
    }
    isClosed = true;
    conn.log.info("the result set has been closed");
  }

  @Override
  public boolean next() throws SQLException {
    checkClosed();
    try {
      ensureResults();
      rowNumber++;
      if (currentRows.hasNext()) {
        currentRow = currentRows.next();
      } else {
        currentRow = null;
      }
      return currentRow != null;
    } catch (InterruptedException ie) {
      Thread.currentThread().interrupt();
      throw new SQLException(ie);
    }
  }

  @Override
  protected Object[] rowAtCursor() throws SQLException {
    if (currentRow == null) {
      throw new SQLException("the row should be not-null, row=" + fetchedRows);
    }

    if (currentRow.length == 0) {
      throw new SQLException("the row should have more than 1 column , row=" + fetchedRows);
    }

    return currentRow;
  }

  private void ensureResults() throws SQLException, InterruptedException {
    if (shouldLoadNextPage()) {
      try {
        DataPage dataPage = cosResult.getResult(nextToken);
        this.nextToken = dataPage.getNextToken();
        currentRows = dataPage.getRows();
      } catch (Exception e) {
        e.printStackTrace();
      }
    }
  }

  protected boolean shouldLoadNextPage() {
    return (rowNumber == 0 && currentRows == null) || (!StringUtils.isNullOrEmpty(nextToken)  && !currentRows.hasNext());
  }
}
