package com.tencent.cloud.dlc.jdbc;

import java.sql.*;
import java.util.*;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicLong;

import com.tencent.cloud.dlc.jdbc.client.Account;
import com.tencent.cloud.dlc.jdbc.client.Dlc;
import com.tencent.cloud.dlc.jdbc.utils.DlcLogger;
import com.tencent.cloud.dlc.jdbc.utils.JdbcColumn;
import com.tencent.cloud.dlc.jdbc.utils.TaskUtils;
import com.tencent.cloud.dlc.jdbc.utils.Utils;
import com.tencentcloudapi.dlc.v20210125.models.KVPair;
import org.slf4j.MDC;

public class DlcConnection extends WrapperAdapter implements Connection {

  private static final AtomicLong CONNECTION_ID_GENERATOR = new AtomicLong(0);
  private final Dlc dlc;
  private final Properties info;
  private final List<Statement> stmtHandles;
  private final String charset;


  /**
   * Per-connection logger. All its statements produced by this connection will share this logger
   */
  protected DlcLogger log;

  private boolean isClosed = false;
  private SQLWarning warningChain = null;
  private String connectionId;
  private boolean disableConnSetting = false;
  private String databaseName = null;
  private String datasourceConnectionName = null;
  private TaskUtils.TaskType taskType = TaskUtils.TaskType.SQLTask;
  private String dataEngineName;
  private String subUin;
  private TaskUtils.ResultType resultType;
  private KVPair[] engineConfig;

  DlcConnection(String url, Properties info) throws SQLException {
    ConnectionResource connectionResource = new ConnectionResource(url, info);
    String secretId = connectionResource.getSecretId();
    String secretKey = connectionResource.getSecretKey();
    String region = connectionResource.getRegion();
    String endpoint = connectionResource.getEndpoint();
    String token = connectionResource.getToken();

    String logConfFile = connectionResource.getLogConfFile();

    connectionId = Long.toString(CONNECTION_ID_GENERATOR.incrementAndGet());
    MDC.put("connectionId", connectionId);
    log = new DlcLogger(
            connectionId, null, logConfFile, false, connectionResource.isEnableDlcLogger());
    String version = Utils.retrieveVersion("driver.version");
    log.info("DLC JDBC driver version : " + version);
    log.info(String.format("endpoint : %s", endpoint));

    Account account = new Account(secretId, secretKey,token);
    dlc = new Dlc(account, region, endpoint);
    this.info = info;
    this.charset = connectionResource.getCharset();
    this.stmtHandles = new ArrayList<>();

    this.disableConnSetting = connectionResource.isDisableConnSetting();
    this.databaseName = connectionResource.getDatabaseName();
    this.datasourceConnectionName = connectionResource.getDatasourceConnectionName();
    this.dataEngineName=connectionResource.getDataEngineName();
    this.subUin = connectionResource.getSubUin();
    this.engineConfig = connectionResource.getEngineConfig();
    try {
      this.taskType = TaskUtils.TaskType.valueOf(connectionResource.getTaskType());
    } catch (Exception e) {
      throw new SQLException("the task type must be SQLTask or SparkSQLTask");
    }
    try {
      this.resultType = TaskUtils.ResultType.valueOf(connectionResource.getResultType());
    } catch (Exception e) {

      throw new SQLException("the result type must be COS or Service");
    }
    String msg = "Connect to dlc successfully";
    log.info(msg);
  }

  @Override
  public CallableStatement prepareCall(String sql) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency)
          throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public CallableStatement prepareCall(
          String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability)
          throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public String nativeSQL(String sql) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean getAutoCommit() throws SQLException {
    return true;
  }

  @Override
  public void setAutoCommit(boolean autoCommit) throws SQLException {
//    if (!autoCommit) {
//      log.error(
//              Thread.currentThread().getStackTrace()[1].getMethodName()
//                      + " to false is not supported!!!");
//      throw new SQLFeatureNotSupportedException("disabling autocommit is not supported");
//    }
  }

  @Override
  public void commit() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public void rollback() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public void rollback(Savepoint savepoint) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public void close() throws SQLException {
    MDC.remove("connectionId");
    if (!isClosed) {
      for (Statement stmt : stmtHandles) {
        if (stmt != null && !stmt.isClosed()) {
          stmt.close();
        }
      }
    }
    isClosed = true;
    log.info("connection closed");
  }

  @Override
  public boolean isClosed() throws SQLException {
    return isClosed;
  }

  @Override
  public DatabaseMetaData getMetaData() throws SQLException {
    checkClosed();
    return new DlcDatabaseMetaData(this);
  }

  @Override
  public boolean isReadOnly() throws SQLException {
    return false;
  }

  @Override
  public void setReadOnly(boolean readOnly) throws SQLException {
//    if (readOnly) {
//      log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
//      throw new SQLFeatureNotSupportedException("enabling read-only is not supported");
//    }
  }

  @Override
  public String getCatalog() throws SQLException {
    return datasourceConnectionName;
  }

  @Override
  public void setCatalog(String catalog) throws SQLException {
    this.datasourceConnectionName = catalog;
  }

  @Override
  public int getTransactionIsolation() throws SQLException {
    return Connection.TRANSACTION_NONE;
  }

  @Override
  public void setTransactionIsolation(int level) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public SQLWarning getWarnings() throws SQLException {
    return warningChain;
  }

  @Override
  public void clearWarnings() throws SQLException {
    warningChain = null;
  }

  @Override
  public Map<String, Class<?>> getTypeMap() throws SQLException {
    return Collections.emptyMap();
  }

  @Override
  public void setTypeMap(Map<String, Class<?>> map) throws SQLException {
//    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
//    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getHoldability() throws SQLException {
    return ResultSet.CLOSE_CURSORS_AT_COMMIT;
  }

  @Override
  public void setHoldability(int holdability) throws SQLException {
//    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
//    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public Savepoint setSavepoint() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public Savepoint setSavepoint(String name) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }


  @Override
  public void releaseSavepoint(Savepoint savepoint) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public Statement createStatement() throws SQLException {
    checkClosed();
    DlcStatement stmt = new DlcStatement(this, false);
    stmtHandles.add(stmt);
    return stmt;
  }

  @Override
  public Statement createStatement(
          int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
    return createStatement(resultSetType,resultSetConcurrency);
  }

  @Override
  public Statement createStatement(int resultSetType, int resultSetConcurrency)
          throws SQLException {
    checkClosed();

    switch (resultSetType) {
      case ResultSet.TYPE_FORWARD_ONLY:
        break;
      default:
        throw new SQLFeatureNotSupportedException(
                "only support statement with ResultSet type: ResultSet.TYPE_FORWARD_ONLY");
    }

    switch (resultSetConcurrency) {
      case ResultSet.CONCUR_READ_ONLY:
        break;

      default:
        throw new SQLFeatureNotSupportedException(
                "only support statement with ResultSet concurrency: CONCUR_READ_ONLY");
    }

    DlcStatement stmt = new DlcStatement(this, false);
    stmtHandles.add(stmt);
    return stmt;
  }

  @Override
  public PreparedStatement prepareStatement(String sql) throws SQLException {
    DlcPreparedStatement stmt = new DlcPreparedStatement(this, sql);
    stmtHandles.add(stmt);
    return stmt;
  }

  @Override
  public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency)
          throws SQLException {
   return prepareStatement(sql);
  }

  @Override
  public PreparedStatement prepareStatement(
          String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability)
          throws SQLException {
    log.info("prepareStatement2");
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
    log.info("prepareStatement3");
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
    log.info("prepareStatement4");
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
    log.info("prepareStatement5");
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public Clob createClob() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public Blob createBlob() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public NClob createNClob() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public SQLXML createSQLXML() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean isValid(int timeout) throws SQLException {
    return true;
  }

  @Override
  public void setClientInfo(String name, String value) throws SQLClientInfoException {
    this.info.put(name, value);
  }

  @Override
  public String getClientInfo(String name) throws SQLException {
    return this.info.getProperty(name);
  }

  @Override
  public Properties getClientInfo() throws SQLException {
    return this.info;
  }

  @Override
  public void setClientInfo(Properties properties) throws SQLClientInfoException {
    this.info.putAll(properties);
  }

  @Override
  public Array createArrayOf(String typeName, Object[] elements) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public Struct createStruct(String typeName, Object[] attributes) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public String getSchema() throws SQLException {
    checkClosed();
    return databaseName;
  }

  @Override
  public void setSchema(String schema) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public void abort(Executor executor) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public void setNetworkTimeout(Executor executor, int milliseconds) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getNetworkTimeout() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  public Dlc getDlc() {
    return this.dlc;
  }

  private void checkClosed() throws SQLException {
    if (isClosed) {
      throw new SQLException("the connection has already been closed");
    }
  }

  public boolean disableConnSetting() {
    return disableConnSetting;
  }

  public String getCharset() {
    return charset;
  }

  public String getDatabaseName() {
    return databaseName;
  }

  public void setDatabaseName(String databaseName) {
    this.databaseName = databaseName;
  }

  public String getDatasourceConnectionName() {
    return datasourceConnectionName;
  }

  public void setDatasourceConnectionName(String datasourceConnectionName) {
    this.datasourceConnectionName = datasourceConnectionName;
  }

  public void setDataEngineName(String dataEngineName) {
    this.dataEngineName = dataEngineName;
  }

  public String getDataEngineName() { return dataEngineName; }

  public String getSubUin() { return subUin; }

  public TaskUtils.TaskType getTaskType() {
    return taskType;
  }

  public void setTaskType(TaskUtils.TaskType taskType) {
    this.taskType = taskType;
  }

  public TaskUtils.ResultType getResultType() {
    return resultType;
  }

  public void setResultType(TaskUtils.ResultType resultType) {
    this.resultType = resultType;
  }

  public KVPair[] getEngineConfig() {
    return engineConfig;
  }

  public void setEngineConfig(KVPair[] engineConfig) {
    this.engineConfig = engineConfig;
  }
}
