package com.tencent.cloud.dlc.jdbc;

import java.sql.*;
import java.util.*;

import com.tencent.cloud.dlc.jdbc.utils.DlcLogger;
import com.tencent.cloud.dlc.jdbc.utils.JdbcColumn;
import com.tencent.cloud.dlc.jdbc.utils.StringUtils;
import com.tencent.cloud.dlc.jdbc.utils.Utils;
import com.tencentcloudapi.common.exception.TencentCloudSDKException;
import com.tencentcloudapi.dlc.v20210125.models.*;
import com.tencentcloudapi.lowcode.v20210108.models.DescribeDataSourceListRequest;

public class DlcDatabaseMetaData extends WrapperAdapter implements DatabaseMetaData {
  // Table types
  public static final String TABLE_TYPE_TABLE = "TABLE";
  public static final String TABLE_TYPE_VIEW = "VIEW";
  // Column names
  public static final String COL_NAME_TABLE_CAT = "TABLE_CAT";
  public static final String COL_NAME_TABLE_CATALOG = "TABLE_CATALOG";
  public static final String COL_NAME_TABLE_SCHEM = "TABLE_SCHEM";
  public static final String COL_NAME_TABLE_NAME = "TABLE_NAME";
  public static final String COL_NAME_TABLE_TYPE = "TABLE_TYPE";
  public static final String COL_NAME_REMARKS = "REMARKS";
  public static final String COL_NAME_TYPE_CAT = "TYPE_CAT";
  public static final String COL_NAME_TYPE_SCHEM = "TYPE_SCHEM";
  public static final String COL_NAME_TYPE_NAME = "TYPE_NAME";
  public static final String COL_NAME_SELF_REFERENCING_COL_NAME = "SELF_REFERENCING_COL_NAME";
  public static final String COL_NAME_REF_GENERATION = "REF_GENERATION";
  public static final String DatasourceConnectionName ="DatasourceConnectionName";

  private static final String DRIVER_NAME = "dlc-jdbc";
  private static final String PRODUCT_NAME = "DLC";
  private static final String SCHEMA_TERM = "default_schema";
  private static final String CATALOG_TERM = "default_catalog";
  private static final String PROCEDURE_TERM = "N/A";
  private static final int TABLE_NAME_LENGTH = 128;

  private static final String DEFAULT_CATALOG_NAME = "DataLakeCatalog";
  private static final Long METADATA_MAX_LENGTH = 1000L;




  private final DlcLogger log;
  private DlcConnection conn;

  public DlcDatabaseMetaData(DlcConnection connection) {
    this.conn = connection;
    this.log = connection.log;
  }

  @Override
  public <T> T unwrap(Class<T> iface) throws SQLException {
    return super.unwrap(iface);
  }

  @Override
  public boolean isWrapperFor(Class<?> iface) throws SQLException {
    return super.isWrapperFor(iface);
  }

  @Override
  public long getMaxLogicalLobSize() throws SQLException {
    return DatabaseMetaData.super.getMaxLogicalLobSize();
  }

  @Override
  public boolean supportsRefCursors() throws SQLException {
    return DatabaseMetaData.super.supportsRefCursors();
  }

  @Override
  public boolean allProceduresAreCallable() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean allTablesAreSelectable() throws SQLException {
    return true;
  }

  @Override
  public String getURL() throws SQLException {
    return conn.getDlc().getEndpoint();
  }

  @Override
  public String getUserName() throws SQLException {
    return conn.getDlc().getAccount().getSecretId();
  }

  @Override
  public boolean isReadOnly() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean nullsAreSortedHigh() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean nullsAreSortedLow() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean nullsAreSortedAtStart() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean nullsAreSortedAtEnd() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public String getDatabaseProductName() throws SQLException {
    return PRODUCT_NAME;
  }

  @Override
  public String getDatabaseProductVersion() throws SQLException {
    return Utils.retrieveVersion("sdk.version");
  }

  @Override
  public String getDriverName() throws SQLException {
    return DRIVER_NAME;
  }

  @Override
  public String getDriverVersion() throws SQLException {
    return Utils.retrieveVersion("driver.version");
  }

  @Override
  public int getDriverMajorVersion() {
    try {
      return Integer.parseInt(Utils.retrieveVersion("driver.version").split("\\.")[0]);
    } catch (Exception e) {
      e.printStackTrace();
      return 1;
    }
  }

  @Override
  public int getDriverMinorVersion() {
    try {
      return Integer.parseInt(Utils.retrieveVersion("driver.version").split("\\.")[1]);
    } catch (Exception e) {
      e.printStackTrace();
      return 0;
    }
  }

  @Override
  public boolean usesLocalFiles() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean usesLocalFilePerTable() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsMixedCaseIdentifiers() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean storesUpperCaseIdentifiers() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean storesLowerCaseIdentifiers() throws SQLException {
    return true;
  }

  @Override
  public boolean storesMixedCaseIdentifiers() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsMixedCaseQuotedIdentifiers() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean storesUpperCaseQuotedIdentifiers() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean storesLowerCaseQuotedIdentifiers() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean storesMixedCaseQuotedIdentifiers() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public String getIdentifierQuoteString() throws SQLException {
    return "`";
  }

  @Override
  public String getSQLKeywords() throws SQLException {
    return "overwrite ";
  }

  @Override
  public String getNumericFunctions() throws SQLException {
    return " ";
  }

  @Override
  public String getStringFunctions() throws SQLException {
    return " ";
  }

  @Override
  public String getSystemFunctions() throws SQLException {
    return " ";
  }

  @Override
  public String getTimeDateFunctions() throws SQLException {
    return "  ";
  }

  @Override
  public String getSearchStringEscape() throws SQLException {
//    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName() );
//    throw new SQLFeatureNotSupportedException();
    return "";
  }

  @Override
  public String getExtraNameCharacters() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsAlterTableWithAddColumn() throws SQLException {
    return true;
  }

  @Override
  public boolean supportsAlterTableWithDropColumn() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsColumnAliasing() throws SQLException {
    return true;
  }

  @Override
  public boolean nullPlusNonNullIsNull() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsConvert() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsConvert(int fromType, int toType) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsTableCorrelationNames() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsDifferentTableCorrelationNames() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsExpressionsInOrderBy() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsOrderByUnrelated() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsGroupBy() throws SQLException {
    return true;
  }

  @Override
  public boolean supportsGroupByUnrelated() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsGroupByBeyondSelect() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsLikeEscapeClause() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsMultipleResultSets() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsMultipleTransactions() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsNonNullableColumns() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsMinimumSQLGrammar() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsCoreSQLGrammar() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsExtendedSQLGrammar() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsANSI92EntryLevelSQL() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsANSI92IntermediateSQL() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsANSI92FullSQL() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsIntegrityEnhancementFacility() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsOuterJoins() throws SQLException {
    return true;
  }

  @Override
  public boolean supportsFullOuterJoins() throws SQLException {
    return true;
  }

  @Override
  public boolean supportsLimitedOuterJoins() throws SQLException {
    return true;
  }

  @Override
  public String getSchemaTerm() throws SQLException {
    return SCHEMA_TERM;
  }

  @Override
  public String getProcedureTerm() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public String getCatalogTerm() throws SQLException {
    return CATALOG_TERM;
  }

  @Override
  public boolean isCatalogAtStart() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public String getCatalogSeparator() throws SQLException {
    return ".";
  }

  @Override
  public boolean supportsSchemasInDataManipulation() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsSchemasInProcedureCalls() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsSchemasInTableDefinitions() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsSchemasInIndexDefinitions() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsSchemasInPrivilegeDefinitions() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsCatalogsInDataManipulation() throws SQLException {
    return true;
  }

  @Override
  public boolean supportsCatalogsInProcedureCalls() throws SQLException {
    return true;
  }

  @Override
  public boolean supportsCatalogsInTableDefinitions() throws SQLException {
    return true;
  }

  @Override
  public boolean supportsCatalogsInIndexDefinitions() throws SQLException {
    return true;
  }

  @Override
  public boolean supportsCatalogsInPrivilegeDefinitions() throws SQLException {
    return true;
  }

  @Override
  public boolean supportsPositionedDelete() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsPositionedUpdate() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsSelectForUpdate() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsStoredProcedures() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsSubqueriesInComparisons() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsSubqueriesInExists() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsSubqueriesInIns() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsSubqueriesInQuantifieds() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsCorrelatedSubqueries() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsUnion() throws SQLException {
    return true;
  }

  @Override
  public boolean supportsUnionAll() throws SQLException {
    return true;
  }

  @Override
  public boolean supportsOpenCursorsAcrossCommit() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsOpenCursorsAcrossRollback() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsOpenStatementsAcrossCommit() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsOpenStatementsAcrossRollback() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxBinaryLiteralLength() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxCharLiteralLength() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxColumnNameLength() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxColumnsInGroupBy() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxColumnsInIndex() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxColumnsInOrderBy() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxColumnsInSelect() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxColumnsInTable() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxConnections() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxCursorNameLength() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxIndexLength() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxSchemaNameLength() throws SQLException {
    return 128;
  }

  @Override
  public int getMaxProcedureNameLength() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxCatalogNameLength() throws SQLException {
    return 128;
  }

  @Override
  public int getMaxRowSize() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean doesMaxRowSizeIncludeBlobs() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxStatementLength() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxStatements() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxTableNameLength() throws SQLException {
    return TABLE_NAME_LENGTH;
  }

  @Override
  public int getMaxTablesInSelect() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getMaxUserNameLength() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getDefaultTransactionIsolation() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsTransactions() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsTransactionIsolationLevel(int level) throws SQLException {
    return false;
  }

  @Override
  public boolean supportsDataDefinitionAndDataManipulationTransactions() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsDataManipulationTransactionsOnly() throws SQLException {
    return false;
  }

  @Override
  public boolean dataDefinitionCausesTransactionCommit() throws SQLException {
    return false;
  }

  @Override
  public boolean dataDefinitionIgnoredInTransactions() throws SQLException {
    return false;
  }

  @Override
  public ResultSet getProcedures(String catalog, String schemaPattern, String procedureNamePattern)
          throws SQLException {
    // Return an empty result set
    DlcResultSetMetaData meta = new DlcResultSetMetaData(getProceduresMeta());
    return new DlcStaticResultSet(getConnection(), meta);
  }

  @Override
  public ResultSet getProcedureColumns(
          String catalog, String schemaPattern, String procedureNamePattern, String columnNamePattern)
          throws SQLException {
    // Return an empty result set
    Column stupidPlaceholders = new Column();
    stupidPlaceholders.setName("STUPID_PLACEHOLDERS");
    stupidPlaceholders.setType(DlcType.STRING.name());

    Column uselessPlaceholders = new Column();
    uselessPlaceholders.setName("USELESS_PLACEHOLDER");
    uselessPlaceholders.setType(DlcType.STRING.name());

    DlcResultSetMetaData meta = new DlcResultSetMetaData(new Column[]{stupidPlaceholders, uselessPlaceholders});
    return new DlcStaticResultSet(getConnection(), meta);
  }

  @Override
  public ResultSet getTables(
          String catalog, String schemaPattern, String tableNamePattern, String[] types)
          throws SQLException {
    List<Object[]> rows = new ArrayList<>();

    log.info("schemaPattern:"+schemaPattern+",tableNamePattern:"+tableNamePattern);

    ResultSet schemas = getSchemas(catalog, schemaPattern);
    while (schemas.next()) {
      String tableSchema = schemas.getString(COL_NAME_TABLE_SCHEM);
      if (catalogMatches(catalog, schemas.getString(COL_NAME_TABLE_CATALOG))
              && schemaMatches(schemaPattern, tableSchema)) {
        DescribeTablesRequest describeTablesRequest = new DescribeTablesRequest();
        describeTablesRequest.setDatasourceConnectionName(catalog);
        describeTablesRequest.setDatabaseName(tableSchema);
        describeTablesRequest.setLimit(METADATA_MAX_LENGTH);

        long pullTableTimes = 0L;
        long totalTableCount = 0L;
        DescribeTablesResponse describeTablesResponse = null;
        do {
          describeTablesRequest.setOffset(pullTableTimes * METADATA_MAX_LENGTH);
          try {
            describeTablesResponse = conn.getDlc().getClient().DescribeTables(describeTablesRequest);
          } catch (TencentCloudSDKException e) {
            throw new SQLException(e);
          }
          if (pullTableTimes == 0) {
            totalTableCount = describeTablesResponse.getTotalCount();
          }

          convertTablesToRow(catalog, tableNamePattern, describeTablesResponse.getTableList(), null, rows);
          pullTableTimes++;
        } while (totalTableCount - METADATA_MAX_LENGTH * pullTableTimes > 0);


        DescribeViewsRequest describeViewsRequest = new DescribeViewsRequest();
        describeViewsRequest.setDatasourceConnectionName(catalog);
        describeViewsRequest.setDatabaseName(tableSchema);
        describeViewsRequest.setLimit(METADATA_MAX_LENGTH);

        long pullViewTimes = 0L;
        long totalViewCount = 0L;
        DescribeViewsResponse describeViewsResponse = null;
        do {
          describeViewsRequest.setOffset(pullTableTimes * METADATA_MAX_LENGTH);
          try {
            describeViewsResponse = conn.getDlc().getClient().DescribeViews(describeViewsRequest);
          } catch (TencentCloudSDKException e) {
            throw new SQLException(e);
          }
          if (pullViewTimes == 0) {
            totalViewCount = describeViewsResponse.getTotalCount();
          }

          convertViewsToRow(catalog, tableNamePattern, describeViewsResponse.getViewList(), null, rows);
          pullViewTimes++;
        } while (totalViewCount - METADATA_MAX_LENGTH * pullViewTimes > 0);
      }
    }

    DlcResultSetMetaData meta =
            new DlcResultSetMetaData(getTableMeta());

    sortRows(rows, new int[]{3, 0, 1, 2});
    return new DlcStaticResultSet(getConnection(), meta, rows.iterator());

  }

  public void convertViewsToRow(String catalog, String tableNamePattern, ViewResponseInfo[] viewResponseInfo, String[] types, List<Object[]> rows) {
    for (ViewResponseInfo info : viewResponseInfo) {
      if (Utils.matchPattern(info.getViewBaseInfo().getViewName(), tableNamePattern)) {
        String tableType = TABLE_TYPE_VIEW;
        if (types != null && types.length != 0) {
          if (!Arrays.asList(types).contains(tableType)) {
            continue;
          }
        }
        Object[] rowVals = {
                catalog,
                info.getViewBaseInfo().getDatabaseName(),
                info.getViewBaseInfo().getViewName(),
                tableType,
                null, null, null, null, null, null};
        rows.add(rowVals);
      }
    }
  }


  public void convertTablesToRow(String catalog, String tableNamePattern, TableResponseInfo[] TableResponseInfo, String[] types, List<Object[]> rows) {
    for (TableResponseInfo info : TableResponseInfo) {
      if (Utils.matchPattern(info.getTableBaseInfo().getTableName(), tableNamePattern)) {
        String tableType = TABLE_TYPE_TABLE;
        if (types != null && types.length != 0) {
          if (!Arrays.asList(types).contains(tableType)) {
            continue;
          }
        }
        Object[] rowVals = {
                catalog,
                info.getTableBaseInfo().getDatabaseName(),
                info.getTableBaseInfo().getTableName(),
                tableType,
                null, null, null, null, null, null};
        rows.add(rowVals);
      }
    }
  }


  private boolean catalogMatches(String catalog, String actual) {
    return catalog == null || catalog.equalsIgnoreCase(actual);
  }

  private boolean schemaMatches(String schemaPattern, String actual) {
    return Utils.matchPattern(actual, schemaPattern);
  }

  @Override
  public ResultSet getSchemas() throws SQLException {
    return getSchemas(null, null);
  }

  @Override
  public ResultSet getCatalogs() throws SQLException {
    Column tableCat = new Column();
    tableCat.setName(COL_NAME_TABLE_CAT);
    tableCat.setType(DlcType.STRING.name());

    String catalogNames = conn.getDatasourceConnectionName();


    Filter filter = new Filter();
    filter.setName(DatasourceConnectionName);

    filter.setValues(new String[]{conn.getDatasourceConnectionName()});
    Filter[] filters = new Filter[]{filter};
    DescribeDatasourceConnectionRequest request =new DescribeDatasourceConnectionRequest();
    request.setFilters(filters);
    List<Object[]> rows = new ArrayList<>();
    try {
      DescribeDatasourceConnectionResponse response = conn.getDlc().getClient().DescribeDatasourceConnection(request);
      Arrays.stream(response.getConnectionSet()).forEach(datasourceConnectionInfo -> {
          rows.add(new String[]{datasourceConnectionInfo.getDatasourceConnectionName()});
      });
    } catch (TencentCloudSDKException e) {
      throw new SQLException(e);
    }
    if (rows.size() == 0) {
      rows.add(new String[]{DEFAULT_CATALOG_NAME});
    }
    DlcResultSetMetaData meta = new DlcResultSetMetaData(new Column[]{tableCat});
    return new DlcStaticResultSet(getConnection(), meta, rows.iterator());
  }

  @Override
  public ResultSet getTableTypes() throws SQLException {
    List<Object[]> rows = new ArrayList<>();

    Column tableType = new Column();
    tableType.setName(COL_NAME_TABLE_TYPE);
    tableType.setType(DlcType.STRING.name());

    DlcResultSetMetaData meta = new DlcResultSetMetaData(new Column[]{tableType});

    rows.add(new String[]{TABLE_TYPE_TABLE});
    rows.add(new String[]{TABLE_TYPE_VIEW});

    return new DlcStaticResultSet(getConnection(), meta, rows.iterator());
  }

  @Override
  public ResultSet getColumns(
          String catalog, String schemaPattern, String tableNamePattern, String columnNamePattern)
          throws SQLException {
    log.info("catalog:"+catalog);
    log.info("schemaPattern:"+schemaPattern);
    log.info("tableNamePattern:"+tableNamePattern);
    if (catalog == null) {
      catalog = DEFAULT_CATALOG_NAME;
    }
    if (schemaPattern == null) {
      throw new SQLException("Database name must be given when getColumns");
    }
    if (tableNamePattern == null) {
      throw new SQLException("Table name must be given when getColumns");
    }

    List<Object[]> rows = new ArrayList<Object[]>();
    if (!tableNamePattern.trim().isEmpty() && !"%".equals(tableNamePattern.trim())
            && !"*".equals(tableNamePattern.trim())) {
      DescribeTableRequest describeTableRequest = new DescribeTableRequest();
      describeTableRequest.setDatasourceConnectionName(catalog);
      describeTableRequest.setDatabaseName(schemaPattern);
      describeTableRequest.setTableName(tableNamePattern);
      try {
        DescribeTableResponse describeTableResponse = conn.getDlc().getClient().DescribeTable(describeTableRequest);
        //add columns
        Column[] columns = describeTableResponse.getTable().getColumns();
        for (int i = 0; i < columns.length; i++) {
          Column col = columns[i];
          JdbcColumn jdbcCol = new JdbcColumn(col.getName(),
                  tableNamePattern,
                  schemaPattern,
                  DlcType.parseFromTypeInfo(col.getType()),
                  col,
                  col.getComment(),
                  i + 1,col.getIsPartition());
          Object[] rowVals =
                  {catalog, jdbcCol.getTableSchema(), jdbcCol.getTableName(), jdbcCol.getColumnName(),
                          (long) jdbcCol.getType(), jdbcCol.getTypeName(), 0, 0,
                          (long) jdbcCol.getDecimalDigits(), (long) jdbcCol.getNumPercRaidx(),
                          (long) jdbcCol.getIsNullable(), jdbcCol.getComment(), null, 0, 0, 0,
                          (long) jdbcCol.getOrdinalPos(), jdbcCol.getIsNullableString(), null, null, null,
                          0,jdbcCol.getIsPartition()};
          rows.add(rowVals);
        }

        String tableType = describeTableResponse.getTable().getTableBaseInfo().getType();
        //add partitions
        Partition[] partitions = describeTableResponse.getTable().getPartitions();
        //todo the partition's type should be COLUMN
        if (partitions != null && partitions.length != 0 && !"MANAGED_TABLE".equalsIgnoreCase(tableType)) {
          for (int i = 0; i < partitions.length; i++) {
            Partition partition = partitions[i];
            if (!StringUtils.isNullOrEmpty(partition.getType())) {
              JdbcColumn jdbcCol = new JdbcColumn(partition.getName(),
                      tableNamePattern,
                      schemaPattern,
                      DlcType.parseFromTypeInfo(partition.getType()),
                      partitionToColumn(partition),
                      partition.getComment(),
                      i + 1 + columns.length,true);
              Object[] rowVals =
                      {catalog, jdbcCol.getTableSchema(), jdbcCol.getTableName(), jdbcCol.getColumnName(),
                              (long) jdbcCol.getType(), jdbcCol.getTypeName(), 0, 0,
                              (long) jdbcCol.getDecimalDigits(), (long) jdbcCol.getNumPercRaidx(),
                              (long) jdbcCol.getIsNullable(), jdbcCol.getComment(), null, 0, 0, 0,
                              (long) jdbcCol.getOrdinalPos(), jdbcCol.getIsNullableString(), null, null, null,
                              0,jdbcCol.getIsPartition()};
              rows.add(rowVals);
            }
          }
        }
      } catch (TencentCloudSDKException e) {
        throw new SQLException(e);
      }
    }

    DlcResultSetMetaData meta = new DlcResultSetMetaData(getColumnMeta());

    return new DlcStaticResultSet(getConnection(), meta, rows.iterator());
  }

  public Column partitionToColumn(Partition partition) {
    if (partition == null) {
      return null;
    }
    Column column = new Column();
    column.setType(partition.getType());
    column.setName(partition.getName());
    column.setComment(partition.getComment());
    return column;
  }

  @Override
  public ResultSet getColumnPrivileges(
          String catalog, String schema, String table, String columnNamePattern) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public ResultSet getTablePrivileges(String catalog, String schemaPattern, String tableNamePattern)
          throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public ResultSet getBestRowIdentifier(
          String catalog, String schema, String table, int scope, boolean nullable)
          throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public ResultSet getVersionColumns(String catalog, String schema, String table)
          throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public ResultSet getPrimaryKeys(String catalog, String schema, String table) throws SQLException {
    // Return an empty result set
    DlcResultSetMetaData meta = new DlcResultSetMetaData(getPrimaryKeysMeta());
    return new DlcStaticResultSet(getConnection(), meta);
  }

  @Override
  public ResultSet getImportedKeys(String catalog, String schema, String table)
          throws SQLException {
    // Return an empty result set
    DlcResultSetMetaData meta = new DlcResultSetMetaData(getImportedKeysMeta());
    return new DlcStaticResultSet(getConnection(), meta);
  }

  @Override
  public ResultSet getExportedKeys(String catalog, String schema, String table)
          throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public ResultSet getCrossReference(String parentCatalog, String parentSchema, String parentTable,
                                     String foreignCatalog, String foreignSchema, String foreignTable)
          throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public ResultSet getTypeInfo() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public ResultSet getIndexInfo(
          String catalog, String schema, String table, boolean unique, boolean approximate)
          throws SQLException {
    // Return an empty result set
    DlcResultSetMetaData meta = new DlcResultSetMetaData(getIndexMeta());
    return new DlcStaticResultSet(getConnection(), meta);
  }

  @Override
  public boolean supportsResultSetType(int type) throws SQLException {
    if (type == ResultSet.TYPE_FORWARD_ONLY) {
      return true;
    } else {
      return false;
    }
  }

  @Override
  public boolean supportsResultSetConcurrency(int type, int concurrency) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean ownUpdatesAreVisible(int type) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean ownDeletesAreVisible(int type) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean ownInsertsAreVisible(int type) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean othersUpdatesAreVisible(int type) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean othersDeletesAreVisible(int type) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean othersInsertsAreVisible(int type) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean updatesAreDetected(int type) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean deletesAreDetected(int type) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean insertsAreDetected(int type) throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsBatchUpdates() throws SQLException {
    return false;
  }

  @Override
  public ResultSet getUDTs(
          String catalog, String schemaPattern, String typeNamePattern, int[] types)
          throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public DlcConnection getConnection() throws SQLException {
    return conn;
  }

  @Override
  public boolean supportsSavepoints() throws SQLException {
    return false;
  }

  @Override
  public boolean supportsNamedParameters() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsMultipleOpenResults() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsGetGeneratedKeys() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public ResultSet getSuperTypes(String catalog, String schemaPattern, String typeNamePattern)
          throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public ResultSet getSuperTables(String catalog, String schemaPattern, String tableNamePattern)
          throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public ResultSet getAttributes(
          String catalog, String schemaPattern, String typeNamePattern, String attributeNamePattern)
          throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsResultSetHoldability(int holdability) throws SQLException {
    return false;
  }

  @Override
  public int getResultSetHoldability() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getDatabaseMajorVersion() throws SQLException {
    try {
      return Integer.parseInt(Utils.retrieveVersion("sdk.version").split("\\.")[0]);
    } catch (Exception e) {
      e.printStackTrace();
      return 1;
    }
  }

  @Override
  public int getDatabaseMinorVersion() throws SQLException {
    try {
      return Integer.parseInt(Utils.retrieveVersion("sdk.version").split("\\.")[1]);
    } catch (Exception e) {
      e.printStackTrace();
      return 0;
    }
  }

  @Override
  public int getJDBCMajorVersion() throws SQLException {

    try {
      return Integer.parseInt(Utils.retrieveVersion("driver.version").split("\\.")[0]);
    } catch (Exception e) {
      e.printStackTrace();
      return 0;
    }
  }

  @Override
  public int getJDBCMinorVersion() throws SQLException {
    try {
      return Integer.parseInt(Utils.retrieveVersion("driver.version").split("\\.")[1]);
    } catch (Exception e) {
      e.printStackTrace();
      return 0;
    }
  }

  @Override
  public int getSQLStateType() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean locatorsUpdateCopy() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean supportsStatementPooling() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public RowIdLifetime getRowIdLifetime() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public ResultSet getSchemas(String catalog, String schemaPattern) throws SQLException {
    log.info("get Schemas:"+catalog+"."+schemaPattern);
    Column tableSchem = new Column();
    tableSchem.setName(COL_NAME_TABLE_SCHEM);
    tableSchem.setType(DlcType.STRING.name());

    Column tableCatalog = new Column();
    tableCatalog.setName(COL_NAME_TABLE_CATALOG);
    tableCatalog.setType(DlcType.STRING.name());

    DlcResultSetMetaData meta = new DlcResultSetMetaData(new Column[]{tableSchem, tableCatalog});
    List<Object[]> rows = new ArrayList<>();
    // in dlc, schema = database, catalog = datasource_connection, the default catalog is CosDataCatalog
    if (catalog == null) {
      catalog = DEFAULT_CATALOG_NAME;
    }
    DescribeDatabasesRequest describeDatabasesRequest = new DescribeDatabasesRequest();
    describeDatabasesRequest.setDatasourceConnectionName(catalog);

    long pullTimes = 0L;
    long totalCount = 0L;
    DescribeDatabasesResponse describeDatabasesResponse = null;
    do {
      describeDatabasesRequest.setOffset(pullTimes * METADATA_MAX_LENGTH);
      describeDatabasesRequest.setLimit(METADATA_MAX_LENGTH);
      try {
        describeDatabasesResponse = conn.getDlc().getClient().DescribeDatabases(describeDatabasesRequest);
      } catch (TencentCloudSDKException e) {
        e.printStackTrace();
        throw new SQLException(e);
      }
      if (pullTimes == 0) {
        totalCount = describeDatabasesResponse.getTotalCount();
      }
      convertDatabasesToRow(catalog, schemaPattern, describeDatabasesResponse.getDatabaseList(), rows);
      pullTimes++;

    } while (totalCount - METADATA_MAX_LENGTH * pullTimes > 0);

    sortRows(rows, new int[]{1, 0});
    return new DlcStaticResultSet(getConnection(), meta, rows.iterator());
  }


  /**
   * Sort rows by specified columns.
   *
   * @param rows Rows. Elements in the list cannot be null and must have the same length.
   * @param columnsToSort Indexes of columns to sort.
   */
  private void sortRows(List<Object[]> rows, int[] columnsToSort) {
    rows.sort((row1, row2) -> {
      Objects.requireNonNull(row1);
      Objects.requireNonNull(row2);
      if (row1.length != row2.length) {
        throw new IllegalArgumentException("Rows have different length");
      }

      for (int i = 0; i < row1.length; i++) {
        for (int idx : columnsToSort) {
          if (row1[idx] != null && row2[idx] != null) {
            int ret = ((String) row1[idx]).compareTo((String) row2[idx]);
            if (ret == 0) {
              continue;
            }
            return ret;
          } else if (row1[idx] != null && row2[idx] == null) {
            return 1;
          } else if (row1[idx] == null && row2[idx] != null) {
            return -1;
          }
        }
      }

      return 0;
    });
  }

  public void convertDatabasesToRow(String catalog, String schemaPattern, DatabaseResponseInfo[] databaseResponseInfos, List<Object[]> rows) {
    for (DatabaseResponseInfo info : databaseResponseInfos) {
      if (Utils.matchPattern(info.getDatabaseName(), schemaPattern)) {
        rows.add(new String[]{info.getDatabaseName(), catalog});
      }
    }
  }

  @Override
  public boolean supportsStoredFunctionsUsingCallSyntax() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean autoCommitFailureClosesAllResultSets() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public ResultSet getClientInfoProperties() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public ResultSet getFunctions(String catalog, String schemaPattern, String functionNamePattern)
          throws SQLException {
    // Return an empty result set
    DlcResultSetMetaData meta = new DlcResultSetMetaData(getFunctionsMeta());
    return new DlcStaticResultSet(getConnection(), meta);
  }

  @Override
  public ResultSet getFunctionColumns(
          String catalog, String schemaPattern, String functionNamePattern, String columnNamePattern)
          throws SQLException {
    //todo construct function columns meta
    DlcResultSetMetaData meta = new DlcResultSetMetaData(getFunctionsMeta());
    return new DlcStaticResultSet(getConnection(), meta);
  }

  @Override
  public ResultSet getPseudoColumns(
          String catalog, String schemaPattern, String tableNamePattern, String columnNamePattern)
          throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean generatedKeyAlwaysReturned() throws SQLException {
    log.error("unsupported method : " + Thread.currentThread().getStackTrace()[1].getMethodName());
    throw new SQLFeatureNotSupportedException();
  }

  public Column[] getTableMeta() {
    List<String> nameList = Arrays.asList(COL_NAME_TABLE_CAT, COL_NAME_TABLE_SCHEM, COL_NAME_TABLE_NAME,
            COL_NAME_TABLE_TYPE, COL_NAME_REMARKS, COL_NAME_TYPE_CAT, COL_NAME_TYPE_SCHEM, COL_NAME_TYPE_NAME,
            COL_NAME_SELF_REFERENCING_COL_NAME, COL_NAME_REF_GENERATION);
    List<DlcType> typeList = Arrays.asList(DlcType.STRING, DlcType.STRING, DlcType.STRING, DlcType.STRING,
            DlcType.STRING, DlcType.STRING, DlcType.STRING, DlcType.STRING, DlcType.STRING, DlcType.STRING);
    return getColumns(nameList, typeList);
  }

  public Column[] getColumns(List<String> nameList, List<DlcType> typeList) {
    List<Column> columnList = new ArrayList<>();
    int columnLength = nameList.size();
    for (int i = 0; i < columnLength; i++) {
      Column column = new Column();
      column.setName(nameList.get(i));
      column.setType(typeList.get(i).name());
      columnList.add(column);
    }
    return columnList.toArray(new Column[columnLength]);
  }


  public Column[] getColumnMeta() {

    List<String> nameList = Arrays.asList("TABLE_CAT", "TABLE_SCHEM", "TABLE_NAME", "COLUMN_NAME", "DATA_TYPE",
            "TYPE_NAME", "COLUMN_SIZE", "BUFFER_LENGTH", "DECIMAL_DIGITS", "NUM_PERC_RADIX", "NULLABLE", "REMARKS",
            "COLUMN_DEF", "SQL_DATA_TYPE", "SQL_DATETIME_SUB", "CHAR_OCTET_LENGTH", "ORDINAL_POSITION", "IS_NULLABLE",
            "SCOPE_CATALOG", "SCOPE_SCHEMA", "SCOPE_TABLE", "SOURCE_DATA_TYPE","IS_PARTITION");
    List<DlcType> typeList = Arrays.asList(DlcType.STRING, DlcType.STRING, DlcType.STRING, DlcType.STRING,
            DlcType.BIGINT, DlcType.STRING, DlcType.BIGINT, DlcType.BIGINT, DlcType.BIGINT, DlcType.BIGINT,
            DlcType.BIGINT, DlcType.STRING, DlcType.STRING, DlcType.BIGINT,
            DlcType.BIGINT, DlcType.BIGINT, DlcType.BIGINT, DlcType.STRING, DlcType.STRING, DlcType.STRING,
            DlcType.STRING, DlcType.BIGINT,DlcType.BOOLEAN);

    return getColumns(nameList, typeList);
  }

  public Column[] getProceduresMeta() {
    List<String> nameList = Arrays.asList("PROCEDURE_CAT", "PROCEDURE_SCHEM", "PROCEDURE_NAME", "RESERVERD",
            "REMARKS", "PROCEDURE_TYPE", "SPECIFIC_NAME");
    List<DlcType> typeList = Arrays.asList(DlcType.STRING, DlcType.STRING, DlcType.STRING,
            DlcType.STRING, DlcType.STRING, DlcType.BIGINT, DlcType.STRING);
    return getColumns(nameList, typeList);
  }


  public Column[] getFunctionsMeta() {
    List<String> nameList = Arrays.asList("FUNCTION_CAT", "FUNCTION_SCHEM",
            "FUNCTION_NAME", "REMARKS", "FUNCTION_TYPE", "SPECIFIC_NAME");
    List<DlcType> typeList = Arrays.asList(DlcType.STRING, DlcType.STRING,
            DlcType.STRING, DlcType.STRING, DlcType.BIGINT, DlcType.STRING);
    return getColumns(nameList, typeList);
  }

  public Column[] getPrimaryKeysMeta() {
    List<String> nameList = Arrays.asList("TABLE_CAT", "TABLE_SCHEM", "TABLE_NAME",
            "COLUMN_NAME", "KEY_SEQ", "PK_NAME");
    List<DlcType> typeList = Arrays.asList(DlcType.STRING, DlcType.STRING,
            DlcType.STRING, DlcType.STRING, DlcType.BIGINT, DlcType.STRING);
    return getColumns(nameList, typeList);
  }

  public Column[] getImportedKeysMeta() {
    List<String> nameList = Arrays.asList("PKTABLE_CAT", "PKTABLE_SCHEM", "PKTABLE_NAME",
            "PKCOLUMN_NAME", "FKTABLE_CAT", "FKTABLE_SCHEM", "FKTABLE_NAME", "FKCOLUMN_NAME",
            "KEY_SEQ", "UPDATE_RULE", "DELETE_RULE", "FK_NAME", "PK_NAME", "DEFERRABILITY");
    List<DlcType> typeList = Arrays.asList(DlcType.STRING, DlcType.STRING, DlcType.STRING,
            DlcType.STRING, DlcType.STRING, DlcType.STRING, DlcType.STRING, DlcType.STRING,
            DlcType.BIGINT, DlcType.BIGINT, DlcType.BIGINT, DlcType.STRING, DlcType.STRING,
            DlcType.STRING);
    return getColumns(nameList, typeList);
  }

  public Column[] getIndexMeta() {
    List<String> nameList = Arrays.asList("TABLE_CAT", "TABLE_SCHEM", "TABLE_NAME", "NON_UNIQUE",
            "INDEX_QUALIFIER", "INDEX_NAME", "TYPE", "ORDINAL_POSITION", "COLUMN_NAME", "ASC_OR_DESC",
            "CARDINALITY", "PAGES", "FILTER_CONDITION");
    List<DlcType> typeList = Arrays.asList(DlcType.STRING, DlcType.STRING, DlcType.STRING, DlcType.BOOLEAN,
            DlcType.STRING, DlcType.STRING, DlcType.SMALLINT, DlcType.SMALLINT, DlcType.STRING,
            DlcType.STRING, DlcType.BIGINT, DlcType.BIGINT, DlcType.STRING);
    return getColumns(nameList, typeList);
  }

}
