/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 *
 */

package com.tencent.cloud.dlc.jdbc;


import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.tencent.cloud.dlc.jdbc.utils.StringUtils;
import com.tencentcloudapi.common.JsonResponseModel;
import com.tencentcloudapi.dlc.v20210125.DlcClient;
import com.tencentcloudapi.dlc.v20210125.models.Column;
import com.tencentcloudapi.dlc.v20210125.models.QueryResultRequest;
import com.tencentcloudapi.dlc.v20210125.models.QueryResultResponse;

public class DlcForwardResultSet extends DlcResultSet implements ResultSet {

  /**
   * For logging the time consumption for fetching 10000 rows
   */
  private static final long ACCUM_FETCHED_ROWS = 100000;
  /**
   * The maximum retry time we allow to tolerate the network problem
   */
  private static final int READER_REOPEN_TIME_MAX = 5;
  protected Iterator<Object[]> currentRows;
  protected Object[] currentRow;
  private long fetchedRows = 0;
  private boolean isClosed = false;
  private long startTime;
  private String nextToken;
  private int rowNumber;

  DlcForwardResultSet(DlcStatement stmt, DlcResultSetMetaData meta, String nextToken)
          throws SQLException {
    this(stmt, meta, System.currentTimeMillis(), nextToken);
  }


  DlcForwardResultSet(DlcStatement stmt, DlcResultSetMetaData meta, long startTime, String nextToken)
          throws SQLException {
    super(stmt.getConnection(), stmt, meta);

    int maxRows = stmt.resultSetMaxRows;
    this.nextToken = nextToken;
    this.startTime = startTime;
  }

  protected void checkClosed() throws SQLException {
    if (isClosed) {
      throw new SQLException("The result set has been closed");
    }
  }

  @Override
  public int getRow() throws SQLException {
    checkClosed();
    return (int) fetchedRows;
  }

  @Override
  public int getType() throws SQLException {
    return ResultSet.TYPE_FORWARD_ONLY;
  }

  @Override
  public boolean isClosed() throws SQLException {
    return isClosed;
  }

  @Override
  public void close() throws SQLException {
    if (isClosed) {
      return;
    }
    isClosed = true;
    conn.log.info("the result set has been closed");
  }

  @Override
  public boolean next() throws SQLException {
    checkClosed();
    try {
      ensureResults();
      rowNumber++;
      if (currentRows == null) {
          throw new SQLException("currentRows is now");
      }
      if (currentRows.hasNext()) {
        currentRow = currentRows.next();
      } else {
        currentRow = null;
      }
      return currentRow != null;
    } catch (InterruptedException ie) {
      Thread.currentThread().interrupt();
      throw new SQLException(ie);
    } catch (Exception e) {
      throw new SQLException(e);
    }
  }

  @Override
  protected Object[] rowAtCursor() throws SQLException {
    if (currentRow == null) {
      throw new SQLException("the row should be not-null, row=" + fetchedRows);
    }

    if (currentRow.length == 0) {
      throw new SQLException("the row should have more than 1 column , row=" + fetchedRows);
    }

    return currentRow;
  }

  private void ensureResults() throws Exception {
    if (shouldLoadNextPage()) {
      try {
        DlcClient client = conn.getDlc().getClient();
        QueryResultRequest queryResultRequest = new QueryResultRequest();
        queryResultRequest.setNextToken(this.nextToken);
        queryResultRequest.setTaskId(stmt.getTaskId());
        QueryResultResponse resultResponse = client.QueryResult(queryResultRequest);
        this.nextToken=resultResponse.getNextToken();
        String resultSet = resultResponse.getResultSet();
        List<Object[]> rows=new ArrayList<>();
        Gson builder= (new GsonBuilder()).excludeFieldsWithoutExposeAnnotation().create();
        List<List<Object>> rsp = builder.fromJson(resultSet,List.class );
        for(List<Object> row:rsp){
             rows.add(row.toArray());
        }
        currentRows = rows.iterator();
      } catch (Exception e) {
        e.printStackTrace();
        conn.log.error(e.getMessage());
        throw new Exception(e);
      }
    }
  }

  protected boolean shouldLoadNextPage() {
    return (rowNumber == 0 && currentRows == null) || (!StringUtils.isNullOrEmpty(nextToken)  && !currentRows.hasNext());
  }
}
