/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 *
 */

package com.tencent.cloud.dlc.jdbc;

import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.tencent.cloud.dlc.jdbc.utils.JdbcColumn;
import com.tencentcloudapi.dlc.v20210125.models.Column;

public class DlcResultSetMetaData extends WrapperAdapter implements ResultSetMetaData {

  private final Column[] columns;
  private final List<DlcType> columnTypes;
  private Map<String, Integer> nameIndexMap;

  private String catalogName = " ";
  private String schemeName = " ";
  private String tableName = " ";

  public DlcResultSetMetaData(Column[] columns) {
    this.columns = columns;
    this.columnTypes = new ArrayList<DlcType>();
    if (columns != null) {
      for (int i = 0; i < columns.length; i++) {
        DlcType type = DlcType.parseFromTypeInfo(columns[i].getType());
        columnTypes.add(type);
      }
    }
  }

  @Override
  public String getCatalogName(int column) throws SQLException {
    return catalogName;
  }

  @Override
  public String getColumnClassName(int column) throws SQLException {
    DlcType type = columnTypes.get(toZeroIndex(column));
    String typeName = type.name();
    if (typeName.equals("varchar")) {
      return String.class.getName();
    }
    return Object.class.getName();
  }

  @Override
  public int getColumnCount() throws SQLException {
    return columns.length;
  }

  @Override
  public int getColumnDisplaySize(int column) throws SQLException {
    DlcType dlcType = columnTypes.get(toZeroIndex(column));
    return JdbcColumn.columnDisplaySize(dlcType);
  }



  @Override
  public String getColumnLabel(int column) throws SQLException {
    return columns[toZeroIndex(column)].getName();
  }

  @Override
  public String getColumnName(int column) throws SQLException {
    return columns[toZeroIndex(column)].getName();
  }

  @Override
  public int getColumnType(int column) throws SQLException {
    DlcType type = DlcType.parseFromTypeInfo(columns[toZeroIndex(column)].getType());
    return JdbcColumn.dlcTypeToSqlType(type);
  }

  @Override
  public String getColumnTypeName(int column) {
    return columns[toZeroIndex(column)].getType();
  }

  @Override
  public int getPrecision(int column) {
    return Math.toIntExact(columns[toZeroIndex(column)].getPrecision());
  }

  @Override
  public int getScale(int column) {
    return Math.toIntExact(columns[toZeroIndex(column)].getScale());
  }

  @Override
  public String getSchemaName(int column) {
    return schemeName;
  }

  @Override
  public String getTableName(int column) throws SQLException {
    return tableName;
  }

  @Override
  public boolean isAutoIncrement(int column) throws SQLException {
    return false;
  }

  @Override
  public boolean isCaseSensitive(int column) throws SQLException {
    DlcType type = columnTypes.get(toZeroIndex(column));
    return JdbcColumn.columnCaseSensitive(type);
  }

  @Override
  public boolean isCurrency(int column) throws SQLException {
    return false;
  }

  @Override
  public boolean isDefinitelyWritable(int column) throws SQLException {
    return false;
  }

  // TODO: check
  @Override
  public int isNullable(int column) throws SQLException {
    return ResultSetMetaData.columnNullable;
  }

  @Override
  public boolean isReadOnly(int column) throws SQLException {
    return true;
  }

  @Override
  public boolean isSearchable(int column) throws SQLException {
    return true;
  }

  @Override
  public boolean isSigned(int column) throws SQLException {
    DlcType type = columnTypes.get(toZeroIndex(column));
    return JdbcColumn.columnSigned(type);
  }

  @Override
  public boolean isWritable(int column) throws SQLException {
    return false;
  }

  public void setTableName(String table) {
    tableName = table;
  }

  public void setSchemeName(String scheme) {
    schemeName = scheme;
  }

  public void setCatalogName(String catalog) {
    catalogName = catalog;
  }

  /**
   * Used by other classes for querying the index from column name
   *
   * @param name
   *     column name
   * @return column index
   */
  public int getColumnIndex(String name) {
    if (nameIndexMap == null) {
      nameIndexMap = new HashMap<String, Integer>();
      for (int i = 0; i < columns.length; ++i) {
        nameIndexMap.put(columns[i].getName(), i + 1);
        nameIndexMap.put(columns[i].getName().toLowerCase(), i + 1);
      }
    }

    Integer index = nameIndexMap.get(name);
    if (index == null) {
      String lowerName = name.toLowerCase();
      if (lowerName.equals(name)) {
        return -1;
      }

      index = nameIndexMap.get(name);
    }

    if (index == null) {
      return -1;
    }

    return index;
  }

  protected int toZeroIndex(int column) {
    if (column <= 0 || column > columns.length) {
      throw new IllegalArgumentException();
    }
    return column - 1;
  }
}
