package com.tencent.cloud.dlc.jdbc;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.sql.*;
import java.time.Duration;
import java.util.Base64;

import com.tencent.cloud.dlc.jdbc.utils.StringUtils;
import com.tencent.cloud.dlc.jdbc.utils.TaskUtils;
import com.tencentcloudapi.common.exception.TencentCloudSDKException;
import com.tencentcloudapi.dlc.v20210125.DlcClient;
import com.tencentcloudapi.dlc.v20210125.models.*;

public class DlcStatement extends WrapperAdapter implements Statement {
  private static final int POLLING_INTERVAL = 500;
  private static final String DLC_TASK_ID = "task-id";
  private static ResultSet EMPTY_RESULT_SET = null;
  boolean updateCountFetched = false;

  static {
    try {
      Column column = new Column();
      column.setName("N/A");
      column.setType(DlcType.STRING.name());
      DlcResultSetMetaData meta =
              new DlcResultSetMetaData(new Column[]{column});
      EMPTY_RESULT_SET = new DlcStaticResultSet(null, meta, null);
    } catch (SQLException e) {
      e.printStackTrace();
    }
  }

  protected int resultSetMaxRows = 0;
  protected int resultSetFetchSize = 10000;
  protected boolean isResultSetScrollable = false;
  protected DlcConnection connHandle;
  protected SQLWarning warningChain = null;
  protected boolean isClosed = false;
  protected ResultSet resultSet = null;
  protected boolean isCancelled = false;
  protected CreateTaskResponse createTaskResponse;
  protected TaskResponseInfo queryExecution;


  DlcStatement(DlcConnection conn) {
    this(conn, false);
  }

  public DlcStatement(DlcConnection connection, boolean isResultSetScrollable) {
    this.connHandle = connection;
    this.isResultSetScrollable = isResultSetScrollable;
  }

  /**
   * check whether the sql is start with `select`,`show`,`desc`.
   * @param sql  `select`,`show`,`desc`
   * @return the result whether the sql is start with `select`,`show`,`desc`
   * @throws SQLException the error
   */
  public boolean isQuery(String sql) throws SQLException {
    if ("DQL".equals(this.queryExecution.getSQLType())){
      return true;
    }
    String sqlWithoutComments = sql.replaceAll("(?<!:)\\/\\/.*|\\/\\*(\\s|.)*?\\*\\/", "");
    BufferedReader reader = new BufferedReader(new StringReader(sqlWithoutComments));
    try {
      String line;
      while ((line = reader.readLine()) != null) {
        if (line.matches("^\\s*(--|#).*")) {
          // skip the comment starting with '--' or '#'
          continue;
        }
        if (line.matches("^\\s*$")) { // skip the whitespace line
          continue;
        }
        // The first none-comment line start with "select"
        if (line.matches("(?i)^(\\s*)(SELECT|SHOW|DESC).*$")) {
          return true;
        } else {
          break;
        }
      }
    } catch (IOException e) {
      throw new SQLException(e);
    }
    return false;
  }

  @Override
  public ResultSet executeQuery(String sql) throws SQLException {
    checkClosed();
    beforeExecute();
    String encodedSql = null;
    try {
      encodedSql = Base64.getEncoder().encodeToString(sql.getBytes("UTF-8"));
    } catch (UnsupportedEncodingException e) {
      connHandle.log.warn(e.getMessage());
      encodedSql = Base64.getEncoder().encodeToString(sql.getBytes());
    }
    runSQL(encodedSql);
    return hasResultSet(sql) ? getResultSet() : EMPTY_RESULT_SET;
  }

  public boolean hasResultSet(String sql) throws SQLException {
    return isQuery(sql);
  }

  private void runSQL(String sql) throws SQLException {
    DlcClient client = connHandle.getDlc().getClient();
    CreateTaskRequest request = new CreateTaskRequest();
    Task task = new Task();
    SQLTask sqlTask = new SQLTask();
    sqlTask.setSQL(sql);
    if (connHandle.getEngineConfig().length>0) {
       sqlTask.setConfig(connHandle.getEngineConfig());
    }

    switch (connHandle.getTaskType()) {
      case SQLTask:
        task.setSQLTask(sqlTask);
        break;
      case SparkSQLTask:
        task.setSparkSQLTask(sqlTask);
        break;
      default:
        throw new SQLException("unsupported task type");
    }
    request.setTask(task);

    request.setDatabaseName(connHandle.getDatabaseName());
    if (StringUtils.isNullOrEmpty(connHandle.getDatasourceConnectionName())) {
      throw new SQLException("data source connection name is empty");
    }
    request.setDatasourceConnectionName(connHandle.getDatasourceConnectionName());
    if (!StringUtils.isNullOrEmpty(connHandle.getDataEngineName())) {
      request.setDataEngineName(connHandle.getDataEngineName());
    }
    if (!StringUtils.isNullOrEmpty(connHandle.getSubUin())) {
      KVPair[] kvPairs = convertRequestConfig();
      request.setConfig(kvPairs);
    }
    try {
      createTaskResponse = client.CreateTask(request);
    } catch (TencentCloudSDKException e) {
      throw new SQLException(e);
    }
    taskIsFinished();
  }

  private KVPair[] convertRequestConfig() {
    KVPair[] kvPairs =new KVPair[1];
    if (!StringUtils.isNullOrEmpty(connHandle.getSubUin())) {
      String encodedSubUin = Base64.getEncoder().encodeToString(connHandle.getSubUin().getBytes());
      KVPair kvPair=new KVPair();
      kvPair.setKey("AuthorityRole");
      kvPair.setValue(encodedSubUin);
      kvPairs[0] = kvPair;
    }
    return kvPairs;
  }

  private boolean taskIsFinished() throws SQLException {
    checkClosed();
    DescribeTasksRequest describeTasksRequest = new DescribeTasksRequest();
    Filter filter = new Filter();
    filter.setName(DLC_TASK_ID);
    filter.setValues(new String[]{createTaskResponse.getTaskId()});
    Filter[] filters = new Filter[]{filter};
    describeTasksRequest.setFilters(filters);
    if (!StringUtils.isNullOrEmpty(connHandle.getSubUin())) {
      KVPair[] kvPairs = convertRequestConfig();
      describeTasksRequest.setConfig(kvPairs);
    }
    DlcClient client = connHandle.getDlc().getClient();

    boolean complete = false;
    try {
      while (!complete) {
        try {
          Thread.sleep(POLLING_INTERVAL);
        } catch (InterruptedException e) {
          break;
        }

        DescribeTasksResponse describeTasksResponse = client.DescribeTasks(describeTasksRequest);
        if (describeTasksResponse.getTotalCount() == 0) {
          throw new SQLException("task not found : " + createTaskResponse.getTaskId());
        } else if (describeTasksResponse.getTotalCount() > 1) {
          throw new SQLException(
                  "Multiple tasks with the same task_id : " + createTaskResponse.getTaskId());
        }
        TaskResponseInfo taskResponseInfo = describeTasksResponse.getTaskList()[0];

        switch (taskResponseInfo.getState().intValue()) {
          case TaskUtils.TASK_SUCCEED:
            connHandle.log.debug("task status : finished");
            complete = true;
            this.queryExecution = taskResponseInfo;
            break;
          case TaskUtils.TASK_FAILED:
            String errorMsg = taskResponseInfo.getOutputMessage();
            connHandle.log.error(errorMsg);
            throw new SQLException(errorMsg);
          case TaskUtils.TASK_DELETED:
            connHandle.log.info("task cancelled");
            throw new SQLException("task cancelled");
          case TaskUtils.TASK_INIT:
          case TaskUtils.TASK_RUNNING:
            connHandle.log.debug("task status: running");
            break;
          default:
        }
        connHandle.log.info("it spend " + taskResponseInfo.getUsedTime() + "ms to run the task");
      }
      connHandle.log.info("task finished");
      return true;
    } catch (TencentCloudSDKException e) {
      throw new SQLException(e);
    }
  }

  private void beforeExecute() throws SQLException {
    // If the statement re-executes another query, the previously-generated resultSet
    // will be implicit closed. And the corresponding temp table will be dropped as well.
    if (resultSet != null) {
      resultSet.close();
      resultSet = null;
    }

    isClosed = false;
    isCancelled = false;
    createTaskResponse = null;
    updateCountFetched = false;
  }

  @Override
  public int executeUpdate(String sql) throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public void close() throws SQLException {
    if (isClosed) {
      return;
    }

    if (resultSet != null) {
      resultSet.close();
      resultSet = null;
    }
    connHandle.log.info("the statement has been closed");
    connHandle = null;
    isClosed = true;
  }

  @Override
  public int getMaxFieldSize() throws SQLException {
    return 0;
  }

  @Override
  public void setMaxFieldSize(int max) throws SQLException {
    if (max < 0) {
      throw new SQLException("max must be >= 0");
    }
    this.resultSetMaxRows = max;
  }

  @Override
  public int getMaxRows() throws SQLException {
    return resultSetMaxRows;
  }

  @Override
  public void setMaxRows(int max) throws SQLException {
  }

  @Override
  public void setEscapeProcessing(boolean enable) throws SQLException {
  }

  @Override
  public int getQueryTimeout() throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public void setQueryTimeout(int seconds) throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public void cancel() throws SQLException {
    checkClosed();
    if (isCancelled) {
      return;
    }
    DlcClient client = connHandle.getDlc().getClient();
    CancelTaskRequest cancelTaskRequest = new CancelTaskRequest();
    cancelTaskRequest.setTaskId(createTaskResponse.getTaskId());
    if (!StringUtils.isNullOrEmpty(connHandle.getSubUin())) {
      KVPair[] kvPairs = convertRequestConfig();
      cancelTaskRequest.setConfig(kvPairs);
    }
    try {
      client.CancelTask(cancelTaskRequest);
    } catch (TencentCloudSDKException e) {
      throw new SQLException(e);
    }
    isCancelled = true;
  }

  @Override
  public SQLWarning getWarnings() throws SQLException {
    return warningChain;
  }

  @Override
  public void clearWarnings() throws SQLException {
    warningChain = null;
  }

  @Override
  public void setCursorName(String name) throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean execute(String sql) throws SQLException {
    checkClosed();
    beforeExecute();
    String encodedSql = null;
    try {
      encodedSql = Base64.getEncoder().encodeToString(sql.getBytes("UTF-8"));
    } catch (UnsupportedEncodingException e) {
      connHandle.log.warn(e.getMessage());
      encodedSql = Base64.getEncoder().encodeToString(sql.getBytes());
    }
    runSQL(encodedSql);
    return hasResultSet(sql);
  }

  @Override
  public ResultSet getResultSet() throws SQLException {
    switch (connHandle.getResultType()){
      case COS:
        getResultSetFromCOS(this);
      default:
        getResultSetFromService();
    }
    return resultSet;
  }

  private void getResultSetFromService() throws SQLException {
    if (resultSet == null || resultSet.isClosed()) {
      try {
        DlcClient client = connHandle.getDlc().getClient();
        QueryResultRequest queryResultRequest = new QueryResultRequest();
        queryResultRequest.setTaskId(createTaskResponse.getTaskId());
        if (!StringUtils.isNullOrEmpty(connHandle.getSubUin())) {
          KVPair[] kvPairs = convertRequestConfig();
          queryResultRequest.setConfig(kvPairs);
        }
        QueryResultResponse resultResponse = client.QueryResult(queryResultRequest);
        Column[] columns = resultResponse.getResultSchema();
        DlcResultSetMetaData metaData = new DlcResultSetMetaData(columns);
        resultSet = new DlcForwardResultSet(this, metaData, "");
      } catch (Exception e) {
        e.printStackTrace();
      }
    }
  }

  private void getResultSetFromCOS(DlcStatement statement) throws SQLException {
    if (resultSet == null || resultSet.isClosed()) {
        COSResult cosResult =new COSResult(queryExecution,Duration.ofSeconds(10),statement);
        DlcResultSetMetaData metaData =  cosResult.getMetaData();
        resultSet = new COSResultSet(this,metaData,cosResult, "");
    }
  }

  @Override
  public int getUpdateCount() throws SQLException {
    checkClosed();
    if (updateCountFetched){
      return -1;
    }
    updateCountFetched = true;
    return 0;
  }

  @Override
  public boolean getMoreResults()  {
    return false;
  }

  @Override
  public int getFetchDirection() throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public void setFetchDirection(int direction) throws SQLException {
  }

  @Override
  public int getFetchSize() throws SQLException {
    checkClosed();
    return resultSetFetchSize;
  }

  @Override
  public void setFetchSize(int rows) throws SQLException {
    checkClosed();
    resultSetFetchSize = rows;
  }

  @Override
  public int getResultSetConcurrency() throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getResultSetType() throws SQLException {
    return ResultSet.TYPE_FORWARD_ONLY;
  }

  @Override
  public void addBatch(String sql) throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public void clearBatch() throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int[] executeBatch() throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public DlcConnection getConnection() throws SQLException {
    return connHandle;
  }

  @Override
  public boolean getMoreResults(int current) throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public ResultSet getGeneratedKeys() throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int executeUpdate(String sql, String[] columnNames) throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean execute(String sql, int[] columnIndexes) throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean execute(String sql, String[] columnNames) throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public int getResultSetHoldability() throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean isClosed() throws SQLException {
    return false;
  }

  @Override
  public boolean isPoolable() throws SQLException {
    return false;
  }

  @Override
  public void setPoolable(boolean poolable) throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public void closeOnCompletion() throws SQLException {
    throw new SQLFeatureNotSupportedException();
  }

  @Override
  public boolean isCloseOnCompletion() throws SQLException {
    return false;
  }

  protected void checkClosed() throws SQLException {
    if (isClosed) {
      throw new SQLException("The statement has been closed");
    }
  }

  public String getTaskId() {
    return createTaskResponse.getTaskId();
  }


}
