package com.tencent.cloud.dlc.jdbc;

import java.io.IOException;
import java.net.URISyntaxException;

import com.tencent.cloud.dlc.jdbc.utils.DlcLogger;
import org.aspectj.lang.JoinPoint;

public aspect PublicMethodAspect {

  pointcut Include(): execution(public * com.tencent.cloud.dlc.jdbc.Dlc*.*(..));

  private DlcLogger logger;

  public PublicMethodAspect() throws IOException, URISyntaxException {
    logger = new DlcLogger(getClass().getName(), null, null, false, false);
  }

  before(): Include() {
    int lineNumber = getCurrentLineNumber(thisJoinPoint);
    String classname = getCurrentClassname(thisJoinPoint);
    String methodName = getCurrentMethodName(thisJoinPoint);
    String args = getCurrentArguments(thisJoinPoint).replaceAll("secret_key=\\w+", "secret_key=***");
    String msg = String.format(
            "Enter: [line %d] [%s] [%s] [%s]", lineNumber, classname, methodName, args);
    logger.debug(msg);
  }

  after() returning(Object ret): Include() {
    int lineNumber = getCurrentLineNumber(thisJoinPoint);
    String classname = getCurrentClassname(thisJoinPoint);
    String methodName = getCurrentMethodName(thisJoinPoint);
    String msg = String.format(
            "Leave: [line %d] [%s] [%s] [%s]", lineNumber, classname, methodName, ret);
    logger.debug(msg);
  }

  after() throwing(Exception e): Include() {
    logger.error("exception happened: ", e);
  }

  private int getCurrentLineNumber(JoinPoint joinPoint) {
    try {
      return joinPoint.getSourceLocation().getLine();
    } catch (Exception e) {
      return -1;
    }
  }

  private String getCurrentClassname(JoinPoint joinPoint) {
    try {
      return joinPoint.getThis().getClass().getName();
    } catch (Exception e) {
      return "N/A";
    }
  }

  private String getCurrentMethodName(JoinPoint joinPoint) {
    try {
      return joinPoint.getSignature().getName();
    } catch (Exception e) {
      return "N/A";
    }
  }

  private String formatObject(Object o) {
    // null
    if (o == null) {
      return "null";
    }
    // object[]
    StringBuilder sb = new StringBuilder();
    if (o instanceof Object[]) {
      sb.append("[");
      int l = ((Object[]) o).length;
      for (int i = 0; i < l; i++) {
        Object oi = ((Object[]) o)[i];
        sb.append(formatObject(oi));
        if (i != l - 1) {
          sb.append(", ");
        }
      }
      sb.append("]");
      return sb.toString();
    }
    // default
    sb.append("'");
    sb.append(o.toString());
    sb.append("'");
    return sb.toString();
  }

  private String getCurrentArguments(JoinPoint joinPoint) {
    try {
      return formatObject(joinPoint.getArgs());
    } catch (Exception e) {
      return "?";
    }
  }
}
