package com.tencent.cloud.dlc.jdbc.cos;

import java.io.IOException;
import java.io.InputStream;

/**
 * @author layyu
 */
public class BoundedInputStream extends InputStream {

    private final InputStream in;
    private final long max;
    private boolean propagateClose;
    private long pos;
    private long mark;

    public BoundedInputStream(InputStream in, long size) {
        this.pos = 0L;
        this.mark = -1L;
        this.propagateClose = true;
        this.max = size;
        this.in = in;
    }

    @Override
    public int read() throws IOException {
        if (this.max >= 0L && this.pos >= this.max) {
            return -1;
        } else {
            int result = this.in.read();
            ++this.pos;
            return result;
        }
    }

    @Override
    public int read(byte[] b) throws IOException {
        return this.read(b, 0, b.length);
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        if (this.max >= 0L && this.pos >= this.max) {
            return -1;
        } else {
            long maxRead = this.max >= 0L ? Math.min((long) len, this.max - this.pos) : (long) len;
            int bytesRead = this.in.read(b, off, (int) maxRead);
            if (bytesRead == -1) {
                return -1;
            } else {
                this.pos += (long) bytesRead;
                return bytesRead;
            }
        }
    }

    @Override
    public long skip(long n) throws IOException {
        long toSkip = this.max >= 0L ? Math.min(n, this.max - this.pos) : n;
        long skippedBytes = this.in.skip(toSkip);
        this.pos += skippedBytes;
        return skippedBytes;
    }

    @Override
    public int available() throws IOException {
        return this.max > 0 && this.pos >= this.max ? 0 : this.in.available();
    }

    @Override
    public String toString() {
        return this.in.toString();
    }

    @Override
    public void close() throws IOException {
        if (this.propagateClose) {
            this.in.close();
        }
    }

    @Override
    public synchronized void reset() throws IOException {
        this.in.reset();
        this.pos = this.mark;
    }

    @Override
    public synchronized void mark(int readlimit) {
        this.in.mark(readlimit);
        this.mark = this.pos;
    }

    @Override
    public boolean markSupported() {
        return this.in.markSupported();
    }

    public boolean isPropagateClose() {
        return this.propagateClose;
    }

    public void setPropagateClose(boolean propagateClose) {
        this.propagateClose = propagateClose;
    }

    public long pos() {
        return this.pos;
    }
}
