package com.tencent.cloud.dlc.jdbc.cos;

import com.qcloud.cos.COSClient;
import com.qcloud.cos.model.*;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;

/**
 * @author layyu
 */
public class CosFileOutputStream extends OutputStream {
    private static final int BLOCK_SIZE = 5 * 1024 * 1024;
    private final String bucket;
    private final String key;
    private OutputStream blockOutputStream;
    private COSClient cosClient;
    private String uploadId;
    private List<PartETag> eTags;
    private int blockId = 0;
    private ByteBuffer byteBuffer;
    private boolean closed = false;
    private int blockWritten = 0;

    public CosFileOutputStream(COSClient cosClient, String bucket, String key) {
        this.cosClient = cosClient;
        this.bucket = bucket;
        this.key = key;
    }

    @Override
    public void write(int b) throws IOException {
        if (this.closed) {
            throw new IOException("writer has been closed");
        }
        if (this.blockOutputStream == null) {
            initBlock();
        }
        byte[] singleBytes = new byte[1];
        singleBytes[0] = (byte) b;
        this.blockOutputStream.write(singleBytes, 0, 1);
        this.blockWritten += 1;
        if (this.blockWritten >= BLOCK_SIZE) {
            this.uploadPart();
            this.blockWritten = 0;
        }
    }

    @Override
    public void write(byte[] b) throws IOException {
        this.write(b, 0, b.length);
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
       if (this.closed) {
           throw new IOException("block stream has been closed");
       }

       if (this.blockOutputStream == null) {
           initBlock();
       }

        while (len > 0) {
            long writeBytes = 0;
            if (this.blockWritten + len > BLOCK_SIZE) {
                writeBytes = BLOCK_SIZE - this.blockWritten;
            } else {
                writeBytes = len;
            }

            this.blockOutputStream.write(b, off, (int) writeBytes);
            this.blockWritten += writeBytes;
            if (this.blockWritten >= BLOCK_SIZE) {
                this.uploadPart();
                this.blockWritten = 0;
            }
            len -= writeBytes;
            off += writeBytes;
        }
    }

    @Override
    public void flush() throws IOException {
        if (this.blockOutputStream == null) {
            initBlock();
        }
        this.blockOutputStream.flush();
    }

    @Override
    public void close() throws IOException {
        if (this.closed) {
            return;
        }

        if (this.blockOutputStream == null) {
            initBlock();
        }

        try {
            this.blockOutputStream.flush();
            this.blockOutputStream.close();
            if (blockId == 0) {
                //single file upload
                putObject();
            } else {
                uploadPart();
                CompleteMultipartUploadRequest request =
                        new CompleteMultipartUploadRequest(bucket, key, uploadId, eTags);
                CompleteMultipartUploadResult result = cosClient.completeMultipartUpload(request);
            }
        } finally {
            this.closed = true;
            this.byteBuffer.clear();
            this.blockWritten = 0;
            this.byteBuffer = null;
            this.cosClient = null;
        }
    }

    public void abort() throws IOException {
        if (this.closed) {
            return;
        }

        try {
            if (this.blockOutputStream != null) {
                this.blockOutputStream.flush();
                this.blockOutputStream.close();
            }
            if (uploadId != null) {
                cosClient.abortMultipartUpload(new AbortMultipartUploadRequest(bucket, key, uploadId));
            }
        } finally {
            this.closed = true;
            this.byteBuffer.clear();
            this.cosClient = null;
            this.byteBuffer = null;
            this.blockOutputStream = null;
        }
    }

    private void initBlock() {
        this.byteBuffer = ByteBuffer.allocate(BLOCK_SIZE);
        this.blockOutputStream = new ByteBufferOutputStream(byteBuffer);
    }

    private void uploadPart() {
        {
            if (uploadId == null) {
                InitiateMultipartUploadResult result = cosClient.initiateMultipartUpload(
                        new InitiateMultipartUploadRequest(bucket, key));
                uploadId = result.getUploadId();
                eTags = new ArrayList<>();
            }
            blockId++;
            UploadPartRequest uploadPartRequest = new UploadPartRequest();
            uploadPartRequest.setUploadId(uploadId);
            byteBuffer.flip();
            uploadPartRequest.setInputStream(new ByteBufferInputStream(byteBuffer));
            UploadPartResult uploadPartResult = cosClient.uploadPart(uploadPartRequest);
            eTags.add(new PartETag(uploadPartResult.getPartNumber(), uploadPartResult.getETag()));
            byteBuffer.clear();
        }
    }

    private void putObject() {
        byteBuffer.flip();
        PutObjectRequest request = new PutObjectRequest(bucket, key,
                new ByteBufferInputStream(byteBuffer), new ObjectMetadata());
        cosClient.putObject(request);
    }
}
