package com.tencent.cloud.dlc.jdbc.cos;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.qcloud.cos.COSClient;
import com.qcloud.cos.exception.CosServiceException;
import com.qcloud.cos.model.*;
import com.tencent.cloud.dlc.jdbc.DlcResultSetMetaData;
import com.tencent.cloud.dlc.jdbc.cos.*;
import com.tencent.cloud.dlc.jdbc.utils.StringUtils;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

public class DlcCosDataParser {

    private static final Gson GSON = new GsonBuilder().disableHtmlEscaping().create();
    private static final String TOKEN_FORMAT = "nextMarker=%s&nextKey=%s&offset=%d";
    private static final Charset UTF_8 = StandardCharsets.UTF_8;


    public DlcResultSetMetaData readMetadata(COSClient cosClient, String bucket, String prefix) throws IOException {
        COSObject cosObject = null;
        GetObjectRequest getObjectRequest = new GetObjectRequest(bucket, getResultMetaFile(prefix));
        try {
            cosObject = cosClient.getObject(getObjectRequest);
        } catch (CosServiceException e) {
            // no such key
            if (e.getStatusCode() == 404) {
                return null;
            }
            throw e;
        }
        if (cosObject == null) {
            return null;
        }
        COSObjectInputStream cosObjectInputStream = cosObject.getObjectContent();
        InputStreamReader inputStreamReader = new InputStreamReader(cosObjectInputStream, StandardCharsets.UTF_8);
        BufferedReader bufferedReader = new BufferedReader(inputStreamReader);
        StringBuilder stringBuilder = new StringBuilder();
        String line;
        while ((line = bufferedReader.readLine()) != null) {
            stringBuilder.append(line).append("\n");
        }
        COSResultMeta cosResultMeta = GSON.fromJson(stringBuilder.toString(), COSResultMeta.class);
        return new DlcResultSetMetaData(cosResultMeta.getColumns());
    }

    private String getResultMetaFile(String prefix) {
        if (!prefix.endsWith("/")) {
            prefix = prefix + "/";
        }
        return prefix + "meta/result.meta.json";
    }


    public  DataPage readCsvDataPage(COSClient cosClient, String bucket, String prefix, String extension,
                                    int maxLine, String nextMarker, String nextReadFile, long offset)
            throws Exception {
        String dataPrefix = getResultDataDir(prefix);
        int lineCount = 0;
        long currentOffset = 0;
        String currentReadFile = null;
        long totalBytes = 0;
        String currentMarker;
        boolean truncated = false;
        List<Object[]> rows = new ArrayList<>();
        ListObjectsRequest listObjectsRequest = new ListObjectsRequest();
        listObjectsRequest.setBucketName(bucket);
        listObjectsRequest.setPrefix(dataPrefix);
        listObjectsRequest.setDelimiter("/");
        listObjectsRequest.setMaxKeys(100);
        ObjectListing objectListing;
        do {
            currentMarker = nextMarker;
            listObjectsRequest.setMarker(currentMarker);
            objectListing = cosClient.listObjects(listObjectsRequest);
            List<COSObjectSummary> objects = objectListing.getObjectSummaries()
                    .stream()
                    .filter(o -> StringUtils.isNullOrEmpty(extension) || o.getKey().endsWith(extension))
                    .sorted(new Comparator<COSObjectSummary>() {
                        @Override
                        public int compare(COSObjectSummary o1, COSObjectSummary o2) {
                            return o1.getKey().compareTo(o2.getKey());
                        }
                    }).collect(Collectors.toList());

            Iterator<COSObjectSummary> objectIterator = objects.iterator();
            while (!truncated && objectIterator.hasNext()) {
                COSObjectSummary o = objectIterator.next();
                currentReadFile = o.getKey();
                currentOffset = 0;
                if (!StringUtils.isNullOrEmpty(nextReadFile)) {
                    if (nextReadFile.compareTo(currentReadFile) > 0) {
                        //skip the one which has been read
                        continue;
                    } else if (nextReadFile.equals(currentReadFile)) {
                        //read from the offset
                        currentOffset = offset;
                    }
                }
                if (o.getSize() <= 0) {
                    continue;
                }
                if (currentOffset >= o.getSize()) {
                    continue;
                }
                long start = Math.max(0, currentOffset);
                long end = o.getSize() - 1;
                GetObjectRequest getObjectRequest = new GetObjectRequest(bucket, currentReadFile);
                getObjectRequest.setRange(start, end);
                COSObject cosObject = cosClient.getObject(getObjectRequest);
                COSObjectInputStream cosObjectInput = cosObject.getObjectContent();
                BoundedInputStream bi = new BoundedInputStream(cosObjectInput, Long.MAX_VALUE);
                InputStreamReader ir = new InputStreamReader(bi, UTF_8);
                BufferedReader br = new BufferedReader(ir);
                CSVReader cr = new CSVReader(br);
                try {
                    String[] record;
                    long preTotalBytes = totalBytes;
                    long consumedBytes = 0;
                    boolean headerSkipped = start > 0;
                    while (!truncated && (record = cr.readNext()) != null) {
                        if (headerSkipped) {
                            rows.add(record);
                            lineCount++;
                        } else {
                            //skip header
                            headerSkipped = true;
                        }
                        consumedBytes = bi.pos() - getByteLength(br.remainingChars(), UTF_8) - ir.remaining();
                        totalBytes = preTotalBytes + consumedBytes;
                        if (lineCount >= maxLine) {
                            truncated = true;
                        }
                    }
                    currentOffset = currentOffset + consumedBytes;
                } finally {
                    try {
                        br.close();
                    } catch (IOException error) {
                        throw error;
                    }
                }
            }
            nextMarker = objectListing.getNextMarker();
            nextReadFile = null;
        } while (objectListing.isTruncated() && !truncated);
        //if truncated, set nextToken to null
        String nextToken = null;
        if (truncated) {
            nextToken = String.format(
                    TOKEN_FORMAT, currentMarker != null ? currentMarker : "", currentReadFile, currentOffset);
        }
        return new DataPage(rows, nextToken);
    }

    public String getResultDataDir(String prefix) {
        if (!prefix.endsWith("/")) {
            prefix = prefix + "/";
        }
        return prefix + "data/";
    }

    private long getByteLength(char[] chars, Charset charset) {
        if (chars == null || chars.length == 0) {
            return 0;
        }
        CharBuffer charBuffer = CharBuffer.allocate(chars.length);
        charBuffer.put(chars);
        charBuffer.flip();
        ByteBuffer byteBuffer = charset.encode(charBuffer);
        return byteBuffer.limit();
    }



}
