package com.tencent.cloud.dlc.jdbc.utils;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.nio.file.Paths;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.*;

import com.tencent.cloud.dlc.jdbc.DlcDriver;

public class DlcLogger {
  private static final String DEFAULT_OUTPUT_DIR = "/tmp";
  private static Map<String, FileHandler> pathToFileHandler = new ConcurrentHashMap<>();

  private boolean enabledlcLogger = false;
  private Logger dlcLogger;
  private org.slf4j.Logger sl4jLogger;

  /**
   * Constructor
   *
   * @param name For both dlc and sl4j logger, name of the logger
   * @param outputPath For dlc logger, output path for file handler
   * @param toConsole For dlc logger, output to console or not
   * @param enabledlcLogger For dlc logger, enable or not
   * @param configFilePath For sl4j logger, config file path
   */
  public DlcLogger(String name,
                    String outputPath,
                    String configFilePath,
                    boolean toConsole,
                    boolean enabledlcLogger) {

    this.enabledlcLogger = enabledlcLogger;

    Objects.requireNonNull(name);

    // Init dlc logger
    if (outputPath == null) {
      outputPath = getDefaultOutputPath();
    }
    if (enabledlcLogger) {
      dlcLogger = Logger.getLogger(name);
      dlcLogger.setLevel(Level.ALL);
      if (toConsole) {
        Handler consoleHandler = new ConsoleHandler();
        consoleHandler.setFormatter(new DlcFormatter());
        consoleHandler.setLevel(Level.ALL);
        dlcLogger.addHandler(consoleHandler);
      }
    }
    try {
      FileHandler fileHandler;
      if (pathToFileHandler.containsKey(outputPath)) {
        fileHandler = pathToFileHandler.get(outputPath);
      } else {
        fileHandler = new FileHandler(outputPath, true);
        fileHandler.setFormatter(new DlcFormatter());
        fileHandler.setLevel(Level.ALL);
        pathToFileHandler.put(outputPath, fileHandler);
      }
      if (enabledlcLogger) {
        dlcLogger.addHandler(fileHandler);
      }
    } catch (IOException e) {
      // ignore
    }

    // Init sl4j logger
    sl4jLogger = LoggerFactory.getLogger(configFilePath, name);
  }

  /**
   * Return the default output path. This method tries to return the dir of source code. If it is
   * not allowed due to security reason, return "/tmp"
   *
   * @return default output path
   */
  public static String getDefaultOutputPath() {
    String outputPath;
    try {
      outputPath = new File(DlcDriver.class.getProtectionDomain().getCodeSource()
          .getLocation().toURI()).getParent();
    } catch (Exception e) {
      outputPath = DEFAULT_OUTPUT_DIR;
    }
    return Paths.get(outputPath, "jdbc.log").toString();
  }

  public synchronized void debug(String msg) {
    if (enabledlcLogger) {
      dlcLogger.fine(msg);
    }
    sl4jLogger.debug(msg);
  }

  public synchronized void info(String msg) {
    if (enabledlcLogger) {
      dlcLogger.info(msg);
    }
    sl4jLogger.info(msg);
  }

  public synchronized void warn(String msg) {
    if (enabledlcLogger) {
      dlcLogger.warning(msg);
    }
    sl4jLogger.warn(msg);
  }

  public synchronized void error(String msg) {
    if (enabledlcLogger) {
      dlcLogger.severe(msg);
    }
    sl4jLogger.error(msg);
  }

  public synchronized void error(String msg, Throwable e) {
    StringWriter sw = new StringWriter();
    PrintWriter pw = new PrintWriter(sw);
    e.printStackTrace(pw);
    if (enabledlcLogger) {
      dlcLogger.severe(msg);
      dlcLogger.severe(sw.toString());
    }
    sl4jLogger.error(msg, e);
  }
}
