/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package com.tencent.cloud.dlc.jdbc.utils;

import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.sql.Types;
import java.util.HashMap;
import java.util.Map;

import com.tencent.cloud.dlc.jdbc.DlcType;
import com.tencentcloudapi.dlc.v20210125.models.Column;

/**
 * Wrap around column attributes in this class so that we can display getColumns() easily.
 */
public class JdbcColumn {
  public static final String DLC_TIMESTAMP_FORMAT = "yyyy-MM-dd HH:mm:ss.SSS";
  public static final String DLC_DATETIME_FORMAT = "yyyy-MM-dd HH:mm:ss";
  public static final String DLC_DATE_FORMAT = "yyyy-MM-dd";
  public static final String DLC_TIME_FORMAT = "HH:mm:ss";
  public static final int DLC_DECIMAL_SCALE = 18;
  private static Map<DlcType, Integer> DLC_SQLTYPE_MAPPER = new HashMap<>();

  static {
    DLC_SQLTYPE_MAPPER.put(DlcType.VOID, Types.NULL);
    DLC_SQLTYPE_MAPPER.put(DlcType.BIGINT, Types.BIGINT);
    DLC_SQLTYPE_MAPPER.put(DlcType.STRING, Types.VARCHAR);
    DLC_SQLTYPE_MAPPER.put(DlcType.DATETIME, Types.TIMESTAMP);
    DLC_SQLTYPE_MAPPER.put(DlcType.DOUBLE, Types.DOUBLE);
    DLC_SQLTYPE_MAPPER.put(DlcType.BOOLEAN, Types.BOOLEAN);
    DLC_SQLTYPE_MAPPER.put(DlcType.DECIMAL, Types.DECIMAL);
    DLC_SQLTYPE_MAPPER.put(DlcType.ARRAY, Types.ARRAY);
    DLC_SQLTYPE_MAPPER.put(DlcType.MAP, Types.JAVA_OBJECT);
    DLC_SQLTYPE_MAPPER.put(DlcType.STRUCT, Types.STRUCT);
    DLC_SQLTYPE_MAPPER.put(DlcType.INT, Types.INTEGER);
    DLC_SQLTYPE_MAPPER.put(DlcType.TINYINT, Types.TINYINT);
    DLC_SQLTYPE_MAPPER.put(DlcType.SMALLINT, Types.SMALLINT);
    DLC_SQLTYPE_MAPPER.put(DlcType.DATE, Types.DATE);
    DLC_SQLTYPE_MAPPER.put(DlcType.TIMESTAMP, Types.TIMESTAMP);
    DLC_SQLTYPE_MAPPER.put(DlcType.FLOAT, Types.FLOAT);
    DLC_SQLTYPE_MAPPER.put(DlcType.CHAR, Types.CHAR);
    DLC_SQLTYPE_MAPPER.put(DlcType.BINARY, Types.BINARY);
    DLC_SQLTYPE_MAPPER.put(DlcType.VARCHAR, Types.VARCHAR);
    DLC_SQLTYPE_MAPPER.put(DlcType.INTERVAL_YEAR_MONTH, Types.OTHER);
    DLC_SQLTYPE_MAPPER.put(DlcType.INTERVAL_DAY_TIME, Types.OTHER);
    DLC_SQLTYPE_MAPPER.put(DlcType.OTHER, Types.OTHER);
  }

  private final String columnName;
  private final String tableName;
  private final String tableSchema;
  private final DlcType type;
  private final String comment;
  private final int ordinalPos;
  private final Column typeInfo;
  private final Boolean isPartition;

  public JdbcColumn(String columnName, String tableName, String tableSchema, DlcType type,
                    Column typeInfo, String comment, int ordinalPos,Boolean isPartition) {
    this.columnName = columnName;
    this.tableName = tableName;
    this.tableSchema = tableSchema;
    this.type = type;
    this.typeInfo = typeInfo;
    this.comment = comment;
    this.ordinalPos = ordinalPos;
    this.isPartition = isPartition;
  }


  public static int dlcTypeToSqlType(DlcType type) throws SQLException {
    if (DLC_SQLTYPE_MAPPER.containsKey(type)) {
      return DLC_SQLTYPE_MAPPER.get(type);
    }
    throw new SQLException("Invalid dlc type: " + type);
  }

  public static boolean columnCaseSensitive(DlcType type) throws SQLException {
    int columnType = dlcTypeToSqlType(type);
    // according to dlcTypeToSqlType possible options are:
    switch (columnType) {
      case Types.NULL:
      case Types.BOOLEAN:
      case Types.TINYINT:
      case Types.SMALLINT:
      case Types.INTEGER:
      case Types.BIGINT:
      case Types.DATE:
      case Types.BINARY:
      case Types.FLOAT:
      case Types.DOUBLE:
      case Types.TIMESTAMP:
      case Types.DECIMAL:
      case Types.OTHER:
      case Types.JAVA_OBJECT:
      case Types.ARRAY:
      case Types.STRUCT:
        return false;
      case Types.CHAR:
      case Types.VARCHAR:
        return true;
      default:
        throw new SQLException("Invalid dlc type: " + columnType);
    }
  }

  public static boolean columnSigned(DlcType type) throws SQLException {
    int columnType = dlcTypeToSqlType(type);
    // according to dlcTypeToSqlType possible options are:
    switch (columnType) {
      case Types.NULL:
      case Types.BOOLEAN:
      case Types.DATE:
      case Types.BINARY:
      case Types.TIMESTAMP:
      case Types.OTHER:
      case Types.JAVA_OBJECT:
      case Types.ARRAY:
      case Types.STRUCT:
      case Types.CHAR:
      case Types.VARCHAR:
        return false;
      case Types.TINYINT:
      case Types.SMALLINT:
      case Types.INTEGER:
      case Types.BIGINT:
      case Types.FLOAT:
      case Types.DOUBLE:
      case Types.DECIMAL:
        return true;
      default:
        throw new SQLException("unknown dlcType");
    }
  }

  public String getColumnName() {
    return columnName;
  }

  public String getTableName() {
    return tableName;
  }

  public String getTableSchema() {
    return tableSchema;
  }

  public int getType() throws SQLException {
    return dlcTypeToSqlType(type);
  }

  public String getComment() {
    return comment;
  }

  public String getTypeName() {
    return typeInfo.getType();
  }

  public int getOrdinalPos() {
    return ordinalPos;
  }

  public int getDecimalDigits() {
    return 0;
  }

  public int getNumPercRaidx() {
    return 10;
  }

  public Boolean getIsPartition() {
    return isPartition;
  }

  public int getIsNullable() {
    return DatabaseMetaData.columnNullable;
  }

  public String getIsNullableString() {
    switch (getIsNullable()) {
      case (DatabaseMetaData.columnNoNulls):
        return "NO";
      case (DatabaseMetaData.columnNullable):
        return "YES";
      case (DatabaseMetaData.columnNullableUnknown):
        return null;
      default:
        return null;
    }
  }

  public static int columnDisplaySize(DlcType dlcType) throws SQLException {
    // according to odpsTypeToSqlType possible options are:
    int columnType = dlcTypeToSqlType(dlcType);
    switch (columnType) {
      case Types.NULL:
        return 4; // "NULL"
      case Types.BOOLEAN:
        return columnPrecision(dlcType);
      case Types.CHAR:
      case Types.VARCHAR:
        return columnPrecision(dlcType);
      case Types.BINARY:
        return Integer.MAX_VALUE; // hive has no max limit for binary
      case Types.TINYINT:
      case Types.SMALLINT:
      case Types.INTEGER:
      case Types.BIGINT:
        return columnPrecision(dlcType) + 1; // allow +/-
      case Types.DATE:
        return 10;
      case Types.TIMESTAMP:
        return columnPrecision(dlcType);
      // see
      // http://download.oracle.com/javase/6/docs/api/constant-values.html#java.lang.Float.MAX_EXPONENT
      case Types.FLOAT:
        return 24; // e.g. -(17#).e-###
      // see
      // http://download.oracle.com/javase/6/docs/api/constant-values.html#java.lang.Double.MAX_EXPONENT
      case Types.DOUBLE:
        return 25; // e.g. -(17#).e-####
      case Types.DECIMAL:
        return columnPrecision(dlcType) + 2; // '-' sign and '.'
      case Types.OTHER:
      case Types.JAVA_OBJECT:
        return columnPrecision(dlcType);
      case Types.ARRAY:
      case Types.STRUCT:
        return Integer.MAX_VALUE;
      default:
        throw new SQLException("Invalid odps type: " + columnType);
    }
  }

  public static int columnPrecision(DlcType dlcType) throws SQLException {
    int columnType = dlcTypeToSqlType(dlcType);
    // according to odpsTypeToSqlType possible options are:
    switch (columnType) {
      case Types.NULL:
        return 0;
      case Types.BOOLEAN:
        return 1;
      case Types.CHAR:
      case Types.VARCHAR:
        return 0xff;
      case Types.BINARY:
        return Integer.MAX_VALUE; // hive has no max limit for binary
      case Types.TINYINT:
        return 3;
      case Types.SMALLINT:
        return 5;
      case Types.INTEGER:
        return 10;
      case Types.BIGINT:
        return 19;
      case Types.FLOAT:
        return 7;
      case Types.DOUBLE:
        return 15;
      case Types.DATE:
        return 10;
      case Types.TIMESTAMP:
        return 29;
      case Types.DECIMAL:
        return 54;
      case Types.OTHER:
      case Types.JAVA_OBJECT: {
        return Integer.MAX_VALUE;
      }
      case Types.ARRAY:
      case Types.STRUCT:
        return Integer.MAX_VALUE;
      default:
        throw new SQLException("Invalid odps type: " + columnType);
    }
  }

}
