/*
 * Copyright (c) 2013-2017 QuartzDesk.com. All Rights Reserved.
 * QuartzDesk.com PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.quartzdesk.api.agent.log;

import com.quartzdesk.api.ApiConst;
import com.quartzdesk.api.ApiDebug;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

/**
 * This wrapper uses Java reflection to look up and invoke the QuartzDesk JVM Agent logging interceptor ({@code
 * com.quartzdesk.agent.api.scheduler.common.log.IExecutingJobLoggingInterceptor}). <p> By using this wrapper,
 * initialization of various QuartzDesk API log interception handlers / appenders no longer fails with
 * ClassNotFoundException when the QuartzDesk JVM Agent is not installed and available on the handler / appender
 * classpath. </p> <p> Please note that this class has no compile-time dependency on the QuartzDesk Agent API, nor on
 * the QuartzDesk Domain API. This is intentional and crucial for the proper function of this wrapper. Please note that
 * there are no QuartzDesk Agent API imports apart from the core Java API ones. </p>
 *
 * @author Jan Moravec
 * @version $Id:$
 */
public class LoggingInterceptorWrapper
{
  /**
   * Default log message priority used if the logging framework specific logging level / priority cannot be mapped.
   */
  public static final MessagePriority DEFAULT_MESSAGE_PRIORITY = MessagePriority.INFO;

  private static final String FQCN_AGENT =
      "com.quartzdesk.agent.Agent";

  private static final String FQCN_AGENT_LOGGING_INTERCEPTOR_EVENT =
      "com.quartzdesk.agent.api.domain.model.log.LoggingEvent";

  private static final String FQCN_AGENT_LOGGING_INTERCEPTOR_EVENT_PRIORITY =
      "com.quartzdesk.agent.api.domain.model.log.LoggingEventPriority";

  /**
   * Instance of {@code com.quartzdesk.agent.api.scheduler.common.log.IExecutingJobLoggingInterceptor} instance.
   */
  private Object loggingInterceptor;

  private Class<?> loggingInterceptorEventClass;
  private Class<?> loggingInterceptorEventPriorityClass;

  /*
   * Cached ILoggingInterceptor methods.
   */
  private Method isInterceptingMethod;
  private Method interceptMethod;

  /*
   * Cached LoggingEvent methods.
   */
  private Method setMessageMethod;
  private Method setPriorityMethod;


  /**
   * Creates a {@link LoggingInterceptorWrapper} instance wrapping the specified QuartzDesk JVM Agent
   * LoggingInterceptor
   * instance.
   *
   * @param loggingInterceptor a QuartzDesk JVM Agent LoggingInterceptor instance.
   * @throws ClassNotFoundException if there is a Java reflection problem finding a class.
   * @throws NoSuchMethodException if there is a Java reflection problem finding a method.
   */
  private LoggingInterceptorWrapper( Object loggingInterceptor )
      throws ClassNotFoundException, NoSuchMethodException
  {
    this.loggingInterceptor = loggingInterceptor;

    Class<?> loggingInterceptorClass = loggingInterceptor.getClass();
    ClassLoader loggingInterceptorClassLoader = loggingInterceptorClass.getClassLoader();

    loggingInterceptorEventClass =
        Class.forName( FQCN_AGENT_LOGGING_INTERCEPTOR_EVENT, true, loggingInterceptorClassLoader );

    loggingInterceptorEventPriorityClass =
        Class.forName( FQCN_AGENT_LOGGING_INTERCEPTOR_EVENT_PRIORITY, true, loggingInterceptorClassLoader );

    //
    // ILoggingInterceptor methods:
    //
    // boolean isIntercepting( Thread thread );
    // void intercept( Thread thread, LoggingEvent loggingEvent );
    //
    isInterceptingMethod = loggingInterceptorClass.getMethod( "isIntercepting", Thread.class );
    interceptMethod = loggingInterceptorClass.getMethod( "intercept", Thread.class, loggingInterceptorEventClass );

    //
    // LoggingEvent methods:
    //
    // void setMessage( String value)
    // void setPriority(LoggingEventPriority value)
    //
    setMessageMethod = loggingInterceptorEventClass.getMethod( "setMessage", String.class );
    setPriorityMethod = loggingInterceptorEventClass.getMethod( "setPriority", loggingInterceptorEventPriorityClass );
  }


  /**
   * Creates and returns a new instance of the {@link LoggingInterceptorWrapper}.
   *
   * @return the created {@link LoggingInterceptorWrapper} instance.
   */
  public static LoggingInterceptorWrapper create()
      throws LoggingInterceptorWrapperException
  {
    try
    {
      Class<?> agentClass = Class.forName( FQCN_AGENT, true, Thread.currentThread().getContextClassLoader() );
      Method getInstanceMethod = agentClass.getMethod( "getInstance" );

      // check method is static
      int getInstanceModifiers = getInstanceMethod.getModifiers();
      if ( Modifier.isPublic( getInstanceModifiers ) && Modifier.isStatic( getInstanceModifiers ) )
      {
        Object agent = getInstanceMethod.invoke( null );

        if ( agent == null )
        {
          throw new LoggingInterceptorWrapperException(
              "Error initializing " + LoggingInterceptorWrapper.class.getName() +
                  ": " + FQCN_AGENT + ".getInstance method returned null."
          );
        }
        else
        {
          Method getLoggingInterceptorMethod = agentClass.getMethod( "getExecutingJobLoggingInterceptor" );
          Object loggingInterceptor = getLoggingInterceptorMethod.invoke( agent );

          if ( loggingInterceptor == null )
          {
            throw new LoggingInterceptorWrapperException(
                "Error initializing " + LoggingInterceptorWrapper.class.getName() +
                    ": " + FQCN_AGENT + ".getExecutingJobLoggingInterceptor method returned null."
            );
          }
          else
          {
            return new LoggingInterceptorWrapper( loggingInterceptor );
          }
        }
      }
      else
      {
        throw new LoggingInterceptorWrapperException(
            "Error initializing " + LoggingInterceptorWrapper.class.getName() +
                ": " + FQCN_AGENT + ".getInstance method is not public static."
        );
      }
    }
    catch ( LoggingInterceptorWrapperException e )
    {
      logWarningException( e );
      throw e;
    }
    catch ( ClassNotFoundException e )
    {
      logWarningException( e );

      // QuartzDesk JVM Agent has not been installed (its classes are not available)
      throw new LoggingInterceptorWrapperException( "Error initializing " + LoggingInterceptorWrapper.class.getName() +
          ": QuartzDesk JVM Agent is not installed (is not on the classpath). Context thread class loader: " +
          ApiConst.NL + ApiDebug.getClassLoaderInfo( Thread.currentThread().getContextClassLoader() ), e );
    }
    catch ( Exception e )
    {
      logWarningException( e );

      // InvocationTargetException, NoSuchMethodException, IllegalAccessException + other unexpected exceptions
      throw new LoggingInterceptorWrapperException(
          "Error initializing " + LoggingInterceptorWrapper.class.getName() + ": " + e.toString(), e );
    }
  }


  /**
   * Invokes the {@code com.quartzdesk.agent.api.scheduler.common.log.IExecutingJobLoggingInterceptor#isIntercepting(Thread)}
   * method on the wrapped QuartzDesk JVM Agent logging interceptor.
   *
   * @param jobThread a job thread.
   * @return the result.
   * @see com.quartzdesk.agent.api.scheduler.common.log.IExecutingJobLoggingInterceptor#isIntercepting(Thread)
   */
  public boolean isIntercepting( Thread jobThread )
  {
    try
    {
      return (Boolean) isInterceptingMethod.invoke( loggingInterceptor, jobThread );
    }
    catch ( Exception e )
    {
      // IllegalAccessException, InvocationTargetException
      logError( "Error invoking isIntercepting(Thread) method on " + loggingInterceptor, e );
    }

    return false;
  }


  /**
   * Invokes the {@code com.quartzdesk.agent.api.scheduler.common.log.IExecutingJobLoggingInterceptor#intercept(Thread,
   * com.quartzdesk.agent.api.domain.model.log.LoggingEvent)} method on the wrapped QuartzDesk JVM
   * Agent logging interceptor.
   *
   * @param thread   a job thread or one of its worker (children) threads registered with the {@link
   *                 WorkerThreadLoggingInterceptorRegistry}.
   * @param priority a log message priority.
   * @param message  a log message.
   * @see com.quartzdesk.agent.api.scheduler.common.log.IExecutingJobLoggingInterceptor#intercept(Thread,
   * com.quartzdesk.agent.api.domain.model.log.LoggingEvent)
   */
  public void intercept( Thread thread, MessagePriority priority, String message )
  {
    try
    {
      // create a LoggingEvent instance
      Object loggingEvent = loggingInterceptorEventClass.newInstance();
      setPriorityMethod.invoke( loggingEvent, convertToAgentLoggingEventPriority( priority ) );
      setMessageMethod.invoke( loggingEvent, message );

      /*
       * The specified thread can be a worker thread (a thread used by the job thread).
       */
      Thread jobThread = WorkerThreadLoggingInterceptorRegistry.INSTANCE.getJobThreadForWorkerThread( thread );
      if ( jobThread == null )
      {
        jobThread = thread;
      }

      interceptMethod.invoke( loggingInterceptor, jobThread, loggingEvent );
    }
    catch ( Exception e )
    {
      // IllegalAccessException, InvocationTargetException
      logError( "Error invoking intercept(Thread, LoggingEvent) method on " + loggingInterceptor, e );
    }
  }


  /**
   * Converts the specified {@link MessagePriority} instance to a {@code com.quartzdesk.agent.api.domain.model.log.LoggingEventPriority}
   * instance (of a class loaded by the Agent classloader!).
   *
   * @param priority a message priority.
   * @return the {@code com.quartzdesk.agent.api.domain.model.log.LoggingEventPriority} instance.
   */
  private Object convertToAgentLoggingEventPriority( MessagePriority priority )
  {
    //return Enum.valueOf( (Class<T>) loggingEventPriorityClass, priority.name() );
    for ( Object enumValueObj : loggingInterceptorEventPriorityClass.getEnumConstants() )
    {
      Enum<?> enumValue = (Enum<?>) enumValueObj;
      if ( priority.name().equals( enumValue.name() ) )
        return enumValueObj;
    }

    throw new IllegalArgumentException(
        "Error converting " + MessagePriority.class.getName() + " value: " + priority + " to " +
            FQCN_AGENT_LOGGING_INTERCEPTOR_EVENT_PRIORITY + " value."
    );
  }


  private static void logWarningException( Throwable e )
  {
    System.out.println( "=== WARNING (source: " + LoggingInterceptorWrapper.class.getName() + ") === " );
    e.printStackTrace( System.out );
    System.out.println( "===" );
  }


  /**
   * Prints the specified error message and exception on the standard error output.
   *
   * @param error an error message.
   * @param e     an exception.
   */
  private static void logError( String error, Throwable e )
  {
    System.out.println( "=== ERROR (source: " + LoggingInterceptorWrapper.class.getName() + ") === " );

    System.out.println( error );

    if ( e != null )
      e.printStackTrace( System.out );

    System.out.println( "===" );
  }


  /**
   * Log message priorities. The priorities directly correspond to {@code com.quartzdesk.agent.api.domain.model.log.LoggingEventPriority}
   * enum values.
   */
  public enum MessagePriority
  {
    TRACE,
    DEBUG,
    INFO,
    WARN,
    ERROR;
  }
}
