/*
 * Copyright (c) 2013-2017 QuartzDesk.com. All Rights Reserved.
 * QuartzDesk.com PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.quartzdesk.api.agent.log;

import java.util.concurrent.Callable;

/**
 * An implementation of the {@link Callable} interface that can be used by job implementation classes
 * to wrap a {@link Callable} instance with the business logic to be executed by a worker thread spawned from
 * the specified job execution thread. This wrapped allows QuartzDesk to intercept log messages produced by
 * executed worker threads.
 *
 * @author Jan Moravec
 * @version $Id:$
 */
public class WorkerThreadCallable<T>
    implements Callable<T>
{
  /**
   * The job execution thread, i.e. the thread spawned by the scheduler to execute a job.
   */
  private Thread jobThread;

  /**
   * The {@link Callable} instance with the worker business logic.
   */
  private Callable<T> workerCallable;


  /**
   * Creates a new {@link WorkerThreadCallable} instance for the specified job execution thread.
   *
   * @param jobThread      a job execution thread.
   * @param workerCallable the wrapped {@link Callable} instance with the worker thread's business logic.
   */
  public WorkerThreadCallable( Thread jobThread, Callable<T> workerCallable )
  {
    this.jobThread = jobThread;
    this.workerCallable = workerCallable;
  }


  @Override
  public T call()
      throws Exception
  {
    /*
     * Start intercepting logging events from the current worker thread (returned by Thread.currentThread())
     * and associate these events with the specified job execution thread.
     */
    WorkerThreadLoggingInterceptorRegistry.INSTANCE.startIntercepting( jobThread );
    try
    {
      // invoke the wrapped Callable instance
      return workerCallable.call();
    }
    finally
    {
      /*
       * Stop intercepting logging events from the current worker thread.
       */
      WorkerThreadLoggingInterceptorRegistry.INSTANCE.stopIntercepting();
    }
  }
}
