/*
 * Copyright (c) 2013-2017 QuartzDesk.com. All Rights Reserved.
 * QuartzDesk.com PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.quartzdesk.api.agent.log;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * A registry that can be used to turn on/off interception of log messages produced by worker threads spawned
 * and used by executing jobs. The registry is to be used like so:
 * <pre>
 *
 * public class MyCallable
 *     implements Callable<Foo>
 * {
 *   private Thread jobThread;
 *
 *   // @param jobThread the main job execution thread that will be using this Callable instance.
 *   public MyCallable( Thread jobThread )
 *   {
 *     this.jobThread = jobThread;
 *   }
 *
 *   public Foo call()
 *       throws Exception
 *   {
 *     // Start intercepting logging events from the current worker thread (returned by Thread.currentThread())
 *     // and associate these events with the specified main job execution thread.
 *     WorkerThreadLoggingInterceptorRegistry.INSTANCE.startIntercepting( jobThread );
 *     try
 *     {
 *       // some business logic and logging
 *       return foo;
 *     }
 *     finally
 *     {
 *       // Stop intercepting logging events from the current worker thread.
 *       WorkerThreadLoggingInterceptorRegistry.INSTANCE.stopIntercepting();
 *     }
 *   }
 * }
 *
 * </pre>
 *
 * @author Jan Moravec
 * @version $Id:$
 */
public class WorkerThreadLoggingInterceptorRegistry
{
  public static final WorkerThreadLoggingInterceptorRegistry INSTANCE = new WorkerThreadLoggingInterceptorRegistry();

  /**
   * The actual registry that maps a worker thread onto the main job execution thread.
   */
  private Map<Thread, Thread> registry = new ConcurrentHashMap<Thread, Thread>();


  /**
   * Private constructor to prevent external instantiation.
   */
  private WorkerThreadLoggingInterceptorRegistry()
  {
  }


  /**
   * Returns the job thread associated with the specified thread.
   *
   * @param workerThread a worker thread.
   * @return the job thread.
   */
  public Thread getJobThreadForWorkerThread( Thread workerThread )
  {
    return registry.get( workerThread );
  }


  /**
   * Starts intercepting log messages produced by the current thread (as returned by {@link Thread#currentThread()})
   * that is used by the specified job thread.
   *
   * @param jobThread a job thread.
   */
  public void startIntercepting( Thread jobThread )
  {
    Thread currentThread = Thread.currentThread();
    if ( currentThread == jobThread )
    {
      throw new IllegalStateException(
          "The job thread argument must not be the current worker thread. It must be the main job thread that spawned or is using the current worker thread." );
    }

    registry.put( currentThread, jobThread );
  }


  /**
   * Stops intercepting log messages produced by the current thread (as returned by {@link Thread#currentThread()}).
   */
  public void stopIntercepting()
  {
    registry.remove( Thread.currentThread() );
  }
}
