/*
 * Copyright (c) 2013-2017 QuartzDesk.com. All Rights Reserved.
 * QuartzDesk.com PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.quartzdesk.api.agent.log;

/**
 * An implementation of the {@link Runnable} interface that can be used by job implementation classes
 * to wrap a {@link Runnable} instance with the business logic to be executed by a worker thread spawned from
 * the specified job execution thread. This wrapped allows QuartzDesk to intercept log messages produced by
 * executed worker threads.
 *
 * @author Jan Moravec
 * @version $Id:$
 */
public class WorkerThreadRunnable
    implements Runnable
{
  /**
   * The job execution thread, i.e. the thread spawned by the scheduler to execute a job.
   */
  private Thread jobThread;

  /**
   * The {@link Runnable} instance with the worker business logic.
   */
  private Runnable workerRunnable;


  /**
   * Creates a new {@link WorkerThreadRunnable} instance for the specified job execution thread.
   *
   * @param jobThread      a job execution thread.
   * @param workerRunnable the wrapped {@link Runnable} instance with the worker thread's business logic.
   */
  public WorkerThreadRunnable( Thread jobThread, Runnable workerRunnable )
  {
    this.jobThread = jobThread;
    this.workerRunnable = workerRunnable;
  }


  @Override
  public void run()
  {
    /*
     * Start intercepting logging events from the current worker thread (returned by Thread.currentThread())
     * and associate these events with the specified job execution thread.
     */
    WorkerThreadLoggingInterceptorRegistry.INSTANCE.startIntercepting( jobThread );
    try
    {
      // invoke the wrapped Runnable instance
      workerRunnable.run();
    }
    finally
    {
      /*
       * Stop intercepting logging events from the current worker thread.
       */
      WorkerThreadLoggingInterceptorRegistry.INSTANCE.stopIntercepting();
    }
  }
}
