/*
 * Copyright (c) 2013-2017 QuartzDesk.com. All Rights Reserved.
 * QuartzDesk.com PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.quartzdesk.api.agent.log.jul;

import com.quartzdesk.agent.Agent;
import com.quartzdesk.agent.IAgent;
import com.quartzdesk.agent.api.domain.model.log.LoggingEvent;
import com.quartzdesk.agent.api.domain.model.log.LoggingEventPriority;
import com.quartzdesk.agent.api.scheduler.common.log.IExecutingJobLoggingInterceptor;
import com.quartzdesk.api.agent.log.WorkerThreadLoggingInterceptorRegistry;

import java.util.logging.ErrorManager;
import java.util.logging.Filter;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.LogRecord;
import java.util.logging.SimpleFormatter;

/**
 * Implementation of a Java util logging handler that passes the log records to the QuartzDesk JVM agent that
 * intercepts
 * log messages produced by executed jobs.
 * <pre>
 * ==== Example logging.properties ====
 * # Handlers
 * handlers = ..., com.quartzdesk.api.common.log.jul.ClassicJulInterceptionHandler
 *
 * # QuartzDesk Agent interception handler
 * # com.quartzdesk.api.agent.log.jul.ClassicJulInterceptionHandler.level = ALL
 * # com.quartzdesk.api.agent.log.jul.ClassicJulInterceptionHandler.formatter = formatter class
 * # com.quartzdesk.api.agent.log.jul.ClassicJulInterceptionHandler.filter = filter class
 *
 * ...
 * ====
 * </pre>
 *
 * <strong> This implementation is statically bound to the QuartzDesk JVM Agent API and to the QuartzDesk domain object
 * API. Therefore this handler requires a JVM with an installed QuartzDesk JVM Agent. </strong>
 *
 * @author Jan Moravec
 * @version $Id:$
 * @see IExecutingJobLoggingInterceptor
 */
@Deprecated
public class ClassicJulInterceptionHandler<E>
    extends Handler
{
  private static final LoggingEventPriority DEFAULT_LOGGING_INTERCEPTOR_EVENT_PRIORITY =
      LoggingEventPriority.INFO;

  private IExecutingJobLoggingInterceptor loggingInterceptor;

  /**
   * Creates a new {@link ClassicJulInterceptionHandler} instance.
   */
  public ClassicJulInterceptionHandler()
  {
    String cname = getClass().getName();

    setLevel( getLevelProperty( cname + ".level", Level.INFO ) );
    setFilter( getFilterProperty( cname + ".filter", null ) );
    setFormatter( getFormatterProperty( cname + ".formatter", new SimpleFormatter() ) );

    /*
     * The following code cannot be here, because JUL handlers are instantiated BEFORE JVM the QuartzDesk JVM Agent.
     * Code moved to the publish method.
     */
//    IAgent agent = AgentLocator.getAgent();
//    loggingInterceptor = agent.getLoggingInterceptor();
  }


  /**
   * If the QuartzDesk agent's logging interceptor wishes to intercept the specified log record, then this method
   * formats the log record and passes the result to the logging interceptor.
   *
   * @param record a log record.
   */
  @Override
  public void publish( LogRecord record )
  {
    // lazy initialization of the loggingInterceptor
    if ( loggingInterceptor == null )
    {
      IAgent agent = Agent.getInstance();
      loggingInterceptor = agent.getExecutingJobLoggingInterceptor();
    }

    // ensure that this log record should be logged by this Handler
    if ( isLoggable( record ) )
    {
      Thread currentThread = Thread.currentThread();

      /*
       * The current thread can be a worker thread (a thread created / used by the main job thread).
       */
      Thread jobThread = WorkerThreadLoggingInterceptorRegistry.INSTANCE.getJobThreadForWorkerThread( currentThread );
      if ( jobThread == null )
      {
        jobThread = currentThread;
      }

      if ( loggingInterceptor != null && loggingInterceptor.isIntercepting( jobThread ) )
      {
        LoggingEvent loggingInterceptorEvent = new LoggingEvent()
            .withPriority( convertLevel2Priority( record.getLevel() ) )
            .withMessage( getFormatter().format( record ) );

        loggingInterceptor.intercept( jobThread, loggingInterceptorEvent );
      }
    }
  }


  @Override
  public void flush()
  {
    // no op
  }


  @Override
  public void close()
      throws SecurityException
  {
    loggingInterceptor = null;
  }


  /**
   * Returns the level for this handler named by the "name". If the property is not defined, then we return the
   * defaultValue. <p/> Method inspired by the {@code getLevelProperty} method in {@link
   * java.util.logging.StreamHandler}.
   *
   * @param name         a logging config property with level name.
   * @param defaultValue the default level.
   * @return the level.
   */
  protected Level getLevelProperty( String name, Level defaultValue )
  {
    LogManager manager = LogManager.getLogManager();
    String val = manager.getProperty( name );
    if ( val == null )
    {
      return defaultValue;
    }
    try
    {
      return Level.parse( val.trim() );
    }
    catch ( Exception ex )
    {
      return defaultValue;
    }
  }


  /**
   * Returns the filter instance of the class named by the "name" property. If the property is not defined or has
   * problems we return the defaultValue. <p/> Method inspired by the {@code getFilterProperty} method in {@link
   * java.util.logging.StreamHandler}.
   *
   * @param name         a logging config property with filter class name.
   * @param defaultValue the default filter.
   * @return the filter.
   */
  protected Filter getFilterProperty( String name, Filter defaultValue )
  {
    LogManager manager = LogManager.getLogManager();
    String val = manager.getProperty( name );
    try
    {
      if ( val != null )
      {
        Class<?> clz = ClassLoader.getSystemClassLoader().loadClass( val );
        return (Filter) clz.newInstance();
      }
    }
    catch ( Exception ex )
    {
      // We got one of a variety of exceptions in creating the
      // class or creating an instance.
      // Drop through.
    }
    // We got an exception.  Return the defaultValue.
    return defaultValue;
  }


  /**
   * Returns the formatter instance of the class named by the "name" property. If the property is not defined or has
   * problems we return the defaultValue. <p/> Method inspired by the {@code getFormatterProperty} method in {@link
   * java.util.logging.StreamHandler}.
   *
   * @param name         a logging config property with formatter class name.
   * @param defaultValue the default formatter.
   * @return the formatter.
   */
  protected Formatter getFormatterProperty( String name, Formatter defaultValue )
  {
    LogManager manager = LogManager.getLogManager();
    String val = manager.getProperty( name );
    try
    {
      if ( val != null )
      {
        Class<?> clz = ClassLoader.getSystemClassLoader().loadClass( val );
        return (Formatter) clz.newInstance();
      }
    }
    catch ( Exception ex )
    {
      // We got one of a variety of exceptions in creating the
      // class or creating an instance.
      // Drop through.
    }
    // We got an exception.  Return the defaultValue.
    return defaultValue;
  }


  /**
   * Returns the {@link LoggingEventPriority} for the specified Log4J level.
   *
   * @param level a Java util logging log level.
   * @return the {@link LoggingEventPriority} for the specified Log4J level.
   */
  protected LoggingEventPriority convertLevel2Priority( Level level )
  {
    if ( level.equals( Level.FINEST ) )
      return LoggingEventPriority.TRACE;
    else if ( level.equals( Level.FINER ) )
      return LoggingEventPriority.TRACE;
    else if ( level.equals( Level.FINE ) )
      return LoggingEventPriority.DEBUG;
    else if ( level.equals( Level.CONFIG ) )
      return LoggingEventPriority.INFO;
    else if ( level.equals( Level.INFO ) )
      return LoggingEventPriority.INFO;
    else if ( level.equals( Level.WARNING ) )
      return LoggingEventPriority.WARN;
    else if ( level.equals( Level.SEVERE ) )
      return LoggingEventPriority.ERROR;
    else
    {
      reportError( "Cannot map logging level: " + level + ". Using default logging interceptor event priority: " +
          DEFAULT_LOGGING_INTERCEPTOR_EVENT_PRIORITY, null, ErrorManager.GENERIC_FAILURE );

      return DEFAULT_LOGGING_INTERCEPTOR_EVENT_PRIORITY;
    }
  }
}
