/*
 * Copyright (c) 2013-2017 QuartzDesk.com. All Rights Reserved.
 * QuartzDesk.com PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.quartzdesk.api.agent.log.log4j2;

import com.quartzdesk.api.agent.log.LoggingInterceptorWrapper;
import com.quartzdesk.api.agent.log.LoggingInterceptorWrapperException;
import com.quartzdesk.api.agent.log.WorkerThreadLoggingInterceptorRegistry;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.appender.AbstractAppender;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginAttr;
import org.apache.logging.log4j.core.config.plugins.PluginElement;
import org.apache.logging.log4j.core.config.plugins.PluginFactory;
import org.apache.logging.log4j.core.helpers.Charsets;
import org.apache.logging.log4j.core.layout.AbstractStringLayout;
import org.apache.logging.log4j.core.layout.PatternLayout;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.nio.charset.Charset;

/**
 * Implementation of a Log4J2 appender that passes the log events to the QuartzDesk JVM agent that intercepts log
 * messages produced by executed jobs.
 * <pre>
 * ==== Example log4j2.xml ====
 * ...
 *
 * &lt;QuartzDesk name="QUARTZDESK_JVM_AGENT"&gt;
 *   &lt;PatternLayout pattern="[%d{ISO8601}] %-5p [%t] [%C:%L] - %m%n"/&gt;
 *   &lt;filters&gt;
 *     &lt;ThresholdFilter level="trace"/&gt;
 *   &lt;/filters&gt;
 * &lt;/QuartzDesk&gt;
 *
 * ...
 *
 * &lt;root level="warn"&gt;
 *   ...
 *   &lt;appender-ref ref="QUARTZDESK_JVM_AGENT"/&gt;
 * &lt;/root&gt;
 * </pre>
 *
 * <strong> This implementation is not statically bound to the QuartzDesk JVM Agent API, nor to the domain object API.
 * Therefore it is safe to use this appender on JVMs running without an installed QuartzDesk JVM Agent. </strong>
 *
 * @author Jan Moravec
 * @version $Id:$
 * @see LoggingInterceptorWrapper
 */
@Plugin( name = "QuartzDeskJvmAgent", category = "Core", elementType = "appender", printObject = true )
public class Log4j2InterceptionAppender<T extends Serializable>
    extends AbstractAppender<T>
{
  /*
   * !!! IMPORTANT NOTE !!!
   *
   * Log4j2 appenders must be registered in org.apache.logging.log4j.core.config.plugins.Log4j2Plugins.dat
   * resources located on the classpath. Log4j2Plugins.dat is a binary file that can be generated using
   * the following command.
   *
   * java -cp log4j-api-2.0-beta7.jar;log4j-core-2.0-beta7.jar;quartzdesk-api-1.0.12-SNAPSHOT.jar org.apache.logging.log4j.core.config.plugins.PluginManager . com.quartzdesk.api.agent.log
   */

  private LoggingInterceptorWrapper loggingInterceptor;

  /**
   * Creates a new {@link Log4j2InterceptionAppender} instance.
   *
   * @param name            The Appender name.
   * @param filter          The Filter to associate with the Appender.
   * @param layout          The layout to use to format the event.
   * @param handleException If true, exceptions will be logged and suppressed. If false errors will be logged and then
   *                        passed to the application.
   */
  protected Log4j2InterceptionAppender( String name, Filter filter, Layout<T> layout,
      boolean handleException )
  {
    super( name, filter, layout, handleException );

    LOGGER.info( "Starting " + Log4j2InterceptionAppender.class.getName() + ", name=" + name );

    try
    {
      loggingInterceptor = LoggingInterceptorWrapper.create();
    }
    catch ( LoggingInterceptorWrapperException e )
    {
//      LOGGER.warn( "Cannot initialize " + LoggingInterceptorWrapper.class.getName() + ". Appender " + getName() +
//          " will be disabled." + Constants.LINE_SEP, e );

      LOGGER.warn( "Cannot initialize " + LoggingInterceptorWrapper.class.getName() + ". Appender " + getName() +
          " will be disabled. Cause: " + e.getMessage() );
    }
  }


  @PluginFactory
  public static <T extends Serializable> Log4j2InterceptionAppender<T> createAppender(
      @PluginAttr( "name" ) String name,
      @PluginAttr( "suppressExceptions" ) String suppress,
      @PluginElement( "layout" ) Layout<T> layout,
      @PluginElement( "filters" ) Filter filter )
  {
    boolean handleExceptions = suppress == null ? true : Boolean.valueOf( suppress );

    if ( name == null )
    {
      LOGGER.error( "No name provided for QuartzDeskAppender appender." );
      return null;
    }

    if ( layout == null )
    {
      //noinspection unchecked
      layout = (Layout<T>) PatternLayout.createLayout( null, null, null, null, null );
    }

    return new Log4j2InterceptionAppender<T>( name, filter, layout, handleExceptions );
  }


  @Override
  public void append( LogEvent event )
  {
    Thread currentThread = Thread.currentThread();

    /*
     * The current thread can be a worker thread (a thread created / used by the main job thread).
     */
    Thread jobThread = WorkerThreadLoggingInterceptorRegistry.INSTANCE.getJobThreadForWorkerThread( currentThread );
    if ( jobThread == null )
    {
      jobThread = currentThread;
    }

    if ( loggingInterceptor != null && loggingInterceptor.isIntercepting( jobThread ) )
    {
      LoggingInterceptorWrapper.MessagePriority priority = convertLevel2Priority( event.getLevel() );

      Layout<T> layout = getLayout();

      if ( layout instanceof AbstractStringLayout )
      {
        byte[] eventBytes = layout.toByteArray( event );

        // we need to determine the charset of the layout that was used to extract bytes from the formatted log event
        Charset charset = getLayoutCharset( (AbstractStringLayout) layout );

        String message = new String( eventBytes, charset );

        loggingInterceptor.intercept( jobThread, priority, message );
      }
      else
      {
        LOGGER.error( "Layout in QuartzDeskAppender must extend AbstractStringLayout. It is typically PatternLayout." );
      }
    }
  }


  /**
   * Returns the {@link LoggingInterceptorWrapper.MessagePriority} for the specified Log4J level.
   *
   * @param level a log4j level.
   * @return the {@link LoggingInterceptorWrapper.MessagePriority} for the specified Log4J level.
   */
  protected LoggingInterceptorWrapper.MessagePriority convertLevel2Priority( Level level )
  {
    // OFF, FATAL, ERROR, WARN, INFO, DEBUG, TRACE, ALL;
    if ( level.equals( Level.TRACE ) )
      return LoggingInterceptorWrapper.MessagePriority.TRACE;
    else if ( level.equals( Level.DEBUG ) )
      return LoggingInterceptorWrapper.MessagePriority.DEBUG;
    else if ( level.equals( Level.INFO ) )
      return LoggingInterceptorWrapper.MessagePriority.INFO;
    else if ( level.equals( Level.WARN ) )
      return LoggingInterceptorWrapper.MessagePriority.WARN;
    else if ( level.equals( Level.ERROR ) )
      return LoggingInterceptorWrapper.MessagePriority.ERROR;
    else if ( level.equals( Level.FATAL ) )  // FATAL mapped to ERROR
      return LoggingInterceptorWrapper.MessagePriority.ERROR;
    else
    {
      LOGGER.warn( "Cannot map logging level: " + level + ". Using default message priority: " +
          LoggingInterceptorWrapper.DEFAULT_MESSAGE_PRIORITY );

      return LoggingInterceptorWrapper.DEFAULT_MESSAGE_PRIORITY;
    }
  }


  /**
   * Returns the {@link Charset} used by the specified layout to format log events and produce their byte array
   * representations.
   *
   * @param layout a layout.
   * @return the {@link Charset}.
   */
  protected <L extends AbstractStringLayout> Charset getLayoutCharset( L layout )
  {
    try
    {
      // getCharset method is protected
      Method method = getCharsetMethod( layout.getClass() );
      if ( method != null )
      {
        method.setAccessible( true );
        Object result = method.invoke( layout );
        if ( result != null && result instanceof Charset )
        {
          return (Charset) result;
        }
      }
    }
    catch ( InvocationTargetException e )
    {
      // should not happen
    }
    catch ( IllegalAccessException e )
    {
      // should not happen
    }

    // fall-back charset
    return Charsets.getSupportedCharset( Charsets.UTF_8.name() );
  }


  /**
   * Returns the {@code getCharset} method from the specified class, or null if not found.
   *
   * @param clazz a class.
   * @return the {@code getCharset} method, or null if not found.
   */
  private Method getCharsetMethod( Class<?> clazz )
  {
    while ( clazz != null )
    {
      try
      {
        return clazz.getDeclaredMethod( "getCharset" );
      }
      catch ( NoSuchMethodException e )
      {
        // search in super-class
        clazz = clazz.getSuperclass();
      }
    }
    return null;
  }
}
