/*
 * Copyright (c) 2013-2017 QuartzDesk.com. All Rights Reserved.
 * QuartzDesk.com PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.quartzdesk.api.agent.log.logback;

import com.quartzdesk.agent.Agent;
import com.quartzdesk.agent.IAgent;
import com.quartzdesk.agent.api.domain.model.log.LoggingEventPriority;
import com.quartzdesk.agent.api.scheduler.common.log.IExecutingJobLoggingInterceptor;
import com.quartzdesk.api.agent.log.WorkerThreadLoggingInterceptorRegistry;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.spi.LoggingEvent;
import ch.qos.logback.core.AppenderBase;
import ch.qos.logback.core.Layout;

/**
 * Implementation of a Logback appender that passes the log events to the QuartzDesk JVM agent that intercepts log
 * messages produced by executed jobs.
 * <pre>
 * ==== Example logback.xml ====
 * ...
 *
 * &lt;!--
 *   Appender that passes all received log events to the QuartzDesk agent to intercept
 *   log events produced by executed Quartz jobs.
 * --&gt;
 * &lt;appender name="QUARTZDESK_JVM_AGENT" class="com.quartzdesk.api.agent.log.logback.ClassicLogbackInterceptionAppender"&gt;
 *   &lt;filter class="ch.qos.logback.classic.filter.ThresholdFilter"&gt;
 *     &lt;level&gt;TRACE&lt;/level&gt;
 *   &lt;/filter&gt;
 *
 *   &lt;layout class="ch.qos.logback.classic.PatternLayout"&gt;
 *     &lt;pattern&gt;[%date] %.-1level [%thread] [%logger:%line] - %msg%n&lt;/pattern&gt;
 *   &lt;/layout&gt;
 * &lt;/appender&gt;
 *
 * ...
 *
 * &lt;root level="WARN"&gt;
 *   ...
 *   &lt;appender-ref ref="QUARTZDESK_JVM_AGENT"/&gt;
 * &lt;/root&gt;
 * </pre>
 *
 * <strong> This implementation is statically bound to the QuartzDesk JVM Agent API and to the QuartzDesk domain object
 * API. Therefore this appender requires a JVM with an installed QuartzDesk JVM Agent. </strong>
 *
 * @author Jan Moravec
 * @version $Id:$
 * @see IExecutingJobLoggingInterceptor
 */
@Deprecated
public class ClassicLogbackInterceptionAppender<E>
    extends AppenderBase<E>
{
  private static final LoggingEventPriority DEFAULT_LOGGING_INTERCEPTOR_EVENT_PRIORITY =
      LoggingEventPriority.INFO;

  protected Layout<E> layout;

  private IExecutingJobLoggingInterceptor loggingInterceptor;

  /**
   * Creates a new {@link ClassicLogbackInterceptionAppender} instance.
   */
  public ClassicLogbackInterceptionAppender()
  {
    setName( "QuartzDesk-" + ClassicLogbackInterceptionAppender.class.getSimpleName() );
  }


  /**
   * Returns the layout used by this appender.
   *
   * @return the layout used by this appender.
   */
  public Layout<E> getLayout()
  {
    return layout;
  }


  /**
   * Sets the layout to be used by this appender.
   *
   * @param layout the layout.
   */
  public void setLayout( Layout<E> layout )
  {
    this.layout = layout;
  }


  /**
   * Starts this appender.
   */
  @Override
  public void start()
  {
    IAgent agent = Agent.getInstance();

    addInfo( "Starting " + ClassicLogbackInterceptionAppender.class.getName() + " for " + agent );

    loggingInterceptor = agent.getExecutingJobLoggingInterceptor();
    super.start();
  }


  /**
   * Stops this appender.
   */
  @Override
  public void stop()
  {
    loggingInterceptor = null;
    super.stop();
  }


  /**
   * If the QuartzDesk agent's logging interceptor wishes to intercept the specified logging event, then this method
   * formats the log event using the provided pattern layout and passes the result to the logging interceptor.
   *
   * @param eventObject a logging event object.
   */
  @Override
  protected void append( E eventObject )
  {
    Thread currentThread = Thread.currentThread();

    /*
     * The current thread can be a worker thread (a thread created / used by the main job thread).
     */
    Thread jobThread = WorkerThreadLoggingInterceptorRegistry.INSTANCE.getJobThreadForWorkerThread( currentThread );
    if ( jobThread == null )
    {
      jobThread = currentThread;
    }

    if ( loggingInterceptor != null && loggingInterceptor.isIntercepting( jobThread ) )
    {
      LoggingEvent loggingEvent = (LoggingEvent) eventObject;

      com.quartzdesk.agent.api.domain.model.log.LoggingEvent loggingInterceptorEvent =
          new com.quartzdesk.agent.api.domain.model.log.LoggingEvent()
              .withPriority( convertLevel2Priority( loggingEvent.getLevel() ) )
              .withMessage( layout.doLayout( eventObject ) );

      loggingInterceptor.intercept( jobThread, loggingInterceptorEvent );
    }
  }


  /**
   * Returns the {@link LoggingEventPriority} for the specified Logback level.
   *
   * @param level a log4j level.
   * @return the {@link LoggingEventPriority} for the specified Logback level.
   */
  protected LoggingEventPriority convertLevel2Priority( Level level )
  {
    if ( level.equals( Level.TRACE ) )
      return LoggingEventPriority.TRACE;
    else if ( level.equals( Level.DEBUG ) )
      return LoggingEventPriority.DEBUG;
    else if ( level.equals( Level.INFO ) )
      return LoggingEventPriority.INFO;
    else if ( level.equals( Level.WARN ) )
      return LoggingEventPriority.WARN;
    else if ( level.equals( Level.ERROR ) )
      return LoggingEventPriority.ERROR;
    else
    {
      addWarn( "Cannot map logging level: " + level + ". Using default logging interceptor event priority: " +
          DEFAULT_LOGGING_INTERCEPTOR_EVENT_PRIORITY );

      return DEFAULT_LOGGING_INTERCEPTOR_EVENT_PRIORITY;
    }
  }
}
