/*
 * Copyright (c) 2013-2017 QuartzDesk.com. All Rights Reserved.
 * QuartzDesk.com PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.quartzdesk.api.agent.log.logback;

import com.quartzdesk.api.agent.log.LoggingInterceptorWrapper;
import com.quartzdesk.api.agent.log.LoggingInterceptorWrapperException;
import com.quartzdesk.api.agent.log.WorkerThreadLoggingInterceptorRegistry;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.spi.LoggingEvent;
import ch.qos.logback.core.AppenderBase;
import ch.qos.logback.core.Layout;

/**
 * Implementation of a Logback appender that passes the log events to the QuartzDesk JVM agent that intercepts log
 * messages produced by executed jobs.
 * <pre>
 * ==== Example logback.xml ====
 * ...
 *
 * &lt;!--
 *   Appender that passes all received log events to the QuartzDesk agent to intercept
 *   log events produced by executed Quartz jobs.
 * --&gt;
 * &lt;appender name="QUARTZDESK_JVM_AGENT" class="com.quartzdesk.api.agent.log.logback.LogbackInterceptionAppender"&gt;
 *   &lt;filter class="ch.qos.logback.classic.filter.ThresholdFilter"&gt;
 *     &lt;level&gt;TRACE&lt;/level&gt;
 *   &lt;/filter&gt;
 *
 *   &lt;layout class="ch.qos.logback.classic.PatternLayout"&gt;
 *     &lt;pattern&gt;[%date] %.-1level [%thread] [%logger:%line] - %msg%n&lt;/pattern&gt;
 *   &lt;/layout&gt;
 * &lt;/appender&gt;
 *
 * ...
 *
 * &lt;root level="WARN"&gt;
 *   ...
 *   &lt;appender-ref ref="QUARTZDESK_JVM_AGENT"/&gt;
 * &lt;/root&gt;
 * </pre>
 *
 * <strong> This implementation is not statically bound to the QuartzDesk JVM Agent API, nor to the domain object API.
 * Therefore it is safe to use this appender on JVMs running without an installed QuartzDesk JVM Agent. </strong>
 *
 * @author Jan Moravec
 * @version $Id:$
 * @see LoggingInterceptorWrapper
 */
public class LogbackInterceptionAppender<E>
    extends AppenderBase<E>
{
  protected Layout<E> layout;

  private LoggingInterceptorWrapper loggingInterceptor;

  /**
   * Creates a new {@link LogbackInterceptionAppender} instance.
   */
  public LogbackInterceptionAppender()
  {
    setName( "QuartzDesk-" + LogbackInterceptionAppender.class.getSimpleName() );
  }


  /**
   * Returns the layout used by this appender.
   *
   * @return the layout used by this appender.
   */
  public Layout<E> getLayout()
  {
    return layout;
  }


  /**
   * Sets the layout to be used by this appender.
   *
   * @param layout the layout.
   */
  public void setLayout( Layout<E> layout )
  {
    this.layout = layout;
  }


  /**
   * Starts this appender.
   */
  @Override
  public void start()
  {
    addInfo( "Starting " + LogbackInterceptionAppender.class.getName() );

    try
    {
      loggingInterceptor = LoggingInterceptorWrapper.create();
    }
    catch ( LoggingInterceptorWrapperException e )
    {
//      addWarn( "Cannot initialize " + LoggingInterceptorWrapper.class.getName() + ". Appender " + getClass().getName() +
//          " will be disabled.", e );
      addWarn( "Cannot initialize " + LoggingInterceptorWrapper.class.getName() + ". Appender " + getClass().getName() +
          " will be disabled. Cause: " + e.getMessage() );
    }

    super.start();
  }


  /**
   * Stops this appender.
   */
  @Override
  public void stop()
  {
    loggingInterceptor = null;
    super.stop();
  }


  /**
   * If the QuartzDesk agent's logging interceptor wishes to intercept the specified logging event, then this method
   * formats the log event using the provided pattern layout and passes the result to the logging interceptor.
   *
   * @param eventObject a logging event object.
   */
  @Override
  protected void append( E eventObject )
  {
    Thread currentThread = Thread.currentThread();

    /*
     * The current thread can be a worker thread (a thread created / used by the main job thread).
     */
    Thread jobThread = WorkerThreadLoggingInterceptorRegistry.INSTANCE.getJobThreadForWorkerThread( currentThread );
    if ( jobThread == null )
    {
      jobThread = currentThread;
    }

    if ( loggingInterceptor != null && loggingInterceptor.isIntercepting( jobThread ) )
    {
      LoggingEvent loggingEvent = (LoggingEvent) eventObject;

      LoggingInterceptorWrapper.MessagePriority priority = convertLevel2Priority( loggingEvent.getLevel() );
      String message = layout.doLayout( eventObject );

      loggingInterceptor.intercept( jobThread, priority, message );
    }
  }


  /**
   * Returns the {@link LoggingInterceptorWrapper.MessagePriority} for the specified Logback level.
   *
   * @param level a log4j level.
   * @return the {@link LoggingInterceptorWrapper.MessagePriority} for the specified Logback level.
   */
  protected LoggingInterceptorWrapper.MessagePriority convertLevel2Priority( Level level )
  {
    if ( level.equals( Level.TRACE ) )
      return LoggingInterceptorWrapper.MessagePriority.TRACE;
    else if ( level.equals( Level.DEBUG ) )
      return LoggingInterceptorWrapper.MessagePriority.DEBUG;
    else if ( level.equals( Level.INFO ) )
      return LoggingInterceptorWrapper.MessagePriority.INFO;
    else if ( level.equals( Level.WARN ) )
      return LoggingInterceptorWrapper.MessagePriority.WARN;
    else if ( level.equals( Level.ERROR ) )
      return LoggingInterceptorWrapper.MessagePriority.ERROR;
    else
    {
      addWarn( "Cannot map logging level: " + level + ". Using default message priority: " +
          LoggingInterceptorWrapper.DEFAULT_MESSAGE_PRIORITY );

      return LoggingInterceptorWrapper.DEFAULT_MESSAGE_PRIORITY;
    }
  }
}
