
/*
 * Copyright (c) 2013-2017 QuartzDesk.com. All Rights Reserved.
 * QuartzDesk.com PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.quartzdesk.api.common.encoding;

import com.quartzdesk.api.ApiConst;

import java.io.ByteArrayOutputStream;
import java.io.UnsupportedEncodingException;

/**
 * This is a utility class which is used to decode a string in the MIME
 * BASE64 format into a byte array. For further information about the MIME
 * BASE64 encoding, see RFC2045.
 *
 * @author Jan Moravec
 * @version $Id:$
 */
public final class BASE64Decoder
{
  /**
   * Private constructor of a utility class.
   */
  private BASE64Decoder()
  {
  }

  /**
   * Decodes a string in the MIME BASE64 format into a byte array.
   *
   * @param s the string in the MIME BASE64 format.
   * @return the byte array of decoded data.
   * @throws EncodingException when the string s is not in MIME BASE64 format.
   */
  public static byte[] decode( String s )
      throws EncodingException
  {
    // CSOFF: MagicNumber
    if ( s == null )
      return null;

    int first;
    int second;
    int third;
    int fourth;

    try
    {
      ByteArrayOutputStream ba = new ByteArrayOutputStream( s.length() );

      byte[] bytes = s.getBytes( ApiConst.ENCODING_ISO_8859_1 );

      for ( int i = 0; i < bytes.length; )
      {
        // skip empty lines
        if ( bytes[i] == '\n' || bytes[i] == '\r' )
        {
          i++;
          continue;
        }

        first = translate( bytes[i++] );
        second = translate( bytes[i++] );
        third = translate( bytes[i++] );
        fourth = translate( bytes[i++] );

        if ( first != -1 )
        {
          if ( second != -1 )
          {
            // append the first decoded char
            ba.write( (byte) ( ( first << 2 ) | ( ( second & 0x30 ) >>> 4 ) ) );
            if ( third != -1 )
            {
              // append the second decoded char
              ba.write( (byte) ( ( ( second & 0xf ) << 4 ) | ( ( third & 0x3c ) >>> 2 ) ) );
              if ( fourth != -1 )
              {
                // append the third decoded character
                ba.write( (byte) ( ( ( third & 0x03 ) << 6 ) | fourth ) );
              }
              else
              {
                // the fourth character is the padding
                break;
              }
            }
            else
            {
              // the third character is the padding and the fourth should be too
              if ( fourth != -1 )
                throw new EncodingException( "Not MIME BASE64 data format." );
              break;
            }
          }
          else
          {
            // second character in a quad is the padding character
            throw new EncodingException( "Second character in a quad cannot be =." );
          }
        }
        else
        {
          // first character in a quad is the padding character
          throw new EncodingException( "First character in a quad cannot be =." );
        }
      }
      return ba.toByteArray();
    }
    catch ( UnsupportedEncodingException e )
    {
      throw new EncodingException( "Unable to extract input string bytes.", e );
    }
    // CSON: MagicNumber
  }


  /**
   * Returns the position of byte b in the MIME BASE64 alphabet. Values
   * from the range <0,63> are returned. Value of -1 is returned when
   * b is the padding character '='.
   *
   * @param b a byte to translate.
   * @return the position of the specified byte in the MIME BASE64 alphabet.
   * @throws EncodingException if byte b is not from the MIME BASE64 alphabet.
   */
  private static int translate( byte b )
      throws EncodingException
  {
    // CSOFF: MagicNumber
    if ( b >= 'A' && b <= 'Z' ) return b - 'A';
    if ( b >= 'a' && b <= 'z' ) return b - 'a' + 26;
    if ( b >= '0' && b <= '9' ) return b - '0' + 52;
    if ( b == '+' ) return 62;
    if ( b == '/' ) return 63;
    if ( b == '=' ) return -1;
    throw new EncodingException( "Not MIME BASE64 character: 0x" + Character.digit( (char) b, 16 ) );
    // CSON: MagicNumber
  }
}

