/*
 * Copyright (c) 2013-2017 QuartzDesk.com. All Rights Reserved.
 * QuartzDesk.com PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.quartzdesk.api.web.initializer;

import javax.servlet.ServletContext;
import java.io.File;
import java.io.InputStream;
import java.util.Map;
import java.util.Properties;

/**
 * QuartzDesk Web Application initializer interface that is used to obtain important resources required for successful
 * application initialization.
 *
 * <p>By default, the QuartzDesk Web Application uses the {@code com.quartzdesk.core.initializer.WorkDirApplicationInitializer}
 * implementation that uses resources stored in the configured work directory specified in the {@code
 * quartzdesk.work.dir} servlet context init parameter or JVM system property.</p>
 *
 * <p>Integrators can specify a custom initializer implementation by specifying the implementation class in the {@code
 * quartzdesk.initializer.class} servlet context init parameter or JVM system property.</p>
 *
 * <p>All implementing classes must provide a no-arg constructor.</p>
 *
 * @author Jan Moravec
 * @version $Id:$
 */
public interface IApplicationInitializer
{
  /**
   * Invoked by the QuartzDesk Web Application during its initialization. This method is invoked before any of the
   * other methods defined in this class are invoked.
   *
   * @param servletContext a servlet context.
   */
  void initialize( ServletContext servletContext );

  /**
   * Invoked by the QuartzDesk Web Application to obtain the OEM directory where the application looks for overridden
   * resource files (images, i18n resources, JavaScript resources etc.).
   *
   * @return the QuartzDesk Web Application's OEM directory.
   */
  File getOemDir();

  /**
   * Invoked by the QuartzDesk Web Application to obtain the directory where the application looks for customization
   * resources.
   *
   * @return the QuartzDesk Web Application's customization directory.
   */
  File getCustomizationDir();

  /**
   * Invoked by the QuartzDesk Web Application to obtain the temporary directory where the application stores temporary
   * files (e.g. exported job execution logs, charts etc.).
   *
   * @return the QuartzDesk Web Application's temporary directory.
   */
  File getTempDir();

  /**
   * Invoked by the QuartzDesk Web Application to obtain the application license key data.
   *
   * @return the QuartzDesk Web Application's license key data.
   */
  InputStream getLicenseKey();

  /**
   * Invoked by the QuartzDesk Web Application to obtain the logging framework configuration data. <p><strong>The
   * logging configuration data must be compatible with the Logback configuration XML DTD/schema.</strong></p>
   *
   * @return the QuartzDesk Web Application logging framework configuration data.
   */
  InputStream getLoggingConfig();

  /**
   * Invoked by the QuartzDesk Web Application to obtain the application logging context properties. Logging frameworks
   * can use use context properties to expand placeholders (typically various file paths, port numbers, hostnames,
   * etc.)
   * in their configuration to avoid hard-coding. <p> For example:
   * <pre>
   *  &lt;appender name="FILE" class="ext.ch.qos.logback.core.rolling.RollingFileAppender"&gt;
   *    &lt;file&gt;${logs.dir}/quartzdesk.log&lt;/file&gt;
   *    &lt;append&gt;true&lt;/append&gt;
   *    ...
   *  &lt;/appender&gt;
   * </pre>
   * </p>
   *
   * @return the logging context properties, or null if no properties should be set.
   */
  Map<String, String> getLoggingContextProperties();

  /**
   * Invoked by the QuartzDesk Web Application to obtain the non-default application configuration properties
   * (typically
   * stored in the {@code [quartzdesk.work.dir]/quartzdesk.properties} file).
   *
   * @return the non-default QuartzDesk Web Application configuration properties.
   */
  Properties getNonDefaultConfigProperties();
}
