//   The contents of this file are subject to the Mozilla Public License
//   Version 1.1 (the "License"); you may not use this file except in
//   compliance with the License. You may obtain a copy of the License at
//   http://www.mozilla.org/MPL/
//
//   Software distributed under the License is distributed on an "AS IS"
//   basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
//   License for the specific language governing rights and limitations
//   under the License.
//
//   The Original Code is RabbitMQ.
//
//   The Initial Developers of the Original Code are LShift Ltd,
//   Cohesive Financial Technologies LLC, and Rabbit Technologies Ltd.
//
//   Portions created before 22-Nov-2008 00:00:00 GMT by LShift Ltd,
//   Cohesive Financial Technologies LLC, or Rabbit Technologies Ltd
//   are Copyright (C) 2007-2008 LShift Ltd, Cohesive Financial
//   Technologies LLC, and Rabbit Technologies Ltd.
//
//   Portions created by LShift Ltd are Copyright (C) 2007-2010 LShift
//   Ltd. Portions created by Cohesive Financial Technologies LLC are
//   Copyright (C) 2007-2010 Cohesive Financial Technologies
//   LLC. Portions created by Rabbit Technologies Ltd are Copyright
//   (C) 2007-2010 Rabbit Technologies Ltd.
//
//   All Rights Reserved.
//
//   Contributor(s): ______________________________________.
//

package com.rabbitmq.client.impl;

import java.io.DataInputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Date;
import java.util.Map;

/**
 * Parses AMQP wire-protocol {@link Method} arguments from a
 * DataInputStream. Methods on this object are usually called from
 * autogenerated code.
 */
public class MethodArgumentReader
{
    /** The stream we are reading from. */
    private final ValueReader in;
    /** If we are reading one or more bits, holds the current packed collection of bits */
    private int bits;
    /** If we are reading one or more bits, keeps track of which bit position we are reading from */
    private int bit;

    /**
     * Private API - resets the bit group accumulator variables when
     * some non-bit argument value is to be read.
     */
    private void clearBits()
    {
        bits = 0;
        bit = 0x100;
    }

    /**
     * Construct a MethodArgumentReader streaming over the given DataInputStream.
     */
    public MethodArgumentReader(DataInputStream in)
    {
        this.in = new ValueReader(in);
        clearBits();
    }

    /** Public API - reads a short string argument. */
    public final String readShortstr()
        throws IOException
    {
        clearBits();
        return in.readShortstr();
    }

    /** Public API - reads a long string argument. */
    public final LongString readLongstr()
        throws IOException
    {
        clearBits();
        return in.readLongstr();
    }

    /** Public API - reads a short integer argument. */
    public final int readShort()
        throws IOException
    {
        clearBits();
        return in.readShort();
    }

    /** Public API - reads an integer argument. */
    public final int readLong()
        throws IOException
    {
        clearBits();
        return in.readLong();
    }

    /** Public API - reads a long integer argument. */
    public final long readLonglong()
        throws IOException
    {
        clearBits();
        return in.readLonglong();
    }

    /** Public API - reads a bit/boolean argument. */
    public final boolean readBit()
        throws IOException
    {
        if (bit > 0x80) {
            bits = in.readOctet();
            bit = 0x01;
        }

        boolean result = (bits&bit) != 0;
        bit = bit << 1;
        return result;
    }

    /** Public API - reads a table argument. */
    public final Map<String, Object> readTable()
        throws IOException
    {
        clearBits();
        return in.readTable();
    }

    /** Public API - reads an octet argument. */
    public final int readOctet()
        throws IOException
    {
        clearBits();
        return in.readOctet();
    }

    /** Public API - reads an timestamp argument. */
    public final Date readTimestamp()
        throws IOException
    {
        clearBits();
        return in.readTimestamp();
    }
}
