/*
 * Decompiled with CFR 0.152.
 */
package com.rabbitmq.client.amqp.oauth2;

import com.rabbitmq.client.amqp.oauth2.CredentialsManager;
import com.rabbitmq.client.amqp.oauth2.Token;
import com.rabbitmq.client.amqp.oauth2.TokenRequester;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.time.Duration;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class TokenCredentialsManager
implements CredentialsManager {
    public static final Function<Instant, Duration> DEFAULT_REFRESH_DELAY_STRATEGY = TokenCredentialsManager.ratioRefreshDelayStrategy(0.8f);
    private static final Logger LOGGER = LoggerFactory.getLogger(TokenCredentialsManager.class);
    private final TokenRequester requester;
    private final ScheduledExecutorService scheduledExecutorService;
    private volatile Token token;
    private final Lock lock = new ReentrantLock();
    private final Map<Long, RegistrationImpl> registrations = new ConcurrentHashMap<Long, RegistrationImpl>();
    private final AtomicLong registrationSequence = new AtomicLong(0L);
    private final AtomicBoolean schedulingRefresh = new AtomicBoolean(false);
    private final Function<Instant, Duration> refreshDelayStrategy;
    private volatile ScheduledFuture<?> refreshTask;

    public TokenCredentialsManager(TokenRequester requester, ScheduledExecutorService scheduledExecutorService, Function<Instant, Duration> refreshDelayStrategy) {
        this.requester = requester;
        this.scheduledExecutorService = scheduledExecutorService;
        this.refreshDelayStrategy = refreshDelayStrategy;
    }

    private void lock() {
        this.lock.lock();
    }

    private void unlock() {
        this.lock.unlock();
    }

    private boolean expiresSoon(Token ignores) {
        return false;
    }

    private Token getToken() {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Requesting new token ({})...", (Object)TokenCredentialsManager.registrationSummary(this.registrations.values()));
        }
        long start = 0L;
        if (LOGGER.isDebugEnabled()) {
            start = System.nanoTime();
        }
        Token token = this.requester.request();
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Got new token in {} ms, token expires on {} ({})", new Object[]{Duration.ofNanos(System.nanoTime() - start), TokenCredentialsManager.format(token.expirationTime()), TokenCredentialsManager.registrationSummary(this.registrations.values())});
        }
        return token;
    }

    @Override
    public CredentialsManager.Registration register(String name, CredentialsManager.AuthenticationCallback updateCallback) {
        Long id = this.registrationSequence.getAndIncrement();
        name = name == null ? id.toString() : name;
        RegistrationImpl registration = new RegistrationImpl(id, name, updateCallback);
        this.registrations.put(id, registration);
        return registration;
    }

    private void updateRegistrations(Token t) {
        this.scheduledExecutorService.execute(() -> {
            LOGGER.debug("Updating {} registration(s)", (Object)this.registrations.size());
            int refreshedCount = 0;
            for (RegistrationImpl registration : this.registrations.values()) {
                if (t.equals(this.token)) {
                    if (!registration.isClosed() && !registration.hasSameToken(t)) {
                        try {
                            registration.updateCallback().authenticate("", this.token.value());
                        }
                        catch (Exception e) {
                            LOGGER.warn("Error while updating token for registration '{}': {}", (Object)registration.name(), (Object)e.getMessage());
                        }
                        registration.registrationToken = this.token;
                        ++refreshedCount;
                        continue;
                    }
                    if (!LOGGER.isDebugEnabled()) continue;
                    LOGGER.debug("Not updating registration {} (closed or already has the new token)", (Object)registration.name());
                    continue;
                }
                if (!LOGGER.isDebugEnabled()) continue;
                LOGGER.debug("Not updating registration {} (the token has changed)", (Object)registration.name());
            }
            LOGGER.debug("Updated {} registration(s)", (Object)refreshedCount);
        });
    }

    private void token(Token t) {
        this.lock();
        try {
            if (!t.equals(this.token)) {
                this.token = t;
                this.scheduleTokenRefresh(t);
            }
        }
        finally {
            this.unlock();
        }
    }

    private void scheduleTokenRefresh(Token t) {
        if (this.schedulingRefresh.compareAndSet(false, true)) {
            if (this.refreshTask != null) {
                this.refreshTask.cancel(false);
            }
            Duration delay = this.refreshDelayStrategy.apply(t.expirationTime());
            if (!this.registrations.isEmpty()) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Scheduling token retrieval in {} ({})", (Object)delay, (Object)TokenCredentialsManager.registrationSummary(this.registrations.values()));
                }
                this.refreshTask = this.scheduledExecutorService.schedule(() -> {
                    Token previousToken = this.token;
                    this.lock();
                    try {
                        if (this.token.equals(previousToken)) {
                            Token newToken = this.getToken();
                            this.token(newToken);
                            this.updateRegistrations(newToken);
                        }
                    }
                    finally {
                        this.unlock();
                    }
                }, delay.toMillis(), TimeUnit.MILLISECONDS);
            } else {
                this.refreshTask = null;
            }
            this.schedulingRefresh.set(false);
        }
    }

    private static String format(Instant instant) {
        return DateTimeFormatter.ISO_INSTANT.format(instant);
    }

    public static Function<Instant, Duration> ratioRefreshDelayStrategy(float ratio) {
        return new RatioRefreshDelayStrategy(ratio);
    }

    private static String registrationSummary(Collection<? extends CredentialsManager.Registration> registrations) {
        return registrations.stream().map(Object::toString).collect(Collectors.joining(", "));
    }

    private static class RatioRefreshDelayStrategy
    implements Function<Instant, Duration> {
        private final float ratio;

        @SuppressFBWarnings(value={"CT_CONSTRUCTOR_THROW"})
        private RatioRefreshDelayStrategy(float ratio) {
            if (ratio < 0.0f || ratio > 1.0f) {
                throw new IllegalArgumentException("Ratio should be > 0 and <= 1: " + ratio);
            }
            this.ratio = ratio;
        }

        @Override
        public Duration apply(Instant expirationTime) {
            Duration expiresIn = Duration.between(Instant.now(), expirationTime);
            Duration delay = expiresIn.isZero() || expiresIn.isNegative() ? Duration.ofSeconds(1L) : Duration.ofMillis((long)((float)expiresIn.toMillis() * this.ratio));
            return delay;
        }
    }

    private final class RegistrationImpl
    implements CredentialsManager.Registration {
        private final Long id;
        private final String name;
        private final CredentialsManager.AuthenticationCallback updateCallback;
        private volatile Token registrationToken;
        private final AtomicBoolean closed = new AtomicBoolean(false);

        private RegistrationImpl(Long id, String name, CredentialsManager.AuthenticationCallback updateCallback) {
            this.id = id;
            this.name = name;
            this.updateCallback = updateCallback;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void connect(CredentialsManager.AuthenticationCallback callback) {
            Token tokenToUse;
            boolean shouldRefresh = false;
            TokenCredentialsManager.this.lock();
            try {
                Token globalToken = TokenCredentialsManager.this.token;
                if (globalToken == null) {
                    TokenCredentialsManager.this.token(TokenCredentialsManager.this.getToken());
                } else if (TokenCredentialsManager.this.expiresSoon(globalToken)) {
                    shouldRefresh = true;
                    TokenCredentialsManager.this.token(TokenCredentialsManager.this.getToken());
                }
                if (!TokenCredentialsManager.this.token.equals(this.registrationToken)) {
                    this.registrationToken = TokenCredentialsManager.this.token;
                }
                tokenToUse = this.registrationToken;
                if (TokenCredentialsManager.this.refreshTask == null) {
                    TokenCredentialsManager.this.scheduleTokenRefresh(tokenToUse);
                }
            }
            finally {
                TokenCredentialsManager.this.unlock();
            }
            callback.authenticate("", tokenToUse.value());
            if (shouldRefresh) {
                TokenCredentialsManager.this.updateRegistrations(tokenToUse);
            }
        }

        @Override
        public void close() {
            if (this.closed.compareAndSet(false, true)) {
                TokenCredentialsManager.this.registrations.remove(this.id);
                ScheduledFuture<?> task = TokenCredentialsManager.this.refreshTask;
                if (TokenCredentialsManager.this.registrations.isEmpty() && task != null) {
                    TokenCredentialsManager.this.lock();
                    try {
                        if (TokenCredentialsManager.this.refreshTask != null) {
                            TokenCredentialsManager.this.refreshTask.cancel(false);
                        }
                    }
                    finally {
                        TokenCredentialsManager.this.unlock();
                    }
                }
            }
        }

        private CredentialsManager.AuthenticationCallback updateCallback() {
            return this.updateCallback;
        }

        private String name() {
            return this.name;
        }

        private boolean hasSameToken(Token t) {
            return t.equals(this.registrationToken);
        }

        private boolean isClosed() {
            return this.closed.get();
        }

        public boolean equals(Object o) {
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            RegistrationImpl that = (RegistrationImpl)o;
            return Objects.equals(this.id, that.id);
        }

        public int hashCode() {
            return Objects.hashCode(this.id);
        }

        public String toString() {
            return this.name();
        }
    }
}

