/*
 * Copyright 2015-2017 Reactific Software LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.reactific.helpers

import java.time.format.DateTimeFormatter
import java.time.{Instant, Period}

import scala.concurrent.duration._

/** A Suite of utilities for manipulating Java 8 time classes
 * Provides mostly standard ways of converting times and durations into
 * strings.
 */
object DateTimeHelpers {

  def dateStr(millis: Long): String = dateStr(Instant.ofEpochMilli(millis))
  def dateStr(dt: Instant): String = DateTimeFormatter.ISO_INSTANT.format(dt)

  implicit class StringBuilderPimps(bldr: StringBuilder) {

    def appendSuffix(value: Long, singular: String): StringBuilder = {
      bldr
        .append(value)
        .append(" ")
        .append { if (value == 1) singular else Pluralizer.pluralize(singular) }
        .append(", ")
    }
  }

  def makeReadable(period: Period): String = {
    val builder = new StringBuilder()
    if (period.getYears > 0) {
      builder.appendSuffix(period.getYears.toLong, "year")
    }
    if (period.getMonths > 0) {
      builder.appendSuffix(period.getMonths.toLong, "month")
    }
    if (period.getDays > 0) {
      builder.appendSuffix(period.getDays.toLong, "day")
    }
    builder.toString().dropRight(2)
  }

  val millisPerSecond: Long = 1000L
  val millisPerMinute: Long = 60 * millisPerSecond
  val millisPerHour: Long = 60 * millisPerMinute
  val millisPerDay: Long = 24 * millisPerHour
  val millisPerYear: Long = millisPerDay * 365

  def makeReadable(duration: java.time.Duration): String = {
    val builder = new StringBuilder()

    var duration_in_millis: Long = // scalastyle:ignore
      duration.toMillis

    val num_years = duration_in_millis / millisPerYear
    if (num_years > 0) {
      builder.appendSuffix(num_years, "year")
      duration_in_millis -= num_years * millisPerYear
    }

    val num_days = duration_in_millis / millisPerDay
    if (num_days > 0) {
      builder.appendSuffix(num_days, "day")
      duration_in_millis -= num_days * millisPerDay
    }

    val num_hours = duration_in_millis / millisPerHour
    if (num_hours > 0) {
      builder.appendSuffix(num_hours, "hour")
      duration_in_millis -= num_hours * millisPerHour
    }

    val num_minutes = duration_in_millis / millisPerMinute
    if (num_minutes > 0) {
      builder.appendSuffix(num_minutes, "minute")
      duration_in_millis -= num_minutes * millisPerMinute
    }

    if (duration_in_millis >= millisPerSecond) {
      val num_seconds: Double = duration_in_millis.toDouble / millisPerSecond
      builder
        .append(f"$num_seconds%2.3f")
        .append(if (num_seconds == 1.0) " second, " else " seconds, ")
    } else {
      builder.append(duration_in_millis).append(" ms, ")
    }
    builder.toString().dropRight(2)
  }

  def makeReadable(duration: Duration): String = {
    makeReadable(java.time.Duration.ofNanos(duration.toNanos))
  }

}
