/*
 * Copyright 2015-2017 Reactific Software LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.reactific.helpers

import java.util.concurrent.{TimeUnit, TimeoutException}

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{Await, Future} // scalastyle:ignore
import scala.util.{Failure, Success, Try}

/** Utilities used throughout Scrupla. Should probably be an aspect, but
 * we just mix this in as needed. Helps with logging, throwing exceptions
 * and dealing with futures.
 */
trait FutureHelper extends LoggingHelper with ThrowingHelper {

  implicit val timeout: FiniteDuration = FiniteDuration(1, TimeUnit.MINUTES)

  /** Handle Exceptions From Asynchronous Results
   * This awaits a future operation that returns HTML and converts any
   * exception that arises during the future operation or waiting into
   * HTML as well. So, this is guaranteed to never let an exception escape
   * and to always return Html.
   *
   * @param future The Future[Html] to wait for
   * @param duration How long to wait, at most
   * @param opName The name of the operation being waited for, to assist
   *               with error messages
   * @return A blob of Html containing either the intended result or
   *         an error message.
   */
  def await[X](
    future: ⇒ Future[X],
    duration: FiniteDuration,
    opName: String
  ): Try[X] = {
    try {
      Success(Await.result(future, duration)) // scalastyle:ignore
    } catch {
      case ix: InterruptedException =>
        // - if the current thread is interrupted while waiting
        Failure(mkThrowable(s"Operation '$opName' was interrupted: ", Some(ix)))
      case tx: TimeoutException =>
        // if after waiting for the specified time awaitable is still not ready
        Failure(
          mkThrowable(
            s"Operation '$opName' timed out after $duration",
            Some(tx)
          )
        )
      case xcptn: Throwable =>
        Failure(
          mkThrowable(
            s"Operation '$opName'interrupted by unrecognized exception:",
            Some(xcptn)
          )
        )
    }
  }

  def await[X](
    future: ⇒ Future[X],
    opName: String
  )(implicit to: FiniteDuration = timeout
  ): Try[X] = {
    await[X](future, to, opName)
  }
}
