/*
 * Copyright 2015-2017 Reactific Software LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.reactific.helpers

import java.util.concurrent.ConcurrentHashMap

import scala.collection.JavaConverters._
import scala.language.postfixOps

/**
 * This trait makes a simple cache from a ConcurrentHashMap accompanied by
 * a getOrElse function to obtain or create values
 *
 * @tparam K    type of the keys
 * @tparam V    type of the values
 */
class MemoryCache[K, V] {

  private val cache = new ConcurrentHashMap[K, V] asScala

  def get(key: K): Option[V] = cache.get(key)

  def getOrElse(key: K)(value: ⇒ V): V = { cache.getOrElseUpdate(key, value) }

  def size: Int = cache.size

  def clear(): Unit = cache.clear()

  def replace(k: K, newV: V): Option[V] = { cache.replace(k, newV) }

  def remove(k: K): Option[V] = cache.remove(k)

}

object MemoryCache {
  def apply[K, V](): MemoryCache[K, V] = new MemoryCache[K, V]
}
