/*
 * Copyright 2015-2017 Reactific Software LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.reactific

/** Reactific Helpers Package.
 * This package contains an assortment of utility classes that defy
 * other categorization. Helpers are typically standalone (they depend on
 * nothing) and are utilized in more than one other package.
 */
package object helpers {

  implicit class ThrowableHacks(t: Throwable) {

    def getLogMessage: String = {

      def causeMsg(x: Throwable): String = {
        Option(x.getCause) match {
          case None ⇒
            ""
          case Some(cause: Throwable) if cause != x ⇒
            s"\ncaused by ${cause.getClass.getName}: " +
              cause.getMessage + causeMsg(cause)
        }
      }

      def where(x: Throwable): String = {

        def elemMsg(ste: StackTraceElement): String = {
          val method = s"${ste.getClassName}.${ste.getMethodName}"
          s"\n$method@${ste.getFileName}:${ste.getLineNumber}"
        }

        val st = x.getStackTrace
        val first = {
          st.find { ste ⇒
            ste.getClassName.startsWith("com.j9tech")
          } match {
            case Some(ste) ⇒
              elemMsg(ste)
            case None ⇒
              st.headOption match {
                case Some(ste) ⇒
                  elemMsg(ste)
                case None ⇒
                  ""
              }
          }
        }
        val last = {
          st.reverse.find { ste ⇒
            ste.getClassName.startsWith("com.j9tech")
          } match {
            case Some(ste) ⇒
              elemMsg(ste)
            case None ⇒
              st.headOption match {
                case Some(ste) ⇒
                  elemMsg(ste)
                case None ⇒
                  ""
              }
          }
        }
        s"$first$last"
      }

      s"${t.getClass.getName}: ${t.getMessage}${causeMsg(t)}${where(t)}"
    }
  }
}
