package com.redfin.fuzzy;

import java.util.Random;

/**
 * Describes a specific variant of a value generated by a {@link Case}. When the fuzzy engine is determining how many
 * test cases to execute, it treats subcases as
 * <a href="https://en.wikipedia.org/wiki/Equivalence_partitioning">equivalency classes</a>: although an individual
 * subcase's return value can vary randomly, it assumes that all values it can return are effectively equal.
 */
public interface Subcase<T> {

	/**
	 * Instructs the subcase to generate and return a new value.
	 *
	 * <p>Subcases are not required to return exactly the same value for subsequent calls to {@code generate}; they may
	 * randomly adjust their output on a per-call basis. However, subcases should adhere to the expectation that they
	 * represent an <em>equivalency class</em> of output: although individual return values may differ, each value they
	 * produce is effectively the same from a testing perspective.
	 * </p>
	 * <p>If the subcase does chose to produce randomized values, it should do so deterministically: two calls provided
	 * with a random number generator with the same seed should produce exactly the same output. In general, this
	 * condition should be satisfied if the test uses the provided {@linkplain Random random number generator} as its
	 * only source of randomness.</p>
	 */
	T generate(Random random);

	/**
	 * Describes a given output produced by this subcase, for use in test failure reports.
	 *
	 * <p>The default implementation of this method attempts to provide helpful descriptions of {@code null}, common
	 * primitives, strings, arrays, and collections. It defers to {@link Object#toString()} for all other descriptions.
	 * </p>
	 *
	 * @param sink the string builder to which the description of the value should be appended.
	 * @param value the value (as returned by this subcase's {@link #generate(Random) generate} method) that should be
	 *        described.
	 */
	default void describeTo(StringBuilder sink, T value) {
		FuzzyPreconditions.checkNotNull(sink);
		FuzzyUtil.inspectTo(sink, value);
	}

}
