/*
 * Copyright 2024 REGnosys
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.regnosys.rosetta.generator.external;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.function.Consumer;

import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;

import com.regnosys.rosetta.rosetta.RosettaModel;
import com.rosetta.util.DemandableLock;


/**
 * Abstract base class with some skeleton implementation for external generators.
 * 
 * Implementors should subclass this class and provide their own concrete implementations of {@link #generate()}
 *  
 */
public abstract class AbstractExternalGenerator implements ExternalGenerator {

	private final String name;
	
	public AbstractExternalGenerator(String name) {
		this.name = name;
	}

	@Override
	public void beforeAllGenerate(ResourceSet set, Collection<? extends RosettaModel> models, String version,
			Consumer<Map<String, ? extends CharSequence>> processResults, DemandableLock generateLock) {
		Map<String, ? extends CharSequence> generate = beforeAllGenerate(set, models, version);
		processResults.accept(generate);
	}

	@Override
	public void beforeGenerate(Resource resource, RosettaModel model, String version,
			Consumer<Map<String, ? extends CharSequence>> processResults, DemandableLock generateLock) {
		Map<String, ? extends CharSequence> generate = beforeGenerate(resource, model, version);
		processResults.accept(generate);
	}

	@Override
	public void generate(Resource resource, RosettaModel model, String version,
			Consumer<Map<String, ? extends CharSequence>> processResults, DemandableLock generateLock) {
		Map<String, ? extends CharSequence> generate = generate(resource, model, version);
		processResults.accept(generate);
	}

	@Override
	public void afterGenerate(Resource resource, RosettaModel model, String version,
			Consumer<Map<String, ? extends CharSequence>> processResults, DemandableLock generateLock) {
		Map<String, ? extends CharSequence> generate = afterGenerate(resource, model, version);
		processResults.accept(generate);
	}

	@Override
	public void afterAllGenerate(ResourceSet set, Collection<? extends RosettaModel> models, String version,
			Consumer<Map<String, ? extends CharSequence>> processResults, DemandableLock generateLock) {
		Map<String, ? extends CharSequence> generate = afterAllGenerate(set, models, version);
		processResults.accept(generate);
	}

	@Override
	public ExternalOutputConfiguration getOutputConfiguration() {
		return new ExternalOutputConfiguration(name, "Code generation configuration");
	}
	
	public Map<String, ? extends CharSequence> beforeAllGenerate(ResourceSet set, Collection<? extends RosettaModel> models, String version) {
		return Collections.emptyMap();
	}
	
	public Map<String, ? extends CharSequence> beforeGenerate(Resource resource, RosettaModel model, String version) {
		return Collections.emptyMap();
	}
	
	/**
	 * Returns a map of {filename -> source code} for all the classes generated by the rosetta source
	 * 
	 * @param	resource  the Rosetta resource containing Rosetta code.
	 * @param 	model  the parsed Rosetta model.
	 * @param 	version  the version of Rosetta files.
	 * @return
	 */
	public abstract Map<String, ? extends CharSequence> generate(Resource resource, RosettaModel model, String version);
	
	public Map<String, ? extends CharSequence> afterGenerate(Resource resource, RosettaModel model, String version) {
		return Collections.emptyMap();
	}
	
	public Map<String, ? extends CharSequence> afterAllGenerate(ResourceSet set, Collection<? extends RosettaModel> models, String version) {
		return Collections.emptyMap();
	}
}
