package com.regnosys.rosetta.generator.java.expression;

import com.google.common.base.Objects;
import com.regnosys.rosetta.generator.GeneratedIdentifier;
import com.regnosys.rosetta.generator.java.JavaScope;
import com.regnosys.rosetta.generator.java.statement.builder.JavaConditionalExpression;
import com.regnosys.rosetta.generator.java.statement.builder.JavaExpression;
import com.regnosys.rosetta.generator.java.statement.builder.JavaStatementBuilder;
import com.regnosys.rosetta.generator.java.statement.builder.JavaVariable;
import com.regnosys.rosetta.generator.java.types.JavaTypeUtil;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperListOfLists;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.util.types.JavaParameterizedType;
import com.rosetta.util.types.JavaPrimitiveType;
import com.rosetta.util.types.JavaReferenceType;
import com.rosetta.util.types.JavaType;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.inject.Inject;
import org.eclipse.xtend2.lib.StringConcatenationClient;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * This service is responsible for coercing an expression from its actual Java type to an `expected` Java type.
 * 
 * Both coercions of item types and of wrapper types are supported. Examples of supported coercions:
 * - `Long` to `BigDecimal`
 * - `BigInteger` to `Integer` (will throw a runtime exception if conversion looses information)
 * - `Boolean` to `ComparisonResult`
 * - `ComparisonResult` to `Boolean`
 * - `String` to `MapperS<String>`
 * - `MapperC<Long>` to `List<BigInteger>`
 * - `MapperC<Integer>` to `BigDecimal`
 * - `MapperC<Boolean>` to `ComparisonResult`
 * - `Void` to `LocalDate`
 * - `Void` to `MapperC<LocalDate>`
 * Item to item coercions and item to wrapper coercions are performed null-safe.
 * 
 * This service is auto-boxing aware. If the expected type is a wrapper class of a primitive type
 * and the input expression is of a primitive type as well, the result will be of a primitive type.
 * 
 * Terminology:
 * - a "wrapper type" refers to any of the following classes:
 *   - `MapperS`
 *   - `MapperC`
 *   - `ComparisonResult`
 *   - `Mapper` (the general interface of the above classes)
 *   - `MapperListOfLists`
 *   - `List`
 * - an "item type" refers to the type of the items of a wrapper type, or to the type itself if it does not refer to a wrapper type.
 * For a precise definition, see the methods `JavaTypeUtil#isWrapper` and `JavaTypeUtil#getItemType`.
 */
@SuppressWarnings("all")
public class TypeCoercionService {
  @Inject
  @Extension
  private JavaTypeUtil typeUtil;

  public JavaStatementBuilder addCoercions(final JavaStatementBuilder expr, final JavaType expected, final JavaScope scope) {
    JavaStatementBuilder _xblockexpression = null;
    {
      final JavaType actual = expr.getExpressionType();
      if ((Objects.equal(this.typeUtil.getItemType(actual), JavaReferenceType.NULL_TYPE) || this.typeUtil.isVoid(this.typeUtil.getItemType(actual)))) {
        return this.empty(expected);
      }
      boolean _equals = Objects.equal(actual, expected);
      if (_equals) {
        return expr;
      }
      boolean _equals_1 = Objects.equal(actual, JavaPrimitiveType.VOID);
      if (_equals_1) {
        throw new IllegalArgumentException("Cannot coerce from primitive type `void`.");
      }
      final Function<JavaExpression, JavaStatementBuilder> _function = (JavaExpression it) -> {
        return this.addCoercions(it, expected, scope);
      };
      _xblockexpression = expr.mapExpression(_function);
    }
    return _xblockexpression;
  }

  public JavaStatementBuilder addCoercions(final JavaExpression expr, final JavaType expected, final JavaScope scope) {
    JavaStatementBuilder _xblockexpression = null;
    {
      final JavaType actual = expr.getExpressionType();
      if ((Objects.equal(this.typeUtil.getItemType(actual), JavaReferenceType.NULL_TYPE) || this.typeUtil.isVoid(this.typeUtil.getItemType(actual)))) {
        return this.empty(expected);
      }
      boolean _equals = Objects.equal(actual, expected);
      if (_equals) {
        return expr;
      }
      boolean _equals_1 = Objects.equal(actual, JavaPrimitiveType.VOID);
      if (_equals_1) {
        throw new IllegalArgumentException("Cannot coerce from primitive type `void`.");
      }
      JavaStatementBuilder _xifexpression = null;
      if ((this.typeUtil.isWrapper(actual) && this.typeUtil.isWrapper(expected))) {
        _xifexpression = this.wrapperToWrapper(expr, expected, scope);
      } else {
        JavaStatementBuilder _xifexpression_1 = null;
        boolean _isWrapper = this.typeUtil.isWrapper(actual);
        if (_isWrapper) {
          _xifexpression_1 = this.wrapperToItem(expr, expected, scope);
        } else {
          JavaStatementBuilder _xifexpression_2 = null;
          boolean _isWrapper_1 = this.typeUtil.isWrapper(expected);
          if (_isWrapper_1) {
            _xifexpression_2 = this.itemToWrapper(expr, expected, scope);
          } else {
            _xifexpression_2 = this.itemToItem(expr, expected, scope);
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  private JavaStatementBuilder itemToItem(final JavaExpression expr, final JavaType expected, final JavaScope scope) {
    JavaStatementBuilder _xblockexpression = null;
    {
      final JavaType actual = expr.getExpressionType();
      final Function<Function<JavaExpression, JavaExpression>, JavaStatementBuilder> _function = (Function<JavaExpression, JavaExpression> itemConversion) -> {
        return this.convertNullSafe(expr, itemConversion, expected, scope);
      };
      _xblockexpression = this.getItemConversion(actual, expected).<JavaStatementBuilder>map(_function).orElse(expr);
    }
    return _xblockexpression;
  }

  private JavaStatementBuilder itemToWrapper(final JavaExpression expr, final JavaType expected, final JavaScope scope) {
    JavaStatementBuilder _xblockexpression = null;
    {
      final JavaType actual = expr.getExpressionType();
      final JavaType expectedItemType = this.typeUtil.getItemType(expected);
      final Function<JavaExpression, JavaExpression> wrapConversion = this.getWrapConversion(expected);
      JavaStatementBuilder _xifexpression = null;
      boolean _extendsMapper = this.typeUtil.extendsMapper(expected);
      if (_extendsMapper) {
        final Function<Function<JavaExpression, JavaExpression>, JavaStatementBuilder> _function = (Function<JavaExpression, JavaExpression> itemConversion) -> {
          return this.convertNullSafe(expr, 
            wrapConversion.<JavaExpression>compose(itemConversion), expected, scope);
        };
        _xifexpression = this.getItemConversion(actual, expectedItemType).<JavaStatementBuilder>map(_function).orElse(expr.mapExpression(wrapConversion));
      } else {
        JavaStatementBuilder _xblockexpression_1 = null;
        {
          final Function<Function<JavaExpression, JavaExpression>, Function<JavaExpression, JavaExpression>> _function_1 = (Function<JavaExpression, JavaExpression> itemConversion) -> {
            return wrapConversion.<JavaExpression>compose(itemConversion);
          };
          final Function<JavaExpression, JavaExpression> totalConversion = this.getItemConversion(actual, expectedItemType).<Function<JavaExpression, JavaExpression>>map(_function_1).orElse(wrapConversion);
          _xblockexpression_1 = this.convertNullSafe(expr, totalConversion, expected, scope);
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  private JavaStatementBuilder wrapperToItem(final JavaExpression expr, final JavaType expected, final JavaScope scope) {
    JavaStatementBuilder _xblockexpression = null;
    {
      final JavaType actual = expr.getExpressionType();
      if ((this.typeUtil.extendsMapper(actual) && (expected instanceof JavaPrimitiveType))) {
        final Function<JavaExpression, JavaStatementBuilder> _function = (JavaExpression it) -> {
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              _builder.append(it);
              _builder.append(".getOrDefault(");
              JavaExpression _empty = TypeCoercionService.this.empty(expected);
              _builder.append(_empty);
              _builder.append(")");
            }
          };
          return JavaExpression.from(_client, expected);
        };
        return expr.mapExpression(_function);
      }
      final JavaExpression unwrappedExpr = this.getUnwrapConversion(actual).apply(expr);
      _xblockexpression = this.itemToItem(unwrappedExpr, expected, scope);
    }
    return _xblockexpression;
  }

  private JavaExpression wrapperToWrapper(final JavaExpression expr, final JavaType expected, final JavaScope scope) {
    final JavaType actual = expr.getExpressionType();
    final JavaType expectedItemType = this.typeUtil.getItemType(expected);
    final Optional<Function<JavaExpression, JavaExpression>> optionalWrappedItemConversion = this.getWrappedItemConversion(actual, expectedItemType, scope);
    JavaType _xifexpression = null;
    boolean _isEmpty = optionalWrappedItemConversion.isEmpty();
    if (_isEmpty) {
      _xifexpression = actual;
    } else {
      _xifexpression = this.typeUtil.changeItemType(actual, expectedItemType);
    }
    final Optional<Function<JavaExpression, JavaExpression>> optionalWrapperConversion = this.getWrapperConversion(_xifexpression, expected);
    if ((optionalWrappedItemConversion.isEmpty() && optionalWrapperConversion.isEmpty())) {
      return expr;
    }
    Function<JavaExpression, JavaExpression> _xifexpression_1 = null;
    boolean _isEmpty_1 = optionalWrappedItemConversion.isEmpty();
    if (_isEmpty_1) {
      _xifexpression_1 = optionalWrapperConversion.orElseThrow();
    } else {
      Function<JavaExpression, JavaExpression> _xifexpression_2 = null;
      boolean _isEmpty_2 = optionalWrapperConversion.isEmpty();
      if (_isEmpty_2) {
        _xifexpression_2 = optionalWrappedItemConversion.orElseThrow();
      } else {
        _xifexpression_2 = optionalWrapperConversion.orElseThrow().<JavaExpression>compose(optionalWrappedItemConversion.orElseThrow());
      }
      _xifexpression_1 = _xifexpression_2;
    }
    final Function<JavaExpression, JavaExpression> totalConversion = _xifexpression_1;
    return totalConversion.apply(expr);
  }

  private Optional<Function<JavaExpression, JavaExpression>> getItemConversion(final JavaType actual, final JavaType expected) {
    boolean _equals = Objects.equal(actual, expected);
    if (_equals) {
      return Optional.<Function<JavaExpression, JavaExpression>>empty();
    }
    JavaReferenceType _referenceType = actual.toReferenceType();
    JavaReferenceType _referenceType_1 = expected.toReferenceType();
    boolean _equals_1 = Objects.equal(_referenceType, _referenceType_1);
    if (_equals_1) {
      final Function<JavaExpression, JavaExpression> _function = (JavaExpression it) -> {
        return it;
      };
      return Optional.<Function<JavaExpression, JavaExpression>>of(_function);
    } else {
      if ((this.typeUtil.extendsNumber(actual.toReferenceType()) && this.typeUtil.extendsNumber(expected.toReferenceType()))) {
        final Function<JavaExpression, JavaExpression> _function_1 = (JavaExpression it) -> {
          return this.getNumberConversionExpression(it, expected);
        };
        return Optional.<Function<JavaExpression, JavaExpression>>of(_function_1);
      }
    }
    return Optional.<Function<JavaExpression, JavaExpression>>empty();
  }

  private Function<JavaExpression, JavaExpression> getWrapConversion(final JavaType wrapperType) {
    Function<JavaExpression, JavaExpression> _xifexpression = null;
    boolean _isList = this.typeUtil.isList(wrapperType);
    if (_isList) {
      final Function<JavaExpression, JavaExpression> _function = (JavaExpression it) -> {
        return this.getItemToListConversionExpression(it);
      };
      _xifexpression = _function;
    } else {
      Function<JavaExpression, JavaExpression> _xifexpression_1 = null;
      if ((this.typeUtil.isMapperS(wrapperType) || this.typeUtil.isMapper(wrapperType))) {
        final Function<JavaExpression, JavaExpression> _function_1 = (JavaExpression it) -> {
          return this.getItemToMapperSConversionExpression(it);
        };
        _xifexpression_1 = _function_1;
      } else {
        Function<JavaExpression, JavaExpression> _xifexpression_2 = null;
        boolean _isMapperC = this.typeUtil.isMapperC(wrapperType);
        if (_isMapperC) {
          final Function<JavaExpression, JavaExpression> _function_2 = (JavaExpression it) -> {
            return this.getItemToMapperCConversionExpression(it);
          };
          _xifexpression_2 = _function_2;
        } else {
          Function<JavaExpression, JavaExpression> _xifexpression_3 = null;
          boolean _isComparisonResult = this.typeUtil.isComparisonResult(wrapperType);
          if (_isComparisonResult) {
            final Function<JavaExpression, JavaExpression> _function_3 = (JavaExpression it) -> {
              return this.getItemToComparisonResultConversionExpression(it);
            };
            _xifexpression_3 = _function_3;
          } else {
            throw this.unexpectedWrapperException(wrapperType);
          }
          _xifexpression_2 = _xifexpression_3;
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }

  private Function<JavaExpression, JavaExpression> getUnwrapConversion(final JavaType wrapperType) {
    Function<JavaExpression, JavaExpression> _xifexpression = null;
    boolean _isList = this.typeUtil.isList(wrapperType);
    if (_isList) {
      final Function<JavaExpression, JavaExpression> _function = (JavaExpression it) -> {
        return this.getListToItemConversionExpression(it);
      };
      _xifexpression = _function;
    } else {
      Function<JavaExpression, JavaExpression> _xifexpression_1 = null;
      boolean _extendsMapper = this.typeUtil.extendsMapper(wrapperType);
      if (_extendsMapper) {
        final Function<JavaExpression, JavaExpression> _function_1 = (JavaExpression it) -> {
          return this.getMapperToItemConversionExpression(it);
        };
        _xifexpression_1 = _function_1;
      } else {
        throw this.unexpectedWrapperException(wrapperType);
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }

  private Optional<Function<JavaExpression, JavaExpression>> getWrappedItemConversion(final JavaType actual, final JavaType expectedItemType, final JavaScope scope) {
    Optional<Function<JavaExpression, JavaExpression>> _xblockexpression = null;
    {
      final JavaType actualItemType = this.typeUtil.getItemType(actual);
      final Function<Function<JavaExpression, JavaExpression>, Function<JavaExpression, JavaExpression>> _function = (Function<JavaExpression, JavaExpression> itemConversion) -> {
        Function<JavaExpression, JavaExpression> _xifexpression = null;
        boolean _isList = this.typeUtil.isList(actual);
        if (_isList) {
          final Function<JavaExpression, JavaExpression> _function_1 = (JavaExpression it) -> {
            return this.getListItemConversionExpression(it, itemConversion, scope);
          };
          _xifexpression = _function_1;
        } else {
          Function<JavaExpression, JavaExpression> _xifexpression_1 = null;
          boolean _isMapperS = this.typeUtil.isMapperS(actual);
          if (_isMapperS) {
            final Function<JavaExpression, JavaExpression> _function_2 = (JavaExpression it) -> {
              return this.getMapperSItemConversionExpression(it, itemConversion, scope);
            };
            _xifexpression_1 = _function_2;
          } else {
            Function<JavaExpression, JavaExpression> _xifexpression_2 = null;
            boolean _isMapperC = this.typeUtil.isMapperC(actual);
            if (_isMapperC) {
              final Function<JavaExpression, JavaExpression> _function_3 = (JavaExpression it) -> {
                return this.getMapperCItemConversionExpression(it, itemConversion, scope);
              };
              _xifexpression_2 = _function_3;
            } else {
              Function<JavaExpression, JavaExpression> _xifexpression_3 = null;
              boolean _isMapperListOfLists = this.typeUtil.isMapperListOfLists(actual);
              if (_isMapperListOfLists) {
                final Function<JavaExpression, JavaExpression> _function_4 = (JavaExpression it) -> {
                  return this.getMapperListOfListsItemConversionExpression(it, itemConversion, scope);
                };
                _xifexpression_3 = _function_4;
              } else {
                throw this.unexpectedWrapperException(actual);
              }
              _xifexpression_2 = _xifexpression_3;
            }
            _xifexpression_1 = _xifexpression_2;
          }
          _xifexpression = _xifexpression_1;
        }
        return _xifexpression;
      };
      _xblockexpression = this.getItemConversion(actualItemType, expectedItemType).<Function<JavaExpression, JavaExpression>>map(_function);
    }
    return _xblockexpression;
  }

  private Optional<Function<JavaExpression, JavaExpression>> getWrapperConversion(final JavaType actual, final JavaType expected) {
    Function<JavaExpression, JavaExpression> _xifexpression = null;
    boolean _isComparisonResult = this.typeUtil.isComparisonResult(actual);
    if (_isComparisonResult) {
      Function<JavaExpression, JavaExpression> _xifexpression_1 = null;
      boolean _isMapperS = this.typeUtil.isMapperS(expected);
      if (_isMapperS) {
        final Function<JavaExpression, JavaExpression> _function = (JavaExpression it) -> {
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              _builder.append(it);
              _builder.append(".asMapper()");
            }
          };
          return JavaExpression.from(_client, this.typeUtil.<MapperS<?>>wrap(this.typeUtil.MAPPER_S, this.typeUtil.BOOLEAN));
        };
        _xifexpression_1 = _function;
      } else {
        Function<JavaExpression, JavaExpression> _xifexpression_2 = null;
        boolean _isMapperC = this.typeUtil.isMapperC(expected);
        if (_isMapperC) {
          _xifexpression_2 = null;
        } else {
          Function<JavaExpression, JavaExpression> _xifexpression_3 = null;
          boolean _isMapperListOfLists = this.typeUtil.isMapperListOfLists(expected);
          if (_isMapperListOfLists) {
            _xifexpression_3 = null;
          } else {
            Function<JavaExpression, JavaExpression> _xifexpression_4 = null;
            boolean _isList = this.typeUtil.isList(expected);
            if (_isList) {
              final Function<JavaExpression, JavaExpression> _function_1 = (JavaExpression it) -> {
                StringConcatenationClient _client = new StringConcatenationClient() {
                  @Override
                  protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                    _builder.append(it);
                    _builder.append(".getMulti()");
                  }
                };
                return JavaExpression.from(_client, this.typeUtil.<List<?>>wrap(this.typeUtil.LIST, this.typeUtil.BOOLEAN));
              };
              _xifexpression_4 = _function_1;
            }
            _xifexpression_3 = _xifexpression_4;
          }
          _xifexpression_2 = _xifexpression_3;
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    } else {
      Function<JavaExpression, JavaExpression> _xifexpression_5 = null;
      boolean _extendsMapper = this.typeUtil.extendsMapper(actual);
      if (_extendsMapper) {
        Function<JavaExpression, JavaExpression> _xifexpression_6 = null;
        boolean _isComparisonResult_1 = this.typeUtil.isComparisonResult(expected);
        if (_isComparisonResult_1) {
          final Function<JavaExpression, JavaExpression> _function_2 = (JavaExpression it) -> {
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append(ComparisonResult.class);
                _builder.append(".of(");
                _builder.append(it);
                _builder.append(")");
              }
            };
            return JavaExpression.from(_client, this.typeUtil.COMPARISON_RESULT);
          };
          _xifexpression_6 = _function_2;
        } else {
          Function<JavaExpression, JavaExpression> _xifexpression_7 = null;
          boolean _extendsMapper_1 = this.typeUtil.extendsMapper(expected);
          if (_extendsMapper_1) {
            Function<JavaExpression, JavaExpression> _xifexpression_8 = null;
            if ((((this.typeUtil.isMapperS(actual) && this.typeUtil.hasWildcardArgument(actual)) && this.typeUtil.isMapperS(expected)) && (!this.typeUtil.hasWildcardArgument(expected)))) {
              final Function<JavaExpression, JavaExpression> _function_3 = (JavaExpression it) -> {
                StringConcatenationClient _client = new StringConcatenationClient() {
                  @Override
                  protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                    _builder.append(it);
                    _builder.append(".map(\"Make mutable\", ");
                    _builder.append(Function.class);
                    _builder.append(".identity())");
                  }
                };
                return JavaExpression.from(_client, this.typeUtil.<MapperS<?>>wrap(this.typeUtil.MAPPER_S, this.typeUtil.getItemType(expected)));
              };
              _xifexpression_8 = _function_3;
            } else {
              Function<JavaExpression, JavaExpression> _xifexpression_9 = null;
              if ((((this.typeUtil.isMapperC(actual) && this.typeUtil.hasWildcardArgument(actual)) && this.typeUtil.isMapperC(expected)) && (!this.typeUtil.hasWildcardArgument(expected)))) {
                final Function<JavaExpression, JavaExpression> _function_4 = (JavaExpression it) -> {
                  StringConcatenationClient _client = new StringConcatenationClient() {
                    @Override
                    protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                      _builder.append(it);
                      _builder.append(".map(\"Make mutable\", ");
                      _builder.append(Function.class);
                      _builder.append(".identity())");
                    }
                  };
                  return JavaExpression.from(_client, this.typeUtil.<MapperC<?>>wrap(this.typeUtil.MAPPER_C, this.typeUtil.getItemType(expected)));
                };
                _xifexpression_9 = _function_4;
              } else {
                Function<JavaExpression, JavaExpression> _xifexpression_10 = null;
                if ((this.typeUtil.isMapperS(actual) && this.typeUtil.isMapperC(expected))) {
                  final Function<JavaExpression, JavaExpression> _function_5 = (JavaExpression it) -> {
                    StringConcatenationClient _client = new StringConcatenationClient() {
                      @Override
                      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                        _builder.append(MapperC.class);
                        _builder.append(".of(");
                        _builder.append(it);
                        _builder.append(")");
                      }
                    };
                    return JavaExpression.from(_client, this.typeUtil.<MapperC<?>>wrap(this.typeUtil.MAPPER_C, this.typeUtil.getItemType(expected)));
                  };
                  _xifexpression_10 = _function_5;
                } else {
                  Function<JavaExpression, JavaExpression> _xifexpression_11 = null;
                  if ((this.typeUtil.isMapperC(actual) && this.typeUtil.isMapperS(expected))) {
                    final Function<JavaExpression, JavaExpression> _function_6 = (JavaExpression it) -> {
                      StringConcatenationClient _client = new StringConcatenationClient() {
                        @Override
                        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                          _builder.append(MapperS.class);
                          _builder.append(".of(");
                          _builder.append(it);
                          _builder.append(".get())");
                        }
                      };
                      return JavaExpression.from(_client, this.typeUtil.<MapperS<?>>wrap(this.typeUtil.MAPPER_S, this.typeUtil.getItemType(expected)));
                    };
                    _xifexpression_11 = _function_6;
                  }
                  _xifexpression_10 = _xifexpression_11;
                }
                _xifexpression_9 = _xifexpression_10;
              }
              _xifexpression_8 = _xifexpression_9;
            }
            _xifexpression_7 = _xifexpression_8;
          } else {
            Function<JavaExpression, JavaExpression> _xifexpression_12 = null;
            boolean _isMapperListOfLists_1 = this.typeUtil.isMapperListOfLists(expected);
            if (_isMapperListOfLists_1) {
              _xifexpression_12 = null;
            } else {
              Function<JavaExpression, JavaExpression> _xifexpression_13 = null;
              boolean _isList_1 = this.typeUtil.isList(expected);
              if (_isList_1) {
                Function<JavaExpression, JavaExpression> _xifexpression_14 = null;
                if ((this.typeUtil.hasWildcardArgument(actual) && (!this.typeUtil.hasWildcardArgument(expected)))) {
                  final Function<JavaExpression, JavaExpression> _function_7 = (JavaExpression it) -> {
                    StringConcatenationClient _client = new StringConcatenationClient() {
                      @Override
                      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                        _builder.append("new ");
                        _builder.append(ArrayList.class);
                        _builder.append("<>(");
                        _builder.append(it);
                        _builder.append(".getMulti())");
                      }
                    };
                    return JavaExpression.from(_client, this.typeUtil.<List<?>>wrap(this.typeUtil.LIST, this.typeUtil.getItemType(expected)));
                  };
                  _xifexpression_14 = _function_7;
                } else {
                  final Function<JavaExpression, JavaExpression> _function_8 = (JavaExpression it) -> {
                    StringConcatenationClient _client = new StringConcatenationClient() {
                      @Override
                      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                        _builder.append(it);
                        _builder.append(".getMulti()");
                      }
                    };
                    return JavaExpression.from(_client, this.typeUtil.<List<?>>wrap(this.typeUtil.LIST, this.typeUtil.getItemType(expected)));
                  };
                  _xifexpression_14 = _function_8;
                }
                _xifexpression_13 = _xifexpression_14;
              }
              _xifexpression_12 = _xifexpression_13;
            }
            _xifexpression_7 = _xifexpression_12;
          }
          _xifexpression_6 = _xifexpression_7;
        }
        _xifexpression_5 = _xifexpression_6;
      } else {
        Function<JavaExpression, JavaExpression> _xifexpression_15 = null;
        boolean _isMapperListOfLists_2 = this.typeUtil.isMapperListOfLists(actual);
        if (_isMapperListOfLists_2) {
          _xifexpression_15 = null;
        } else {
          Function<JavaExpression, JavaExpression> _xifexpression_16 = null;
          boolean _isList_2 = this.typeUtil.isList(actual);
          if (_isList_2) {
            Function<JavaExpression, JavaExpression> _xifexpression_17 = null;
            boolean _isComparisonResult_2 = this.typeUtil.isComparisonResult(expected);
            if (_isComparisonResult_2) {
              final Function<JavaExpression, JavaExpression> _function_9 = (JavaExpression it) -> {
                StringConcatenationClient _client = new StringConcatenationClient() {
                  @Override
                  protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                    _builder.append(ComparisonResult.class);
                    _builder.append(".of(");
                    _builder.append(MapperC.class);
                    _builder.append(".of(");
                    _builder.append(it);
                    _builder.append("))");
                  }
                };
                return JavaExpression.from(_client, this.typeUtil.COMPARISON_RESULT);
              };
              _xifexpression_17 = _function_9;
            } else {
              Function<JavaExpression, JavaExpression> _xifexpression_18 = null;
              boolean _isMapperS_1 = this.typeUtil.isMapperS(expected);
              if (_isMapperS_1) {
                final Function<JavaExpression, JavaExpression> _function_10 = (JavaExpression it) -> {
                  StringConcatenationClient _client = new StringConcatenationClient() {
                    @Override
                    protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                      _builder.append(MapperS.class);
                      _builder.append(".of(");
                      _builder.append(it);
                      _builder.append(".get(0))");
                    }
                  };
                  return JavaExpression.from(_client, this.typeUtil.<MapperS<?>>wrap(this.typeUtil.MAPPER_S, this.typeUtil.getItemType(expected)));
                };
                _xifexpression_18 = _function_10;
              } else {
                Function<JavaExpression, JavaExpression> _xifexpression_19 = null;
                if ((this.typeUtil.isMapperC(expected) || this.typeUtil.isMapper(expected))) {
                  final Function<JavaExpression, JavaExpression> _function_11 = (JavaExpression it) -> {
                    StringConcatenationClient _client = new StringConcatenationClient() {
                      @Override
                      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                        _builder.append(MapperC.class);
                        _builder.append(".<");
                        JavaType _itemType = TypeCoercionService.this.typeUtil.getItemType(expected);
                        _builder.append(_itemType);
                        _builder.append(">of(");
                        _builder.append(it);
                        _builder.append(")");
                      }
                    };
                    return JavaExpression.from(_client, this.typeUtil.<MapperC<?>>wrap(this.typeUtil.MAPPER_C, this.typeUtil.getItemType(expected)));
                  };
                  _xifexpression_19 = _function_11;
                } else {
                  Function<JavaExpression, JavaExpression> _xifexpression_20 = null;
                  boolean _isMapperListOfLists_3 = this.typeUtil.isMapperListOfLists(expected);
                  if (_isMapperListOfLists_3) {
                    _xifexpression_20 = null;
                  } else {
                    Function<JavaExpression, JavaExpression> _xifexpression_21 = null;
                    if (((this.typeUtil.isList(expected) && this.typeUtil.hasWildcardArgument(actual)) && (!this.typeUtil.hasWildcardArgument(expected)))) {
                      final Function<JavaExpression, JavaExpression> _function_12 = (JavaExpression it) -> {
                        StringConcatenationClient _client = new StringConcatenationClient() {
                          @Override
                          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                            _builder.append("new ");
                            _builder.append(ArrayList.class);
                            _builder.append("(");
                            _builder.append(it);
                            _builder.append(")");
                          }
                        };
                        return JavaExpression.from(_client, this.typeUtil.<List<?>>wrap(this.typeUtil.LIST, this.typeUtil.getItemType(expected)));
                      };
                      _xifexpression_21 = _function_12;
                    }
                    _xifexpression_20 = _xifexpression_21;
                  }
                  _xifexpression_19 = _xifexpression_20;
                }
                _xifexpression_18 = _xifexpression_19;
              }
              _xifexpression_17 = _xifexpression_18;
            }
            _xifexpression_16 = _xifexpression_17;
          }
          _xifexpression_15 = _xifexpression_16;
        }
        _xifexpression_5 = _xifexpression_15;
      }
      _xifexpression = _xifexpression_5;
    }
    return Optional.<Function<JavaExpression, JavaExpression>>ofNullable(_xifexpression);
  }

  private JavaStatementBuilder convertNullSafe(final JavaExpression expr, final Function<JavaExpression, JavaExpression> conversion, final JavaType expected, final JavaScope scope) {
    final JavaType actual = expr.getExpressionType();
    if ((actual instanceof JavaPrimitiveType)) {
      return expr.mapExpression(conversion);
    }
    final Function<JavaExpression, JavaStatementBuilder> _function = (JavaExpression it) -> {
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append(it);
          _builder.append(" == null");
        }
      };
      JavaExpression _from = JavaExpression.from(_client, JavaPrimitiveType.BOOLEAN);
      JavaExpression _empty = this.empty(expected);
      JavaExpression _apply = conversion.apply(it);
      return new JavaConditionalExpression(_from, _empty, _apply, 
        this.typeUtil);
    };
    return expr.declareAsVariable(true, StringExtensions.toFirstLower(actual.getSimpleName()), scope).mapExpression(_function);
  }

  private JavaExpression empty(final JavaType expected) {
    JavaExpression _xblockexpression = null;
    {
      final JavaType itemType = this.typeUtil.getItemType(expected);
      JavaExpression _xifexpression = null;
      boolean _isList = this.typeUtil.isList(expected);
      if (_isList) {
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append(Collections.class);
            _builder.append(".<");
            _builder.append(itemType);
            _builder.append(">emptyList()");
          }
        };
        _xifexpression = JavaExpression.from(_client, this.typeUtil.<List<?>>wrap(this.typeUtil.LIST, itemType));
      } else {
        JavaExpression _xifexpression_1 = null;
        if ((this.typeUtil.isMapperS(expected) || this.typeUtil.isMapper(expected))) {
          StringConcatenationClient _client_1 = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              _builder.append(MapperS.class);
              _builder.append(".<");
              _builder.append(itemType);
              _builder.append(">ofNull()");
            }
          };
          _xifexpression_1 = JavaExpression.from(_client_1, this.typeUtil.<MapperS<?>>wrap(this.typeUtil.MAPPER_S, itemType));
        } else {
          JavaExpression _xifexpression_2 = null;
          boolean _isMapperC = this.typeUtil.isMapperC(expected);
          if (_isMapperC) {
            StringConcatenationClient _client_2 = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append(MapperC.class);
                _builder.append(".<");
                _builder.append(itemType);
                _builder.append(">ofNull()");
              }
            };
            _xifexpression_2 = JavaExpression.from(_client_2, this.typeUtil.<MapperC<?>>wrap(this.typeUtil.MAPPER_C, itemType));
          } else {
            JavaExpression _xifexpression_3 = null;
            boolean _isComparisonResult = this.typeUtil.isComparisonResult(expected);
            if (_isComparisonResult) {
              StringConcatenationClient _client_3 = new StringConcatenationClient() {
                @Override
                protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                  _builder.append(ComparisonResult.class);
                  _builder.append(".successEmptyOperand(\"\")");
                }
              };
              _xifexpression_3 = JavaExpression.from(_client_3, this.typeUtil.COMPARISON_RESULT);
            } else {
              JavaExpression _xifexpression_4 = null;
              boolean _equals = Objects.equal(expected, JavaPrimitiveType.BOOLEAN);
              if (_equals) {
                StringConcatenationClient _client_4 = new StringConcatenationClient() {
                  @Override
                  protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                    _builder.append("false");
                  }
                };
                _xifexpression_4 = JavaExpression.from(_client_4, JavaPrimitiveType.BOOLEAN);
              } else {
                JavaExpression _xifexpression_5 = null;
                if ((expected instanceof JavaPrimitiveType)) {
                  throw new IllegalArgumentException((("No empty representation for primitive type `" + expected) + "`."));
                } else {
                  _xifexpression_5 = JavaExpression.NULL;
                }
                _xifexpression_4 = _xifexpression_5;
              }
              _xifexpression_3 = _xifexpression_4;
            }
            _xifexpression_2 = _xifexpression_3;
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  private JavaExpression getNumberConversionExpression(final JavaExpression expression, final JavaType expected) {
    JavaExpression _xblockexpression = null;
    {
      final JavaType actual = expression.getExpressionType();
      JavaExpression _xifexpression = null;
      boolean _isInteger = this.typeUtil.isInteger(actual.toReferenceType());
      if (_isInteger) {
        JavaExpression _xifexpression_1 = null;
        boolean _isLong = this.typeUtil.isLong(expected.toReferenceType());
        if (_isLong) {
          StringConcatenationClient _xifexpression_2 = null;
          boolean _equals = Objects.equal(actual, JavaPrimitiveType.INT);
          if (_equals) {
            StringConcatenationClient _xifexpression_3 = null;
            boolean _equals_1 = Objects.equal(expected, JavaPrimitiveType.LONG);
            if (_equals_1) {
              StringConcatenationClient _client = new StringConcatenationClient() {
                @Override
                protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                  _builder.append(expression);
                }
              };
              _xifexpression_3 = _client;
            } else {
              StringConcatenationClient _client_1 = new StringConcatenationClient() {
                @Override
                protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                  _builder.append("(long) ");
                  _builder.append(expression);
                }
              };
              _xifexpression_3 = _client_1;
            }
            _xifexpression_2 = _xifexpression_3;
          } else {
            StringConcatenationClient _client_2 = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append(expression);
                _builder.append(".longValue()");
              }
            };
            _xifexpression_2 = _client_2;
          }
          _xifexpression_1 = JavaExpression.from(_xifexpression_2, JavaPrimitiveType.LONG);
        } else {
          JavaExpression _xifexpression_4 = null;
          boolean _isBigInteger = this.typeUtil.isBigInteger(expected);
          if (_isBigInteger) {
            StringConcatenationClient _client_3 = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append(BigInteger.class);
                _builder.append(".valueOf(");
                _builder.append(expression);
                _builder.append(")");
              }
            };
            _xifexpression_4 = JavaExpression.from(_client_3, this.typeUtil.BIG_INTEGER);
          } else {
            JavaExpression _xifexpression_5 = null;
            boolean _isBigDecimal = this.typeUtil.isBigDecimal(expected);
            if (_isBigDecimal) {
              StringConcatenationClient _client_4 = new StringConcatenationClient() {
                @Override
                protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                  _builder.append(BigDecimal.class);
                  _builder.append(".valueOf(");
                  _builder.append(expression);
                  _builder.append(")");
                }
              };
              _xifexpression_5 = JavaExpression.from(_client_4, this.typeUtil.BIG_DECIMAL);
            } else {
              throw this.unexpectedCaseException(actual, expected);
            }
            _xifexpression_4 = _xifexpression_5;
          }
          _xifexpression_1 = _xifexpression_4;
        }
        _xifexpression = _xifexpression_1;
      } else {
        JavaExpression _xifexpression_6 = null;
        boolean _isLong_1 = this.typeUtil.isLong(actual.toReferenceType());
        if (_isLong_1) {
          JavaExpression _xifexpression_7 = null;
          boolean _isInteger_1 = this.typeUtil.isInteger(expected.toReferenceType());
          if (_isInteger_1) {
            StringConcatenationClient _client_5 = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append(Math.class);
                _builder.append(".toIntExact(");
                _builder.append(expression);
                _builder.append(")");
              }
            };
            _xifexpression_7 = JavaExpression.from(_client_5, JavaPrimitiveType.INT);
          } else {
            JavaExpression _xifexpression_8 = null;
            boolean _isBigInteger_1 = this.typeUtil.isBigInteger(expected);
            if (_isBigInteger_1) {
              StringConcatenationClient _client_6 = new StringConcatenationClient() {
                @Override
                protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                  _builder.append(BigInteger.class);
                  _builder.append(".valueOf(");
                  _builder.append(expression);
                  _builder.append(")");
                }
              };
              _xifexpression_8 = JavaExpression.from(_client_6, this.typeUtil.BIG_INTEGER);
            } else {
              JavaExpression _xifexpression_9 = null;
              boolean _isBigDecimal_1 = this.typeUtil.isBigDecimal(expected);
              if (_isBigDecimal_1) {
                StringConcatenationClient _client_7 = new StringConcatenationClient() {
                  @Override
                  protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                    _builder.append(BigDecimal.class);
                    _builder.append(".valueOf(");
                    _builder.append(expression);
                    _builder.append(")");
                  }
                };
                _xifexpression_9 = JavaExpression.from(_client_7, this.typeUtil.BIG_DECIMAL);
              } else {
                throw this.unexpectedCaseException(actual, expected);
              }
              _xifexpression_8 = _xifexpression_9;
            }
            _xifexpression_7 = _xifexpression_8;
          }
          _xifexpression_6 = _xifexpression_7;
        } else {
          JavaExpression _xifexpression_10 = null;
          boolean _isBigInteger_2 = this.typeUtil.isBigInteger(actual);
          if (_isBigInteger_2) {
            JavaExpression _xifexpression_11 = null;
            boolean _isInteger_2 = this.typeUtil.isInteger(expected.toReferenceType());
            if (_isInteger_2) {
              StringConcatenationClient _client_8 = new StringConcatenationClient() {
                @Override
                protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                  _builder.append(expression);
                  _builder.append(".intValueExact()");
                }
              };
              _xifexpression_11 = JavaExpression.from(_client_8, JavaPrimitiveType.INT);
            } else {
              JavaExpression _xifexpression_12 = null;
              boolean _isLong_2 = this.typeUtil.isLong(expected.toReferenceType());
              if (_isLong_2) {
                StringConcatenationClient _client_9 = new StringConcatenationClient() {
                  @Override
                  protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                    _builder.append(expression);
                    _builder.append(".longValueExact()");
                  }
                };
                _xifexpression_12 = JavaExpression.from(_client_9, JavaPrimitiveType.LONG);
              } else {
                JavaExpression _xifexpression_13 = null;
                boolean _isBigDecimal_2 = this.typeUtil.isBigDecimal(expected);
                if (_isBigDecimal_2) {
                  StringConcatenationClient _client_10 = new StringConcatenationClient() {
                    @Override
                    protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                      _builder.append("new ");
                      _builder.append(BigDecimal.class);
                      _builder.append("(");
                      _builder.append(expression);
                      _builder.append(")");
                    }
                  };
                  _xifexpression_13 = JavaExpression.from(_client_10, this.typeUtil.BIG_DECIMAL);
                } else {
                  throw this.unexpectedCaseException(actual, expected);
                }
                _xifexpression_12 = _xifexpression_13;
              }
              _xifexpression_11 = _xifexpression_12;
            }
            _xifexpression_10 = _xifexpression_11;
          } else {
            JavaExpression _xifexpression_14 = null;
            boolean _isBigDecimal_3 = this.typeUtil.isBigDecimal(actual);
            if (_isBigDecimal_3) {
              JavaExpression _xifexpression_15 = null;
              boolean _isInteger_3 = this.typeUtil.isInteger(expected.toReferenceType());
              if (_isInteger_3) {
                StringConcatenationClient _client_11 = new StringConcatenationClient() {
                  @Override
                  protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                    _builder.append(expression);
                    _builder.append(".intValueExact()");
                  }
                };
                _xifexpression_15 = JavaExpression.from(_client_11, JavaPrimitiveType.INT);
              } else {
                JavaExpression _xifexpression_16 = null;
                boolean _isLong_3 = this.typeUtil.isLong(expected.toReferenceType());
                if (_isLong_3) {
                  StringConcatenationClient _client_12 = new StringConcatenationClient() {
                    @Override
                    protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                      _builder.append(expression);
                      _builder.append(".longValueExact()");
                    }
                  };
                  _xifexpression_16 = JavaExpression.from(_client_12, JavaPrimitiveType.LONG);
                } else {
                  JavaExpression _xifexpression_17 = null;
                  boolean _isBigInteger_3 = this.typeUtil.isBigInteger(expected);
                  if (_isBigInteger_3) {
                    StringConcatenationClient _client_13 = new StringConcatenationClient() {
                      @Override
                      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                        _builder.append(expression);
                        _builder.append(".toBigIntegerExact()");
                      }
                    };
                    _xifexpression_17 = JavaExpression.from(_client_13, this.typeUtil.BIG_INTEGER);
                  } else {
                    throw this.unexpectedCaseException(actual, expected);
                  }
                  _xifexpression_16 = _xifexpression_17;
                }
                _xifexpression_15 = _xifexpression_16;
              }
              _xifexpression_14 = _xifexpression_15;
            } else {
              throw this.unexpectedCaseException(actual, expected);
            }
            _xifexpression_10 = _xifexpression_14;
          }
          _xifexpression_6 = _xifexpression_10;
        }
        _xifexpression = _xifexpression_6;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  private JavaExpression getItemToListConversionExpression(final JavaExpression expression) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append(Collections.class);
        _builder.append(".singletonList(");
        _builder.append(expression);
        _builder.append(")");
      }
    };
    return JavaExpression.from(_client, this.typeUtil.<List<?>>wrap(this.typeUtil.LIST, expression.getExpressionType()));
  }

  private JavaExpression getItemToMapperSConversionExpression(final JavaExpression expression) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append(MapperS.class);
        _builder.append(".of(");
        _builder.append(expression);
        _builder.append(")");
      }
    };
    return JavaExpression.from(_client, this.typeUtil.<MapperS<?>>wrap(this.typeUtil.MAPPER_S, expression.getExpressionType()));
  }

  private JavaExpression getItemToMapperCConversionExpression(final JavaExpression expression) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append(MapperC.class);
        _builder.append(".of(");
        _builder.append(Collections.class);
        _builder.append(".singletonList(");
        _builder.append(expression);
        _builder.append("))");
      }
    };
    return JavaExpression.from(_client, this.typeUtil.<MapperC<?>>wrap(this.typeUtil.MAPPER_C, expression.getExpressionType()));
  }

  private JavaExpression getItemToComparisonResultConversionExpression(final JavaExpression expression) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append(ComparisonResult.class);
        _builder.append(".of(");
        _builder.append(MapperS.class);
        _builder.append(".of(");
        _builder.append(expression);
        _builder.append("))");
      }
    };
    return JavaExpression.from(_client, this.typeUtil.COMPARISON_RESULT);
  }

  private JavaExpression getListToItemConversionExpression(final JavaExpression expression) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append(MapperC.class);
        _builder.append(".of(");
        _builder.append(expression);
        _builder.append(").get()");
      }
    };
    return JavaExpression.from(_client, this.typeUtil.getItemType(expression.getExpressionType()));
  }

  private JavaExpression getMapperToItemConversionExpression(final JavaExpression expression) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append(expression);
        _builder.append(".get()");
      }
    };
    return JavaExpression.from(_client, this.typeUtil.getItemType(expression.getExpressionType()));
  }

  private JavaExpression getListItemConversionExpression(final JavaExpression expression, final Function<JavaExpression, JavaExpression> itemConversion, final JavaScope scope) {
    JavaExpression _xblockexpression = null;
    {
      final JavaType actualItemType = this.typeUtil.getItemType(expression.getExpressionType());
      final JavaScope lambdaScope = scope.lambdaScope();
      final GeneratedIdentifier lambdaParam = lambdaScope.createUniqueIdentifier(StringExtensions.toFirstLower(actualItemType.getSimpleName()));
      JavaVariable _javaVariable = new JavaVariable(lambdaParam, actualItemType);
      final JavaExpression resultItem = itemConversion.apply(_javaVariable);
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append(expression);
          _builder.append(".stream()");
          _builder.newLineIfNotEmpty();
          _builder.append("\t");
          _builder.append(".<");
          JavaType _expressionType = resultItem.getExpressionType();
          _builder.append(_expressionType, "\t");
          _builder.append(">map(");
          _builder.append(lambdaParam, "\t");
          _builder.append(" -> ");
          _builder.append(resultItem, "\t");
          _builder.append(")");
          _builder.newLineIfNotEmpty();
          _builder.append("\t");
          _builder.append(".collect(");
          _builder.append(Collectors.class, "\t");
          _builder.append(".toList())");
          _builder.newLineIfNotEmpty();
        }
      };
      _xblockexpression = JavaExpression.from(_client, 
        this.typeUtil.<List<?>>wrap(this.typeUtil.LIST, resultItem.getExpressionType()));
    }
    return _xblockexpression;
  }

  private JavaExpression getMapperSItemConversionExpression(final JavaExpression expression, final Function<JavaExpression, JavaExpression> itemConversion, final JavaScope scope) {
    JavaExpression _xblockexpression = null;
    {
      final JavaType actualItemType = this.typeUtil.getItemType(expression.getExpressionType());
      final JavaScope lambdaScope = scope.lambdaScope();
      final GeneratedIdentifier lambdaParam = lambdaScope.createUniqueIdentifier(StringExtensions.toFirstLower(actualItemType.getSimpleName()));
      JavaVariable _javaVariable = new JavaVariable(lambdaParam, actualItemType);
      final JavaExpression resultItem = itemConversion.apply(_javaVariable);
      final JavaParameterizedType<MapperS<?>> resultType = this.typeUtil.<MapperS<?>>wrap(this.typeUtil.MAPPER_S, resultItem.getExpressionType());
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append(expression);
          _builder.append(".<");
          JavaType _itemType = TypeCoercionService.this.typeUtil.getItemType(resultType);
          _builder.append(_itemType);
          _builder.append(">map(\"Type coercion\", ");
          _builder.append(lambdaParam);
          _builder.append(" -> ");
          _builder.append(lambdaParam);
          _builder.append(" == null ? null : ");
          _builder.append(resultItem);
          _builder.append(")");
        }
      };
      _xblockexpression = JavaExpression.from(_client, resultType);
    }
    return _xblockexpression;
  }

  private JavaExpression getMapperCItemConversionExpression(final JavaExpression expression, final Function<JavaExpression, JavaExpression> itemConversion, final JavaScope scope) {
    JavaExpression _xblockexpression = null;
    {
      final JavaType actualItemType = this.typeUtil.getItemType(expression.getExpressionType());
      final JavaScope lambdaScope = scope.lambdaScope();
      final GeneratedIdentifier lambdaParam = lambdaScope.createUniqueIdentifier(StringExtensions.toFirstLower(actualItemType.getSimpleName()));
      JavaVariable _javaVariable = new JavaVariable(lambdaParam, actualItemType);
      final JavaExpression resultItem = itemConversion.apply(_javaVariable);
      final JavaParameterizedType<MapperC<?>> resultType = this.typeUtil.<MapperC<?>>wrap(this.typeUtil.MAPPER_C, resultItem.getExpressionType());
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append(expression);
          _builder.append(".<");
          JavaType _itemType = TypeCoercionService.this.typeUtil.getItemType(resultType);
          _builder.append(_itemType);
          _builder.append(">map(\"Type coercion\", ");
          _builder.append(lambdaParam);
          _builder.append(" -> ");
          _builder.append(resultItem);
          _builder.append(")");
        }
      };
      _xblockexpression = JavaExpression.from(_client, resultType);
    }
    return _xblockexpression;
  }

  private JavaExpression getMapperListOfListsItemConversionExpression(final JavaExpression expression, final Function<JavaExpression, JavaExpression> itemConversion, final JavaScope scope) {
    JavaExpression _xblockexpression = null;
    {
      final JavaType actualItemType = this.typeUtil.getItemType(expression.getExpressionType());
      final JavaScope listToListLambdaScope = scope.lambdaScope();
      final GeneratedIdentifier mapperCParam = listToListLambdaScope.createUniqueIdentifier("mapperC");
      JavaParameterizedType<MapperC<?>> _wrap = this.typeUtil.<MapperC<?>>wrap(this.typeUtil.MAPPER_C, actualItemType);
      JavaVariable _javaVariable = new JavaVariable(mapperCParam, _wrap);
      final JavaExpression resultMapperC = this.getMapperCItemConversionExpression(_javaVariable, itemConversion, listToListLambdaScope);
      final JavaParameterizedType<MapperListOfLists<?>> resultType = this.typeUtil.<MapperListOfLists<?>>wrap(this.typeUtil.MAPPER_LIST_OF_LISTS, this.typeUtil.getItemType(resultMapperC.getExpressionType()));
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append(expression);
          _builder.append(".<");
          JavaType _itemType = TypeCoercionService.this.typeUtil.getItemType(resultType);
          _builder.append(_itemType);
          _builder.append(">mapListToList(");
          _builder.append(mapperCParam);
          _builder.append(" -> ");
          _builder.append(resultMapperC);
          _builder.append(")");
        }
      };
      _xblockexpression = JavaExpression.from(_client, resultType);
    }
    return _xblockexpression;
  }

  private IllegalArgumentException unexpectedCaseException(final JavaType actual, final JavaType expected) {
    return new IllegalArgumentException((((("Cannot coerce from " + actual) + " to ") + expected) + "."));
  }

  private IllegalArgumentException unexpectedWrapperException(final JavaType expectedWrapper) {
    return new IllegalArgumentException((("Cannot wrap to " + expectedWrapper) + "."));
  }
}
