/*
 * Decompiled with CFR 0.152.
 */
package com.renomad.minum.database;

import com.renomad.minum.database.DbData;
import com.renomad.minum.database.DbException;
import com.renomad.minum.logging.ILogger;
import com.renomad.minum.queue.AbstractActionQueue;
import com.renomad.minum.queue.ActionQueue;
import com.renomad.minum.state.Context;
import com.renomad.minum.utils.FileUtils;
import com.renomad.minum.utils.Invariants;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Stream;

public final class Db<T extends DbData<?>> {
    static final String DATABASE_FILE_SUFFIX = ".ddps";
    private final T emptyInstance;
    private final Lock loadDataLock = new ReentrantLock();
    private final Lock writeLock = new ReentrantLock();
    private final Lock deleteLock = new ReentrantLock();
    private final Lock modificationLock = new ReentrantLock();
    private final Path fullPathForIndexFile;
    final AtomicLong index;
    private final Path dbDirectory;
    private final AbstractActionQueue actionQueue;
    private final ILogger logger;
    private final Map<Long, T> data = new ConcurrentHashMap<Long, T>();
    private final FileUtils fileUtils;
    private boolean hasLoadedData = false;

    public Db(Path dbDirectory, Context context, T instance) {
        this.actionQueue = new ActionQueue("DatabaseWriter " + String.valueOf(dbDirectory), context).initialize();
        this.logger = context.getLogger();
        this.dbDirectory = dbDirectory;
        this.fullPathForIndexFile = dbDirectory.resolve("index.ddps");
        this.emptyInstance = instance;
        this.fileUtils = new FileUtils(this.logger, context.getConstants());
        if (Files.exists(this.fullPathForIndexFile, new LinkOption[0])) {
            long indexValue;
            try (FileReader fileReader = new FileReader(this.fullPathForIndexFile.toFile(), StandardCharsets.UTF_8);
                 BufferedReader br = new BufferedReader(fileReader);){
                String s = br.readLine();
                if (s == null) {
                    throw new DbException("index file for " + String.valueOf(dbDirectory) + " returned null when reading a line from it");
                }
                Invariants.mustBeFalse(s.isBlank(), "Unless something is terribly broken, we expect a numeric value here");
                String trim = s.trim();
                indexValue = Long.parseLong(trim);
            }
            catch (Exception e) {
                throw new DbException("Exception while reading " + String.valueOf(this.fullPathForIndexFile) + " in Db constructor", e);
            }
            this.index = new AtomicLong(indexValue);
        } else {
            this.index = new AtomicLong(1L);
        }
        this.actionQueue.enqueue("create directory" + String.valueOf(dbDirectory), () -> this.fileUtils.makeDirectory(dbDirectory));
    }

    public void stop() {
        this.actionQueue.stop();
    }

    public void stop(int count, int sleepTime) {
        this.actionQueue.stop(count, sleepTime);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public T write(T newData) {
        if (((DbData)newData).getIndex() < 0L) {
            throw new DbException("Negative indexes are disallowed");
        }
        this.writeLock.lock();
        try {
            if (!this.hasLoadedData) {
                this.loadData();
            }
            boolean newIndexCreated = false;
            this.modificationLock.lock();
            try {
                if (((DbData)newData).getIndex() == 0L) {
                    ((DbData)newData).setIndex(this.index.getAndIncrement());
                    newIndexCreated = true;
                } else {
                    boolean dataEntryExists = this.data.values().stream().anyMatch(x -> x.getIndex() == newData.getIndex());
                    if (!dataEntryExists) {
                        throw new DbException(String.format("Positive indexes are only allowed when updating existing data. Index: %d", ((DbData)newData).getIndex()));
                    }
                }
                this.logger.logTrace(() -> String.format("in thread %s, writing data %s", Thread.currentThread().getName(), newData));
                this.data.put(((DbData)newData).getIndex(), newData);
            }
            finally {
                this.modificationLock.unlock();
            }
            boolean finalNewIndexCreated = newIndexCreated;
            this.actionQueue.enqueue("persist data to disk", () -> {
                Path fullPath = this.dbDirectory.resolve(newData.getIndex() + DATABASE_FILE_SUFFIX);
                this.logger.logTrace(() -> String.format("writing data to %s", fullPath));
                String serializedData = newData.serialize();
                Invariants.mustBeFalse(serializedData == null || serializedData.isBlank(), "the serialized form of data must not be blank. Is the serialization code written properly? Our datatype: " + String.valueOf(this.emptyInstance));
                this.fileUtils.writeString(fullPath, serializedData);
                if (finalNewIndexCreated) {
                    this.fileUtils.writeString(this.fullPathForIndexFile, String.valueOf(newData.getIndex() + 1L));
                }
            });
            Object t = newData;
            return t;
        }
        finally {
            this.writeLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void delete(T dataToDelete) {
        this.deleteLock.lock();
        try {
            boolean hasResetIndex;
            long dataIndex;
            if (!this.hasLoadedData) {
                this.loadData();
            }
            this.modificationLock.lock();
            try {
                if (dataToDelete == null) {
                    throw new DbException("Db.delete was given a null value to delete");
                }
                dataIndex = ((DbData)dataToDelete).getIndex();
                if (!this.data.containsKey(dataIndex)) {
                    throw new DbException("no data was found with index of " + dataIndex);
                }
                this.logger.logTrace(() -> String.format("in thread %s, deleting data with index %d", Thread.currentThread().getName(), dataIndex));
                this.data.remove(dataIndex);
                if (this.data.isEmpty()) {
                    this.index.set(1L);
                    hasResetIndex = true;
                } else {
                    hasResetIndex = false;
                }
            }
            finally {
                this.modificationLock.unlock();
            }
            this.actionQueue.enqueue("delete data from disk", () -> {
                Path fullPath = this.dbDirectory.resolve(dataIndex + DATABASE_FILE_SUFFIX);
                this.logger.logTrace(() -> String.format("deleting data at %s", fullPath));
                try {
                    Invariants.mustBeTrue(fullPath.toFile().exists(), String.valueOf(fullPath) + " must already exist before deletion");
                    Files.delete(fullPath);
                    if (hasResetIndex) {
                        this.fileUtils.writeString(this.fullPathForIndexFile, String.valueOf(1));
                    }
                }
                catch (Exception ex) {
                    this.logger.logAsyncError(() -> "failed to delete file " + String.valueOf(fullPath) + " during deleteOnDisk. Exception: " + String.valueOf(ex));
                }
            });
        }
        finally {
            this.deleteLock.unlock();
        }
    }

    void loadDataFromDisk() {
        if (!Files.exists(this.dbDirectory, new LinkOption[0])) {
            this.logger.logDebug(() -> String.valueOf(this.dbDirectory) + " directory missing, adding nothing to the data list");
            return;
        }
        this.walkAndLoad(this.dbDirectory);
    }

    void walkAndLoad(Path dbDirectory) {
        try (Stream<Path> pathStream = Files.walk(dbDirectory, new FileVisitOption[0]);){
            List<Path> listOfFiles = pathStream.filter(path -> Files.isRegularFile(path, new LinkOption[0]) && !path.getFileName().toString().startsWith("index")).toList();
            for (Path p : listOfFiles) {
                this.readAndDeserialize(p);
            }
        }
        catch (IOException e) {
            throw new DbException(e);
        }
    }

    void readAndDeserialize(Path p) throws IOException {
        Path fileName = p.getFileName();
        if (fileName == null) {
            throw new DbException("At readAndDeserialize, path " + String.valueOf(p) + " returned a null filename");
        }
        String filename = fileName.toString();
        int startOfSuffixIndex = filename.indexOf(46);
        if (startOfSuffixIndex == -1) {
            throw new DbException("the files must have a ddps suffix, like 1.ddps.  filename: " + filename);
        }
        String fileContents = Files.readString(p);
        if (fileContents.isBlank()) {
            this.logger.logDebug(() -> String.valueOf(fileName) + " file exists but empty, skipping");
        } else {
            try {
                DbData deserializedData = (DbData)((DbData)this.emptyInstance).deserialize(fileContents);
                Invariants.mustBeTrue(deserializedData != null, "deserialization of " + String.valueOf(this.emptyInstance) + " resulted in a null value. Was the serialization method implemented properly?");
                int fileNameIdentifier = Integer.parseInt(filename.substring(0, startOfSuffixIndex));
                Invariants.mustBeTrue(deserializedData.getIndex() == (long)fileNameIdentifier, "The filename must correspond to the data's index. e.g. 1.ddps must have an id of 1");
                this.data.put(deserializedData.getIndex(), deserializedData);
            }
            catch (Exception e) {
                throw new DbException("Failed to deserialize " + String.valueOf(p) + " with data (\"" + fileContents + "\"). Caused by: " + String.valueOf(e));
            }
        }
    }

    public Collection<T> values() {
        if (!this.hasLoadedData) {
            this.loadData();
        }
        return Collections.unmodifiableCollection(this.data.values());
    }

    private void loadData() {
        this.loadDataLock.lock();
        try {
            Db.loadDataCore(this.hasLoadedData, this::loadDataFromDisk);
            this.hasLoadedData = true;
        }
        finally {
            this.loadDataLock.unlock();
        }
    }

    static void loadDataCore(boolean hasLoadedData, Runnable loadDataFromDisk) {
        if (!hasLoadedData) {
            loadDataFromDisk.run();
        }
    }
}

