/*
 * Decompiled with CFR 0.152.
 */
package com.renomad.minum.web;

import com.renomad.minum.logging.ILogger;
import com.renomad.minum.security.ForbiddenUseException;
import com.renomad.minum.state.Constants;
import com.renomad.minum.state.Context;
import com.renomad.minum.utils.StringUtils;
import com.renomad.minum.web.Body;
import com.renomad.minum.web.BodyType;
import com.renomad.minum.web.ContentDisposition;
import com.renomad.minum.web.CountBytesRead;
import com.renomad.minum.web.Headers;
import com.renomad.minum.web.IBodyProcessor;
import com.renomad.minum.web.IInputStreamUtils;
import com.renomad.minum.web.InputStreamUtils;
import com.renomad.minum.web.Partition;
import com.renomad.minum.web.StreamingMultipartPartition;
import com.renomad.minum.web.UrlEncodedDataGetter;
import com.renomad.minum.web.UrlEncodedKeyValue;
import com.renomad.minum.web.WebServerException;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

final class BodyProcessor
implements IBodyProcessor {
    private final ILogger logger;
    private final IInputStreamUtils inputStreamUtils;
    private final Constants constants;
    private static final Pattern multiformNameRegex = Pattern.compile("\\bname\\b=\"(?<namevalue>.*?)\"");
    private static final Pattern multiformFilenameRegex = Pattern.compile("\\bfilename\\b=\"(?<namevalue>.*?)\"");

    BodyProcessor(Context context) {
        this.constants = context.getConstants();
        this.logger = context.getLogger();
        this.inputStreamUtils = new InputStreamUtils(this.constants.maxReadLineSizeBytes);
    }

    @Override
    public Body extractData(InputStream is, Headers h) {
        String contentType = h.contentType();
        if (h.contentLength() >= 0) {
            if (h.contentLength() >= this.constants.maxReadSizeBytes) {
                throw new ForbiddenUseException("It is disallowed to process a body with a length more than " + this.constants.maxReadSizeBytes + " bytes");
            }
        } else {
            List<String> transferEncodingHeaders = h.valueByKey("transfer-encoding");
            if (List.of("chunked").equals(transferEncodingHeaders)) {
                this.logger.logDebug(() -> "client sent chunked transfer-encoding.  Minum does not automatically read bodies of this type.");
            }
            return Body.EMPTY;
        }
        return this.extractBodyFromInputStream(h.contentLength(), contentType, is);
    }

    Body extractBodyFromInputStream(int contentLength, String contentType, InputStream is) {
        if (contentLength == 0) {
            this.logger.logDebug(() -> "the length of the body was 0, returning an empty Body");
            return Body.EMPTY;
        }
        if (contentType.contains("application/x-www-form-urlencoded")) {
            return this.parseUrlEncodedForm(is, contentLength);
        }
        if (contentType.contains("multipart/form-data")) {
            String boundaryValue = BodyProcessor.determineBoundaryValue(contentType);
            return this.parseMultipartForm(contentLength, boundaryValue, is);
        }
        this.logger.logDebug(() -> "did not recognize a key-value pattern content-type, returning the raw bytes for the body.  Content-Type was: " + contentType);
        return new Body(Map.of(), this.inputStreamUtils.read(contentLength, is), List.of(), BodyType.UNRECOGNIZED);
    }

    private Body parseMultipartForm(int contentLength, String boundaryValue, InputStream inputStream) {
        if (boundaryValue.isBlank()) {
            this.logger.logDebug(() -> "The boundary value was blank for the multipart input. Returning an empty map");
            return new Body(Map.of(), new byte[0], List.of(), BodyType.UNRECOGNIZED);
        }
        ArrayList<Partition> partitions = new ArrayList<Partition>();
        try {
            int countOfPartitions = 0;
            for (StreamingMultipartPartition p : this.getMultiPartIterable(inputStream, boundaryValue, contentLength)) {
                if (++countOfPartitions >= 1000) {
                    throw new WebServerException("Error: body had excessive number of partitions (" + countOfPartitions + ").  Maximum allowed: 1000");
                }
                partitions.add(new Partition(p.getHeaders(), p.readAllBytes(), p.getContentDisposition()));
            }
        }
        catch (Exception ex) {
            this.logger.logDebug(() -> "Unable to parse this body. returning what we have so far.  Exception message: " + ex.getMessage());
            return new Body(Map.of(), new byte[0], partitions, BodyType.MULTIPART);
        }
        if (partitions.isEmpty()) {
            return new Body(Map.of(), new byte[0], List.of(), BodyType.UNRECOGNIZED);
        }
        return new Body(Map.of(), new byte[0], partitions, BodyType.MULTIPART);
    }

    private static String determineBoundaryValue(String contentType) {
        String boundaryKey = "boundary=";
        String boundaryValue = "";
        int indexOfBoundaryKey = contentType.indexOf(boundaryKey);
        if (indexOfBoundaryKey > 0) {
            boundaryValue = contentType.substring(indexOfBoundaryKey + boundaryKey.length());
        }
        return boundaryValue;
    }

    Body parseUrlEncodedForm(InputStream is, int contentLength) {
        if (contentLength == 0) {
            return Body.EMPTY;
        }
        HashMap<String, byte[]> postedPairs = new HashMap<String, byte[]>();
        try {
            int countOfPartitions = 0;
            for (UrlEncodedKeyValue keyValue : this.getUrlEncodedDataIterable(is, contentLength)) {
                String decodedValue;
                byte[] convertedValue;
                if (++countOfPartitions >= 1000) {
                    throw new WebServerException("Error: body had excessive number of partitions (" + countOfPartitions + ").  Maximum allowed: 1000");
                }
                String value = new String(keyValue.getUedg().readAllBytes(), StandardCharsets.US_ASCII);
                String key = keyValue.getKey();
                byte[] result = postedPairs.put(key, convertedValue = (decodedValue = StringUtils.decode(value)) == null ? "".getBytes(StandardCharsets.UTF_8) : decodedValue.getBytes(StandardCharsets.UTF_8));
                if (result == null) continue;
                throw new WebServerException("Error: key (" + key + ") was duplicated in the post body - previous version was " + new String(result, StandardCharsets.US_ASCII) + " and recent data was " + decodedValue);
            }
        }
        catch (Exception ex) {
            this.logger.logDebug(() -> "Unable to parse this body. returning what we have so far.  Exception message: " + ex.getMessage());
            return new Body(postedPairs, new byte[0], List.of(), BodyType.UNRECOGNIZED);
        }
        return new Body(postedPairs, new byte[0], List.of(), BodyType.FORM_URL_ENCODED);
    }

    @Override
    public Iterable<UrlEncodedKeyValue> getUrlEncodedDataIterable(final InputStream inputStream, final long contentLength) {
        return () -> new Iterator<UrlEncodedKeyValue>(this){
            final CountBytesRead countBytesRead = new CountBytesRead();

            @Override
            public boolean hasNext() {
                return (long)this.countBytesRead.getCount() < contentLength;
            }

            @Override
            public UrlEncodedKeyValue next() {
                if (!this.hasNext()) {
                    throw new NoSuchElementException();
                }
                String key = "";
                ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
                while (true) {
                    int result = 0;
                    try {
                        result = inputStream.read();
                        this.countBytesRead.increment();
                    }
                    catch (IOException e) {
                        throw new WebServerException(e);
                    }
                    if (result == -1) break;
                    byte myByte = (byte)result;
                    if (myByte == 61) {
                        key = byteArrayOutputStream.toString(StandardCharsets.US_ASCII);
                        break;
                    }
                    if (byteArrayOutputStream.size() >= 50) {
                        throw new WebServerException("Maximum size for name attribute is 50 ascii characters");
                    }
                    byteArrayOutputStream.write(myByte);
                }
                if (key.isBlank()) {
                    throw new WebServerException("Unable to parse this body. no key found during parsing");
                }
                if ((long)this.countBytesRead.getCount() == contentLength) {
                    return new UrlEncodedKeyValue(key, new UrlEncodedDataGetter(InputStream.nullInputStream(), this.countBytesRead, contentLength));
                }
                return new UrlEncodedKeyValue(key, new UrlEncodedDataGetter(inputStream, this.countBytesRead, contentLength));
            }
        };
    }

    @Override
    public Iterable<StreamingMultipartPartition> getMultiPartIterable(final InputStream inputStream, final String boundaryValue, final int contentLength) {
        return () -> new Iterator<StreamingMultipartPartition>(){
            final CountBytesRead countBytesRead = new CountBytesRead();
            boolean hasReadFirstPartition = false;

            @Override
            public boolean hasNext() {
                return contentLength - this.countBytesRead.getCount() > boundaryValue.length();
            }

            @Override
            public StreamingMultipartPartition next() {
                if (!this.hasNext()) {
                    throw new NoSuchElementException();
                }
                if (!this.hasReadFirstPartition) {
                    try {
                        String s = BodyProcessor.this.inputStreamUtils.readLine(inputStream);
                        this.countBytesRead.incrementBy(s.length() + 2);
                        this.hasReadFirstPartition = true;
                        if (!s.contains(boundaryValue)) {
                            throw new IOException("Error: First line must contain the expected boundary value. Expected to find: " + boundaryValue + " in: " + s);
                        }
                    }
                    catch (IOException e) {
                        throw new WebServerException(e);
                    }
                }
                List<String> allHeaders = Headers.getAllHeaders(inputStream, BodyProcessor.this.inputStreamUtils);
                int lengthOfHeaders = allHeaders.stream().map(String::length).reduce(0, Integer::sum);
                int extraCrLfs = 2 * allHeaders.size() + 2;
                this.countBytesRead.incrementBy(lengthOfHeaders + extraCrLfs);
                Headers headers = new Headers(allHeaders);
                List<String> cds = headers.valueByKey("Content-Disposition");
                if (cds == null) {
                    throw new WebServerException("Error: no Content-Disposition header on partition in Multipart/form data");
                }
                String contentDisposition = String.join((CharSequence)";", cds);
                Matcher nameMatcher = multiformNameRegex.matcher(contentDisposition);
                Matcher filenameMatcher = multiformFilenameRegex.matcher(contentDisposition);
                String name = "";
                if (!nameMatcher.find()) {
                    throw new WebServerException("Error: No name value set on multipart partition");
                }
                name = nameMatcher.group("namevalue");
                String filename = "";
                if (filenameMatcher.find()) {
                    filename = filenameMatcher.group("namevalue");
                }
                return new StreamingMultipartPartition(headers, inputStream, new ContentDisposition(name, filename), boundaryValue, this.countBytesRead, contentLength);
            }
        };
    }
}

