package com.rokt.roktsdk

import android.app.Activity
import android.app.Application
import com.rokt.roktsdk.embedded.Widget
import java.lang.ref.WeakReference

/**
 * ### The main interface to Rokt SDK
 *
 * Use the member methods to interact with the Rokt SDK
 */
object Rokt {

    /**
     * ### Initializes Rokt Widget with Activity
     *
     * Initialize the Rokt SDK prior to using it in any Activity.
     * The recommended way of calling the init method is inside Application class or inside Launcher Activity class.
     *
     * ***Kotlin***
     *
     *  ```
     *  // Replace xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx with your Rokt Tag ID
     *   // Replace y.y with the application version
     *  Rokt.init("xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx", "y.y", this@LauncherActivity)
     *  ```
     *
     * ***Java***
     *
     *  ```
     *  // Replace xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx with your Rokt Tag ID
     *   // Replace y.y with the application version
     *  Rokt.INSTANCE.init("xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx", "y.y", LauncherActivity.this)
     *  ```
     *
     * @param roktTagId the TagID relating to your application
     * @param appVersion version of your application not the SDK
     * @param activity the Activity in which you are calling this method
     * @param fontPostScriptNames optional set of font postscript names used in your layouts if you want to provide
     * your own TypeFaces in execute
     * @param fontFilePathMap optional map of font postscript names to file paths if you want to provide your own
     * fonts from the assets/ directory
     * @param callback optional callback for init complete
     */
    fun init(
        roktTagId: String,
        appVersion: String,
        activity: Activity,
        fontPostScriptNames: Set<String> = emptySet(),
        fontFilePathMap: Map<String, String> = emptyMap(),
        callback: RoktInitCallback? = null,
    )

    /**
     * ### Initializes Rokt Widget with Application
     *
     * Initialize the Rokt SDK prior to using it in any Activity.
     * The recommended way of calling the init method is inside Application class or inside Launcher Activity class.
     *
     * ***Kotlin***
     *
     *  ```
     *  // Replace xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx with your Rokt Tag ID
     *  // Replace y.y with the application version
     *  Rokt.init("xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx", "y.y", this@YourApplication)
     *  ```
     *
     * ***Java***
     *
     *  ```
     *  // Replace xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx with your Rokt Tag ID
     *  // Replace y.y with the application version
     *  Rokt.INSTANCE.init("xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx", "y.y", YourApplication.this)
     *  ```
     *
     * @param roktTagId the TagID relating to your application
     * @param appVersion version of your application not the SDK
     * @param application Application object
     * @param fontPostScriptNames optional set of font postscript names used in your layouts if you want to provide
     * your own TypeFaces in execute
     * @param fontFilePathMap optional map of font postscript names to file paths if you want to provide
     * your own fonts from the assets/ directory
     * @param callback optional callback for init complete
     */
    fun init(
        roktTagId: String,
        appVersion: String,
        application: Application,
        fontPostScriptNames: Set<String> = emptySet(),
        fontFilePathMap: Map<String, String> = emptyMap(),
        callback: RoktInitCallback? = null,
    )

    /**
     * ### Execute Rokt Widget
     *
     * This is the entry point to display a widget to the consumer.
     *
     * The Rokt widget view displays after a short delay, configurable via the Rokt platform.
     *
     * The SDK provides optional callback events for when the view loads and unloads.
     *
     * Your app dictates which consumer attributes are passed through to Rokt
     *
     * ***Kotlin***
     *
     *  ```
     *  val attributes = hashMapOf(
     *      Pair("email", "j.smith@example.com"),
     *      Pair("firstname", "Jenny"),
     *      Pair("lastname", "Smith"),
     *      Pair("mobile", "(323) 867-5309"),
     *      Pair("postcode", "90210"),
     *      Pair("country", "US"))
     *
     *      // Optional Embedded widget placeholder mapping the placeholder view with placement location configuration
     *      val widget = findViewById<Widget>(R.id.roktWidget)
     *      val placeHolders = hashMapOf(
     *          Pair("RoktEmbedded1", WeakReference(widget))
     *      )
     *
     *  Rokt.execute("RoktExperience",
     *      attributes,
     *      object : Rokt.RoktCallback {
     *          override fun onUnload(reason: Rokt.UnloadReasons) {
     *          }
     *          override fun onLoad() {
     *          }
     *          override fun onShouldHideLoadingIndicator() {
     *          }
     *          override fun onShouldShowLoadingIndicator() {
     *          }
     *      },
     *      placeHolders // optional Embedded widget placement mapping
     *  )
     * ```
     *
     * ***Java***
     *
     * ```
     *  Map<String,String> attributes = new HashMap<String, String>();
     *  attributes.put("email", "j.smith@example.com");
     *  attributes.put("firstname", "Jenny");
     *  attributes.put("lastname", "Smith");
     *  attributes.put("mobile", "(323) 867-5309");
     *  attributes.put("postcode", "90210");
     *  attributes.put("country", "US");
     *
     *  // Optional Embedded widget placeholder mapping the placeholder view with placement location configuration
     *  Widget widget = findViewById(R.id.roktWidget);
     *  Map<String, WeakReference<Widget>> placeHolders = new HashMap<>();
     *  placeHolders.put("RoktEmbedded1", new WeakReference<>(widget));
     *
     *  Rokt.INSTANCE.execute("RoktExperience",
     *      attributes,
     *      new Rokt.RoktCallback() {
     *          @Override
     *          public void onLoad() {
     *          }
     *          @Override
     *          public void onUnload(Rokt.UnloadReasons unloadReasons) {
     *          }
     *          @Override
     *          public void onShouldHideLoadingIndicator() {
     *          }
     *          @Override
     *          public void onShouldShowLoadingIndicator() {
     *          }
     *      },
     *      placeHolders // optional Embedded widget placement mapping
     *   )
     * ```
     *
     * @param viewName the viewname of the widget you are trying to fetch
     * @param attributes list of attributes the host application can pass when fetching the widget for example First/Last Name, etc
     * @param callback callback for load/unload
     * @param placeholders optional. map of embedded widget placements with placement location configuration
     * @param fontTypefaces optional. map of font postscriptnames to Typeface WeakReferences that will be used in
     * your layouts. Must have provided the postscript names in init
     * @param config optional. app level config object
     */
    @JvmOverloads
    fun execute(
        viewName: String,
        attributes: Map<String, String>? = null,
        callback: RoktCallback,
        placeholders: Map<String, WeakReference<Widget>>? = null,
        fontTypefaces: Map<String, WeakReference<Typeface>>? = null,
        config: RoktConfig? = null,
    )

    /**
     *  ### Execute Rokt Widget
     *
     * This is the entry point to display a widget to the consumer.
     *
     * The Rokt widget view displays after a short delay, configurable via the Rokt platform.
     *
     * The SDK provides optional callback events for when the view loads and unloads.
     *
     * Your app dictates which consumer attributes are passed through to Rokt
     *
     * ***Kotlin***
     *
     *  ```
     *  val attributes = hashMapOf(
     *      Pair("email", "j.smith@example.com"),
     *      Pair("firstname", "Jenny"),
     *      Pair("lastname", "Smith"),
     *      Pair("mobile", "(323) 867-5309"),
     *      Pair("postcode", "90210"),
     *      Pair("country", "US"))
     *
     *      // Optional Embedded widget placeholder mapping the placeholder view with placement location configuration
     *      val widget = findViewById<Widget>(R.id.roktWidget)
     *      val placeHolders = hashMapOf(
     *          Pair("Location1", WeakReference(widget))
     *      )
     *
     *  val roktEventListener = object : RoktEventListener {
     *      override fun onEvent(event: RoktEvent) {
     *     }
     *  }
     *   val fontTypefaces: Map<String, WeakReference<Typeface>> =
     *   mapOf("monospace" to WeakReference(Typeface.MONOSPACE))
     *
     *  Rokt.execute("RoktExperience",
     *      attributes,
     *      eventCollector: RoktEventCollector(roktEventListener, lifecycle) ,
     *      placeHolders, // optional Embedded widget placement mapping
     *      fontTypefaces
     *  )
     * ```
     *
     * ***Java***
     *
     * ```
     *  Map<String,String> attributes = new HashMap<>(){
     *             {
     *                 put("email", "j.smith@example.com");
     *                 put("firstname", "Jenny");
     *                 put("lastname", "Smith");
     *                 put("mobile", "(323) 867-5309");
     *                 put("postcode", "90210");
     *                 put("country", "US");
     *             }
     *         };
     *
     *         Widget widget = findViewById(R.id.roktWidget);
     *         Map<String, WeakReference<Widget>> placeHolders = new HashMap<>();
     *         placeHolders.put("RoktEmbedded1", new WeakReference<>(widget));
     *         RoktEventListener eventListener = new RoktEventListener() {
     *             @Override
     *             public void onEvent(RoktEvent event) {
     *
     *             }
     *         };
     *
     *         Map<String, WeakReference<Typeface>> fontTypefaces = new HashMap<>(){
     *             {
     *                 put("monosppace", new WeakReference(Typeface.MONOSPACE));
     *             }
     *         };
     *         Rokt.INSTANCE.execute(
     *             "RoktExperience",
     *             attributes,
     *             new RoktEventCollector(roktEventListener, lifecycle),
     *             placeHolders,
     *             fontTypefaces
     *         );
     * ```
     * @param viewName the viewname of the widget you are trying to fetch
     * @param attributes list of attributes the host application can pass when fetching the widget for example First/Last Name, etc
     * @param eventCollector callback for rokt events
     * @param placeholders optional. map of embedded widget placements with placement location configuration
     * @param fontTypefaces optional. map of font postscriptnames to Typeface WeakReferences that will be used in
     * your layouts. Must have provided the postscript names in init
     * @param config optional. app level config object
     */
    @JvmOverloads
    fun execute(
        viewName: String,
        attributes: Map<String, String>? = null,
        eventCollector: RoktEventCollector? = null,
        placeholders: Map<String, WeakReference<Widget>>? = null,
        fontTypefaces: Map<String, WeakReference<Typeface>>? = null,
        config: RoktConfig? = null,
    )

    /**
     *  ### Execute Rokt Widget
     *
     * This is the entry point to display a widget to the consumer.
     *
     * The Rokt widget view displays after a short delay, configurable via the Rokt platform.
     *
     * The SDK provides optional callback events for when the view loads and unloads.
     *
     * Your app dictates which consumer attributes are passed through to Rokt
     *
     * ***Kotlin***
     *
     *  ```
     *  val attributes = hashMapOf(
     *      Pair("email", "j.smith@example.com"),
     *      Pair("firstname", "Jenny"),
     *      Pair("lastname", "Smith"),
     *      Pair("mobile", "(323) 867-5309"),
     *      Pair("postcode", "90210"),
     *      Pair("country", "US"))
     *
     *      // Optional Embedded widget placeholder mapping the placeholder view with placement location configuration
     *      val widget = findViewById<Widget>(R.id.roktWidget)
     *      val placeHolders = hashMapOf(
     *          Pair("Location1", WeakReference(widget))
     *      )
     *   val fontTypefaces: Map<String, WeakReference<Typeface>> =
     *   mapOf("monospace" to WeakReference(Typeface.MONOSPACE))
     *
     * coroutineScope.launch {
     *      Rokt.execute("RoktExperience",
     *          attributes,
     *          placeHolders // optional Embedded widget placement mapping
     *          fontTypefaces
     *      ).collect { event ->
     *      // receive the events here
     *      Log.d("Rokt", "received events $event")
     *  }
     * }
     * ```
     *
     * @param viewName the viewname of the widget you are trying to fetch
     * @param attributes list of attributes the host application can pass when fetching the widget for example First/Last Name, etc
     * @param placeholders optional. map of embedded widget placements with placement location configuration
     * @param fontTypefaces optional. map of font postscriptnames to Typeface WeakReferences that will be used in
     * your layouts. Must have provided the postscript names in init
     * @param config optional. app level config object
     * @return the Flow of Rokt Events.
     */
    @JvmOverloads
    fun executeWithEvents(
        viewName: String,
        attributes: Map<String, String>? = null,
        placeholders: Map<String, WeakReference<Widget>>? = null,
        fontTypefaces: Map<String, WeakReference<Typeface>>? = null,
        config: RoktConfig? = null,
    ): Flow<RoktEvent>

    /**
     * Returns a Flow of Rokt events associated with a viewName.
     *
     * @param viewName The name of the widget to listen for events on.
     * @return A Flow emitting Rokt events.
     */
    fun events(viewName: String): Flow<RoktEvent>

    /**
     * Returns a Flow of global Rokt events.
     *
     * @return A Flow emitting Rokt events. such as `InitComplete`
     */
    fun globalEvents(): Flow<RoktEvent> = events(viewName = "")

    /**
     * ### Enable or disable debug logging from the library
     *
     * @param enable enables or disables debug logging
     */
    fun setLoggingEnabled(enable: Boolean)

    /**
     * ### Optional callback for init complete
     */
    interface RoktInitCallback {
        /**
         * onInitComplete callback will be triggered when the SDK has finished initialization
         */
        fun onInitComplete(success: Boolean)
    }

    /**
     *
     * ### Optional callback events for when the view loads and unloads.
     *
     */
    interface RoktCallback {
        /**
         *
         * onLoad Callback will be triggered immediately when the Rokt Widget displays.
         */
        fun onLoad()

        /**
         *
         * onUnLoad Callback will be triggered if the Rokt Widget failed to show or it closed.
         */
        fun onUnload(reason: UnloadReasons)

        /**
         *
         * onShouldShowLoadingIndicator callback will be triggered if Rokt Widget start processing
         */
        fun onShouldShowLoadingIndicator()

        /**
         *
         * onShouldHideLoadingIndicator callback will be triggered if Rokt Widget end processing
         */
        fun onShouldHideLoadingIndicator()
    }

    /**
     * ### Reasons for unloading the widget
     */
    enum class UnloadReasons {
        /**
         * Called when there are no offers to display so the widget does not get loaded in
         */
        NO_OFFERS,

        /**
         * Widget has been rendered and has been completed
         */
        FINISHED,

        /**
         * Operation to fetch widget took too long to resolve
         */
        TIMEOUT,

        /**
         * Some error has regarding the network
         */
        NETWORK_ERROR,

        /**
         * Widget is empty
         */
        NO_WIDGET,

        /**
         * Init request was not successful
         */
        INIT_FAILED,

        /**
         * Placeholder string mismatch
         */
        UNKNOWN_PLACEHOLDER,

        /**
         * Catch-all for all issues
         */
        UNKNOWN,
    }
}

sealed interface RoktEvent {
    /**
     * ShowLoadingIndicator event will be triggered before SDK calls Rokt backend
     */
    object ShowLoadingIndicator : RoktEvent

    /**
     * HideLoadingIndicator event will be triggered when SDK obtains a success or failure from
     * Rokt backend
     */
    object HideLoadingIndicator : RoktEvent

    /**
     * OfferEngagement event will be triggered if User engaged with the offer
     * @param id - placement identifier
     */
    data class OfferEngagement(val id: String) : RoktEvent

    /**
     * PositiveEngagement event will be triggered if User positively engaged with the offer
     * @param id - placement identifier
     */
    data class PositiveEngagement(val id: String) : RoktEvent

    /**
     * FirstPositiveEngagement event will be triggered when the user positively engaged with the offer first time
     * @param id - placement identifier
     * @param fulfillmentAttributes - an interface using which additional attributes can be sent to the SDK
     */
    data class FirstPositiveEngagement(val id: String, val fulfillmentAttributes: FulfillmentAttributes) : RoktEvent

    /**
     * PlacementInteractive event will be triggered when placement has been rendered and is interactable
     * @param id - placement identifier
     */
    data class PlacementInteractive(val id: String) : RoktEvent

    /**
     * PlacementReady event will be triggered when placement is ready to display but has not rendered content yet
     * @param id - placement identifier
     */
    data class PlacementReady(val id: String) : RoktEvent

    /**
     * PlacementClosed event will be triggered when placement closes by user
     * @param id - placement identifier
     */
    data class PlacementClosed(val id: String) : RoktEvent

    /**
     * PlacementCompleted event will be triggered when the offer progression moves to the end and no more
     * offer to display
     * @param id - placement identifier
     */
    data class PlacementCompleted(val id: String) : RoktEvent

    /**
     * PlacementFailure event will be triggered when placement could not be displayed due to some failure
     * @param id - optional placement identifier
     */
    data class PlacementFailure(val id: String? = null) : RoktEvent

    /**
     * InitComplete event will be triggered when SDK has finished initialization
     * @param success - true if init was successful
     */
    data class InitComplete(val success: Boolean) : RoktEvent
}
