package com.s24.gs1;

import java.util.Arrays;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static com.s24.gs1.StringUtil.*;
import static com.s24.gs1.StringUtil.precedeWithZeros;

import static java.lang.String.format;

/**
 * Removes non-digit characters from a GTIN and adds leading zeros to GTINs according to the
 * specified target length.
 *
 * <p>To have unified GTIN values, you can use this normalizer. It precedes a GTIN with zeros. Due to
 * the checksum calculation mechanism this won't affect the validity of a GTIN. Any non-digit
 * characters will be removed.</p>
 *
 */
public class GtinNormalizer {

    private static final Pattern NON_DIGITS = Pattern.compile("[^\\d]");

    private final int targetLength;

    /**
     * Creates a new GTIN normalizer.
     *
     * @param targetLength A valid GTIN length to normalize GTINs
     *
     * @throws IllegalArgumentException If the target length is not a allowed GTIN format length
     */
    public GtinNormalizer(int targetLength) {

        if (!GtinFormat.isValidLength(targetLength)) {
            String validLengths = Arrays.stream(GtinFormat.values())
                                              .map(GtinFormat::getLength)
                                              .map(String::valueOf)
                                              .collect(Collectors.joining(", "));
            throw new IllegalArgumentException(format("The target length must be one of %s", validLengths));
        }

        this.targetLength = targetLength;
    }

    /**
     * Adds leading zeros to a GTIN value.
     *
     * @param gtin The GTIN to normalize. Must not be {@code null} or empty.
     * @return The GTIN padded to the given target length and
     * @throws IllegalArgumentException If the input was {@code null} or empty
     */
    public String normalize(CharSequence gtin) {

        if (isBlank(gtin)) {
            throw new IllegalArgumentException("GTIN must not be null or blank");
        }

        String onlyDigits = NON_DIGITS.matcher(gtin).replaceAll("");
        int length = onlyDigits.length();

        if (length > targetLength) {
            return onlyDigits.substring(length - targetLength);
        }

        if (length == targetLength) {
            return onlyDigits;
        }

        return precedeWithZeros(onlyDigits, targetLength);

    }




}
