package com.s24.gs1;

import static com.s24.gs1.StringUtil.*;

/**
 * Checks, whether a GTIN is valid or not.
 *
 * <p>A GTIN string is valid, if it</p>
 *
 * <ul>
 *     <li>Is not empty or null</li>
 *     <li>It does not contain any non-digits</li>
 *     <li>Has a valid length</li>
 *     <li>Has a valid checksum</li>
 *     <li>Has a valid check digit at the end</li>
 * </ul>
 */
public class GtinValidator {

    private final CharSequence gtin;

    public GtinValidator(final CharSequence gtin) {
        this.gtin = gtin;
    }

    /**
     * Validates the GTIN input string.
     *
     * @return {@code true} if the GTIN is valid
     */
    public boolean validate() {

        if (isBlank(gtin)) {
            return false;
        }

        if (containsNonDigits()) {
            return false;
        }

        int length = gtin.length();

        if (!GtinFormat.isValidLength(length)) {
            return false;
        }

        int checksum = calculateChecksum();

        if (checksum == 0) {
            return false;
        }

        int checkDigit = (10 - (checksum % 10)) % 10;
        int lastDigit = getNumericValue(length - 1);

        return checkDigit == lastDigit;

    }

    private boolean containsNonDigits() {
        for (int charIndex = 0; charIndex < gtin.length(); charIndex++) {
            if (!Character.isDigit(gtin.charAt(charIndex))) {
                return true;
            }
        }
        return false;
    }

    private int getNumericValue(final int position) {
        final char ch = gtin.charAt(position);
        return Character.getNumericValue(ch);
    }

    private int calculateChecksum() {
        int checksum = 0;
        int lastNonCheckDigitIndex = gtin.length() - 2;
        for (int charIndex = 0; charIndex < gtin.length() - 1; charIndex++) {
            int charValue = getNumericValue(charIndex);
            int positionFromRight = lastNonCheckDigitIndex - charIndex;
            int weight = positionFromRight % 2 == 0 ? 3 : 1;
            checksum += charValue * weight;
        }
        return checksum;
    }

}
