/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.ServiceLoader;
import java.util.stream.Stream;

import com.sap.cds.reflect.CdsModel;

/**
 * A utility to iterate the public {@link UrlResourcePath}s of Servlets created by {@link ServletAdapterFactory}
 * according to the passed {@link CdsModel}.
 * <p>
 * Define an instance of {@link UrlResourcePathVisitor} to take further action on public endpoints.
 */
public class ServletUrlResourcePaths {

	private List<UrlResourcePath> servletPaths = new ArrayList<>();

	/**
	 * Create an instance based on t underlying {@link CdsModel}.
	 *
	 * @param cdsModel	the {@link CdsModel}.
	 */
	public ServletUrlResourcePaths(CdsModel cdsModel) {
		Objects.requireNonNull(cdsModel, "cdsModel is required to find public cds endpoints");

		// we need to touch the endpoints of all servlet-based adapters
		ServiceLoader<AdapterFactory> factoryIterator = ServiceLoader.load(AdapterFactory.class);
		factoryIterator.iterator().forEachRemaining(factory -> {
			if (factory instanceof ServletAdapterFactory) {
				ServletAdapterFactory servletFactory = (ServletAdapterFactory) factory;
				if(servletFactory.isEnabled()) {
					UrlResourcePath servletPath = servletFactory.getServletPath(cdsModel);
					servletPaths.add(servletPath);
				}
			}
		});
	}

	/**
	 * Returns the base paths found in the model.
	 *
	 * @return	a {@link Stream} of {@link UrlResourcePath} defining the base endpoints.
	 */
	public Stream<UrlResourcePath> getBasePaths() {
		return servletPaths.stream();
	}

	/**
	 * The interface of a visitor
	 */
	public interface UrlResourcePathVisitor {

		/**
		 * Called when a public {@link UrlResourcePath} is found.
		 * @param publicPath	The public {@link UrlResourcePath}
		 */
		void foundPublicPath(UrlResourcePath publicPath);

		/**
		 * Called when a {@link UrlResourcePath} is found with a public event.
		 * @param path	The {@link UrlResourcePath}
		 * @param publicEvents	The public events on the given path
		 */
		void foundPublicEvents(UrlResourcePath path, Stream<String> publicEvents);
	}

	/**
	 * Starts enumerating all cds servlet paths recursively (depth-first order).
	 * @param visitor	a visitor being notified during iteration
	 */
	public void visit(UrlResourcePathVisitor visitor) {
		Objects.requireNonNull(visitor, "visitor is required to handle public endpoints");
		servletPaths.forEach(p -> visitServletPath(p, visitor));
	}

	private void visitServletPath(UrlResourcePath servletPath, UrlResourcePathVisitor visitor) {

		// first decide each sub path recursively
		servletPath.subPaths().forEach(subPath -> {
			visitServletPath(subPath, visitor);
		});

		if (servletPath.isPublic()) {
			visitor.foundPublicPath(servletPath);

		} else {
			// some http methods might still be public even if the path itself is closed
			visitor.foundPublicEvents(servletPath, servletPath.publicEvents());
		}

	}
}
