/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2.metadata.mtx;

import java.io.IOException;
import java.io.InputStream;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.io.ByteStreams;
import com.sap.cds.adapter.odata.v2.CdsRequestGlobals;
import com.sap.cds.adapter.odata.v2.metadata.MetadataInfo;
import com.sap.cds.adapter.odata.v2.metadata.ODataV2EdmProvider;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.LocaleUtils;

/**
 * Default implementation of the EDMX parsing. It looks for the EDMX in the applications resource folder.
 */
public class DefaultEdmxProviderAccessor extends AbstractEdmxProviderAccessor {

	private static final Logger log = LoggerFactory.getLogger(DefaultEdmxProviderAccessor.class);

	private final Map<String, MetadataInfo> metadataInfoMap = new ConcurrentHashMap<>();
	private String basePath = "edmx";

	public DefaultEdmxProviderAccessor(CdsRuntime runtime) {
		String configBasePath = runtime.getEnvironment().getCdsProperties().getOdataV2().getEdmxPath();
		if(getClass().getClassLoader().getResource(configBasePath) != null) {
			this.basePath = configBasePath;
		}
	}

	@Override
	protected MetadataInfo getMetadataInfo(String serviceName) {
		Locale locale = CdsRequestGlobals.currentContext().getParameterInfo().getLocale();
		MetadataInfo info = getMetadataInfo(serviceName, locale);
		if(locale != null && (info.getEdmxMetadataProvider() == null || info.getEtag() == null)) {
			// retry without locale
			info = getMetadataInfo(serviceName, null);
		}
		return info;
	}

	private MetadataInfo getMetadataInfo(String unlocalizedServiceName, Locale locale) {
		String serviceName;
		if(locale != null) {
			String language = LocaleUtils.getLocaleForBundle(locale).toString();
			serviceName = unlocalizedServiceName + "_" + language;
		} else {
			serviceName = unlocalizedServiceName;
		}

		return metadataInfoMap.computeIfAbsent(serviceName, (key) -> {
			ODataV2EdmProvider provider = null;
			String etag = null;
			String resourcePathToEdmx = basePath + "/" + serviceName + XML; // getResourceAsStream accepts unnormalized paths containing e.g. "//"
			InputStream edmxStream = getClass().getClassLoader().getResourceAsStream(resourcePathToEdmx);
			if(edmxStream != null) {
				try {
					byte[] edmxBytes = ByteStreams.toByteArray(edmxStream);
					provider = loadMetadataFiles(unlocalizedServiceName, edmxBytes);
					etag = calculateMetadataEtag(edmxBytes);
				} catch (Exception e) { // NOSONAR
					throw new ErrorStatusException(CdsErrorStatuses.INVALID_METADATA_V2, resourcePathToEdmx, e);
				} finally {
					try {
						edmxStream.close();
					} catch (IOException e) {
						log.warn("Failed to close edmx input stream", e);
					}
				}
			}
			return new MetadataInfo(provider, etag);
		});
	}

}
