/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v2.metadata.mtx;

import java.nio.charset.StandardCharsets;
import java.util.Locale;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.adapter.odata.v2.metadata.MetadataInfo;
import com.sap.cds.adapter.odata.v2.metadata.ODataV2EdmProvider;
import com.sap.cds.mtx.MetaDataAccessor;
import com.sap.cds.mtx.ModelId;
import com.sap.cds.mtx.impl.CacheParams;
import com.sap.cds.mtx.impl.MetaDataAccessorImpl;
import com.sap.cds.mtx.impl.SidecarAccess;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.mt.ExtensibilityService;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.mtx.MtxUtils;

/**
 * An implementation of the EDMX parsing that uses the MTX Sidecar to retrieve the edmx files.
 */
public class MtxEdmxProviderAccessor extends AbstractEdmxProviderAccessor {

	private static Logger log = LoggerFactory.getLogger(MtxEdmxProviderAccessor.class);

	private final MetaDataAccessor<MetadataInfo> accessor;
	private final CdsRuntime runtime;

	public MtxEdmxProviderAccessor(CdsRuntime runtime) {
		this.runtime = runtime;

		MtxUtils mtxUtils = new MtxUtils(runtime);
		SidecarAccess access = mtxUtils.getSidecarAccess();
		CacheParams cacheParams = mtxUtils.getCacheParams();
		this.accessor = new MetaDataAccessorImpl<>(access, cacheParams, (edmx, service) -> {
			byte[] edmxBytes = edmx.getBytes(StandardCharsets.UTF_8);
			ODataV2EdmProvider edmProvider;
			try {
				edmProvider = loadMetadataFiles(service, edmxBytes);
			} catch (Exception e) { // NOSONAR
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_METADATA_V2_MTX, service, e);
			}
			String etag = calculateMetadataEtag(edmxBytes);
			return new MetadataInfo(edmProvider, etag);
		}, null);
		ExtensibilityService extService = runtime.getServiceCatalog().getService(ExtensibilityService.class, ExtensibilityService.DEFAULT_NAME);
		extService.on(ExtensibilityService.EVENT_MODEL_CHANGED, null, context -> {
			accessor.refresh(context.getUserInfo().getTenant());
		});
	}

	@Override
	protected MetadataInfo getMetadataInfo(String serviceName) {
		RequestContext requestContext = RequestContext.getCurrent(runtime);
		Locale locale = requestContext.getParameterInfo().getLocale();
		String localeString = locale != null ? locale.getLanguage() : Locale.ENGLISH.getLanguage();
		UserInfo userInfo = requestContext.getUserInfo();
		String tenant = userInfo.getTenant();
		Set<String> features = requestContext.getFeatureTogglesInfo().getEnabledFeatureToggles().map(ft -> ft.getName()).collect(Collectors.toSet());
		ModelId modelId = ModelId.create(tenant).features(features).service(serviceName).language(localeString).build();
		int maxAgeSeconds = Integer.MAX_VALUE;

		log.debug("Retrieving model for service name '{}', locale '{}' and features {}", serviceName, localeString, features);
		try {
			return accessor.getEdmx(modelId, maxAgeSeconds);
		} catch (ServiceException e) {
			throw e;
		} catch (Throwable t) {
			throw new ErrorStatusException(CdsErrorStatuses.EDMX_READ_FAILED, serviceName, t);
		}
	}

}
